/*
 * #%L
 * Pollen :: Services
 * $Id: PollenServiceSupport.java 3606 2012-08-13 14:04:23Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5.1/pollen-services/src/main/java/org/chorem/pollen/services/PollenServiceSupport.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.entities.PollenDAOHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaContextImplementor;
import org.nuiton.topia.persistence.TopiaDAO;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.DateUtil;

import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

import static org.nuiton.i18n.I18n.l_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenServiceSupport implements PollenService {

    protected PollenServiceContext serviceContext;

    @Override
    public void setServiceContext(PollenServiceContext serviceContext) {
        this.serviceContext = serviceContext;
    }

    public <E extends TopiaEntity> List<E> getEntities(PollenDAOHelper.PollenEntityEnum entityType) {
        Class<E> contract = (Class<E>) entityType.getContract();
        return getEntities(contract);
    }

    public <E extends TopiaEntity> List<E> getEntities(Class<E> entityType) {
        Preconditions.checkNotNull(entityType);
        try {
            TopiaDAO<E> dao = getDAO(entityType);
            List<E> result = dao.findAll();
            return result;
        } catch (TopiaException eee) {
            throw new PollenTechnicalException(
                    "Could not obtain data for type " + entityType, eee);
        }
    }

    public <E extends TopiaEntity> E getEntityById(Class<E> entityType, String id) {
        Preconditions.checkNotNull(entityType);
        Preconditions.checkArgument(StringUtils.isNotEmpty(id), "The " + entityType.getSimpleName() + " id is undefined");
        try {
            TopiaDAO<E> dao = getDAO(entityType);
            E result = dao.findByTopiaId(id);
            return result;
        } catch (TopiaException eee) {
            throw new PollenTechnicalException(
                    "Could not obtain data of id [" + id + "] for type " +
                    entityType, eee);
        }
    }

    public <T extends TopiaEntity, D extends TopiaDAO<? super T>> D getDAO(Class<T> entityType) {
        Preconditions.checkNotNull(entityType);
        try {
            D dao = PollenDAOHelper.getDAO(getTransaction(), entityType);
            return dao;
        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not obtain dao of type " + entityType.getName(), e);
        }
    }

    public String decorateDateTime(Date date) {
        String result = null;
        if (date != null) {
            result = DateUtil.formatDate(date, getDateTimePattern());
        }
        return result;
    }

    public String decorateDate(Date date) {
        String result = null;
        if (date != null) {
            result = DateUtil.formatDate(date, getDatePattern());
        }
        return result;
    }

    public String getDateTimePattern() {
        String result = l_(serviceContext.getLocale(), "pollen.common.dateTimePattern");
        return result;
    }

    public String getDatePattern() {
        String result = l_(serviceContext.getLocale(), "pollen.common.datePattern");
        return result;
    }

    public <E extends PollenService> E newService(Class<E> serviceClass) {
        E service = serviceContext.newService(serviceClass);
        return service;
    }

    protected <T extends TopiaEntity, D extends TopiaDAO<? super T>> T newInstance(D dao) {
        try {
            return (T) dao.newInstance();
        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not obtain new instance from dao " + dao, e);
        }
    }

    protected <T extends TopiaEntity, D extends TopiaDAO<? super T>> void delete(D dao, T entity) {
        try {
            dao.delete(entity);
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not delete entity " +
                                               entity.getTopiaId(), e);
        }
    }

    protected <T extends TopiaEntity, D extends TopiaDAO<? super T>> T create(D dao) {
        try {
            return (T) dao.create();
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not create entity ", e);
        }
    }
    protected <T extends TopiaEntity, D extends TopiaDAO<? super T>> T createWithProperties(D dao, Object... properties) {
        try {
            return (T) dao.create(properties);
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not create entity ", e);
        }
    }

    protected <T extends TopiaEntity, D extends TopiaDAO<? super T>> T update(D dao, T entity) {
        try {
            return (T) dao.update(entity);
        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not update entity ", e);
        }
    }

    protected TopiaContext getTransaction() {
        return serviceContext.getTransaction();
    }

    protected Locale getLocale() {
        return serviceContext.getLocale();
    }

    protected PollenConfiguration getConfiguration() {
        return serviceContext.getConfiguration();
    }

    protected void commitTransaction(String errorMessage) {
        try {
            getTransaction().commitTransaction();
        } catch (TopiaException eee) {
            throw new PollenTechnicalException(errorMessage, eee);
        }
    }

    protected void flushTransaction() throws TopiaException {
        TopiaContextImplementor tx = (TopiaContextImplementor) getTransaction();
        tx.getHibernate().flush();
    }

    /**
     * Generates a unique id usable for {@link Poll#getPollId()} or
     * {@link PollAccount#getAccountId()}.
     *
     * @return a fresh generated id
     * @since 1.4
     */
    protected String generateId() {
        return UUID.randomUUID().toString().replaceAll("-", "");
    }

}
