package org.chorem.pollen;
/*
 * #%L
 * Pollen :: Services
 * $Id: PollenUserSecurityContext.java 3627 2012-08-24 15:47:45Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5.1/pollen-services/src/main/java/org/chorem/pollen/PollenUserSecurityContext.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.collections.CollectionUtils;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.services.impl.SecurityService;

import java.io.Serializable;
import java.util.Collections;
import java.util.Set;

/**
 * To store security context for a given user at each request.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public class PollenUserSecurityContext implements Serializable {

    private static final long serialVersionUID = 5348851531764083245L;

    /**
     * Connected user account.
     * <p/>
     * Can be {@code null} if user is not connected.
     */
    protected final UserAccount userAccount;

    /**
     * Poll uri defining the poll - pollAccount to use.
     *
     * @see PollUri
     */
    protected final PollUri pollUri;

    /**
     * Current loaded poll according to the {@link #pollUri}.
     * <p/>
     * <strong>Note:</strong> this poll should be used by any action needed it.
     */
    protected Poll poll;

    /**
     * User roles on the poll.
     * <p/>
     * Loaded by the method {@link #loadUserRoles(SecurityService)}.
     */
    protected Set<PollenUserSecurityRole> userRoles;

    public static PollenUserSecurityContext newContext(UserAccount userAccount, PollUri pollUri) {
        return newContext(userAccount, pollUri, null);
    }

    public static PollenUserSecurityContext newContext(UserAccount userAccount,
                                                       String pollId,
                                                       String accountId,
                                                       Poll poll) {
        return newContext(userAccount,
                          PollUri.newPollUri(pollId, accountId),
                          poll);
    }

    public static PollenUserSecurityContext newContext(UserAccount userAccount,
                                                       PollUri pollUri,
                                                       Poll poll) {
        PollenUserSecurityContext result = new PollenUserSecurityContext(
                userAccount, pollUri
        );
        if (poll != null) {
            result.setPoll(poll);
        }
        return result;
    }

    protected PollenUserSecurityContext(UserAccount userAccount, PollUri pollUri) {
        this.userAccount = userAccount;
        this.pollUri = pollUri;
    }

    public boolean isPollExists() {
        return poll != null;
    }

    public boolean isAdmin() {
        return userAccount != null && userAccount.isAdministrator();
    }

    public boolean isWithAccountId() {
        return pollUri != null && pollUri.isAccountIdNotBlank();
    }

    public boolean isCreator() {
        return userRoles != null && userRoles.contains(PollenUserSecurityRole.CREATOR);
    }

    public boolean isVoter() {
        return userRoles != null && userRoles.contains(PollenUserSecurityRole.VOTER);
    }

    public boolean isRestrictedVoter() {
        return userRoles != null && userRoles.contains(PollenUserSecurityRole.RESTRICTED_VOTER);
    }

    public boolean hasNoRole() {
        return CollectionUtils.isEmpty(userRoles);
    }

    public Set<PollenUserSecurityRole> getUserRoles() {
        return CollectionUtils.isEmpty(userRoles) ?
               Collections.<PollenUserSecurityRole>emptySet() :
               ImmutableSet.copyOf(userRoles);
    }

    public UserAccount getUserAccount() {
        return userAccount;
    }

    public PollUri getPollUri() {
        return pollUri;
    }

    public Poll getPoll() {
        return poll;
    }

    public void setPoll(Poll poll) {
        this.poll = poll;
    }

    public void loadUserRoles(SecurityService securityService) {
        Preconditions.checkNotNull(securityService);
        Preconditions.checkNotNull(pollUri);
        Preconditions.checkNotNull(poll);
        userRoles = securityService.getUserRoles(poll,
                                                 getAccountId(),
                                                 userAccount);
    }

    public String getAccountId() {
        return pollUri == null ? null : pollUri.getAccountId();
    }

    public void removeAccountIdWhenConnected(PollUrl url) {
        if (userAccount != null) {

            // remove accountId from url
            url.getPollUri().setAccountId(null);
        }
    }

    public boolean isConnected() {
        return userAccount != null;
    }

    /**
     * To define a security role for a given pollen user which can be
     * identified by his user account or a accountId.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 1.4
     */
    public enum PollenUserSecurityRole {

        /**
         * Creator of the poll.
         * <p/>
         * This role can access to everything, but can not vote.
         */
        CREATOR,
        /** A user that has voted on a poll. */
        VOTER,
        /**
         * A user that was invited to a restricted poll.
         * <p/>
         * It might have alreay voted or not.
         */
        RESTRICTED_VOTER
    }
}
