/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: PollFeedService.java 3257 2012-04-13 11:28:02Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-services/src/main/java/org/chorem/pollen/services/impl/PollFeedService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.collect.Lists;
import com.sun.syndication.feed.synd.SyndContent;
import com.sun.syndication.feed.synd.SyndContentImpl;
import com.sun.syndication.feed.synd.SyndEntry;
import com.sun.syndication.feed.synd.SyndEntryImpl;
import com.sun.syndication.feed.synd.SyndFeed;
import com.sun.syndication.feed.synd.SyndFeedImpl;
import com.sun.syndication.io.SyndFeedInput;
import com.sun.syndication.io.SyndFeedOutput;
import com.sun.syndication.io.XmlReader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.services.PollenServiceSupport;

import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.util.Date;
import java.util.List;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.l_;

/**
 * Service to manage feed of a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollFeedService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(PollFeedService.class);

    public boolean isFeedExists(Poll poll) {
        File file = getFeedLocation(poll);
        return file.exists();
    }

    /**
     * Obtain the feed file of the given poll.
     *
     * @param poll poll
     * @return the location of the feed of the given poll
     */
    public File getFeedLocation(Poll poll) {

        File feedDirectory = getConfiguration().getFeedDirectory();
        String filename = poll.getPollId() + ".xml";
        File result = new File(feedDirectory, filename);
        return result;
    }

    public void onPollCreated(Poll poll, PollUrl pollVoteUrl) {
        File file = getFeedLocation(poll);

        String title = l_(getLocale(), "pollen.feed.pollTitle", poll.getTitle());
        String description = l_(getLocale(), "pollen.feed.pollDescription", poll.getDescription());
        try {

            SyndFeed feed = new SyndFeedImpl();
            feed.setFeedType("atom_1.0");
            feed.setTitle(title);
            feed.setLink(pollVoteUrl.getUrl());
            feed.setDescription(description);

            Writer writer = new FileWriter(file);
            SyndFeedOutput output = new SyndFeedOutput();
            output.output(feed, writer);
            writer.close();

            if (log.isDebugEnabled()) {
                log.debug("Feed created: " + file.getAbsolutePath());
            }

            title = _("pollen.feed.createPollContent",
                      poll.getCreator().getVotingId());

            String content = _("pollen.feed.createPollContent",
                               poll.getDescription());

            addFeedEntry(poll, title, content, pollVoteUrl);
        } catch (Exception e) {
            throw new PollenTechnicalException("Could not create feed", e);
        }
    }

    public void onAddComment(Comment comment, PollUrl pollVoteUrl) {

        String title = _("pollen.feed.addCommentTitle",
                         comment.getAuthor());

        String content = _("pollen.feed.addCommentContent",
                           comment.getText());

        addFeedEntry(comment.getPoll(), title, content, pollVoteUrl);
    }

    public void onAddVote(Vote vote,
                          boolean anonymousVote,
                          String pollResult,
                          PollUrl pollVoteUrl) {

        String userId;
        if (anonymousVote) {
            userId = _("pollen.common.anonymous");
        } else {
            userId = vote.getPollAccount().getVotingId();

        }
        String title = _("pollen.feed.addVoteTitle", userId);

        String content = _("pollen.feed.addVoteContent", pollResult);

        addFeedEntry(vote.getPoll(), title, content, pollVoteUrl);
    }

    public void onAddChoice(Choice choice, PollUrl pollVoteUrl) {

        String title = _("pollen.feed.addChoiceTitle", choice.getName());

        String content = _("pollen.feed.addChoiceContent",
                           choice.getDescription());

        addFeedEntry(choice.getPoll(), title, content, pollVoteUrl);
    }

    /**
     * Delete the poll feed file.
     *
     * @param poll poll
     */
    public void deleteFeed(Poll poll) {

        File feedLocation = getFeedLocation(poll);
        if (feedLocation.exists()) {
            boolean wasDeleted = feedLocation.delete();
            if (!wasDeleted) {
                throw new PollenTechnicalException("Could not delete feed");
            }
            if (log.isDebugEnabled()) {
                log.debug("Feed deleted: " + feedLocation.getAbsolutePath());
            }
        }
    }

    /**
     * Ajout d'une entrée dans un flux.
     *
     * @param poll        poll
     * @param title       titre de l'entrée
     * @param content     contenu de l'entrée
     * @param pollVoteUrl poll vote url
     */
    protected void addFeedEntry(Poll poll,
                                String title,
                                String content,
                                PollUrl pollVoteUrl) {

        //TODO tchemitshould remove this, may migrates when going to version 2.0
        //TODO pass on all polls from database and creates the feed if not found.
        if (!isFeedExists(poll)) {
            onPollCreated(poll, pollVoteUrl);
        }

        File file = getFeedLocation(poll);

        try {
            List entries = Lists.newArrayList();
            SyndFeedInput input = new SyndFeedInput();
            SyndFeed feed = input.build(new XmlReader(file));
            entries.addAll(feed.getEntries());

            SyndEntry entry;
            SyndContent description;
            entry = new SyndEntryImpl();
            entry.setTitle(title);
            entry.setLink(pollVoteUrl.getUrl());
            entry.setPublishedDate(new Date());
            description = new SyndContentImpl();
            description.setType("text/plain");
            description.setValue(content);
            entry.setDescription(description);
            entries.add(entry);

            feed.setEntries(entries);

            Writer writer = new FileWriter(file);
            try {
                SyndFeedOutput output = new SyndFeedOutput();
                output.output(feed, writer);
            } finally {
                writer.close();
            }

            if (log.isDebugEnabled()) {
                log.debug("Feed updated: " + file.getAbsolutePath());
            }
        } catch (Exception e) {
            throw new PollenTechnicalException("Could not feed feed", e);
        }
    }
}
