/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: FavoriteListImportCSV.java 3307 2012-04-24 10:45:44Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-services/src/main/java/org/chorem/pollen/services/impl/FavoriteListImportCSV.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.services.FavoriteListImport;
import org.chorem.pollen.services.PollenServicePredicates;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.FavoriteListImportException;
import org.nuiton.util.StringUtil;
import org.nuiton.util.csv.Import;
import org.nuiton.util.csv.ImportModel;
import org.nuiton.util.csv.ImportRuntimeException;
import org.nuiton.util.csv.ImportableColumn;
import org.nuiton.util.csv.ModelBuilder;
import org.nuiton.util.csv.ValueParser;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.text.ParseException;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * Implementation of {@link FavoriteListImport} as a service for CSV import.
 * Nuiton-CSV is used to manage the {@link Import} using a specific {@link
 * ImportModel} for pollen. Empty rows (no votingId or no email) are ignored.
 *
 * Created: 16/04/12
 *
 * @author fdesbois <desbois@codelutin.com>
 * @since 1.3
 */
public class FavoriteListImportCSV extends PollenServiceSupport implements FavoriteListImport {
    
    private static final Log log = LogFactory.getLog(FavoriteListImportCSV.class);

    private static final ValueParser<String> EMAIL_PARSER = new ValueParser<String>() {

        @Override
        public String parse(String value) throws ParseException {
            if (value != null && !StringUtil.isEmail(value)) {
                throw new ParseException(_("pollen.error.email.invalid"), 0);
            }
            return value;
        }
    };
    
    protected class FavoriteListImportModel implements ImportModel<PollAccount> {

        @Override
        public char getSeparator() {
            return ',';
        }

        @Override
        public void pushCsvHeaderNames(List<String> headerNames) {
        }

        @Override
        public PollAccount newEmptyInstance() {
            return newInstance(getDAO(PollAccount.class));
        }

        @Override
        public Iterable<ImportableColumn<PollAccount, Object>> getColumnsForImport() {
            ModelBuilder modelBuilder = new ModelBuilder();
            modelBuilder.newMandatoryColumn(PollAccount.PROPERTY_VOTING_ID, 
                                            PollAccount.PROPERTY_VOTING_ID);
            modelBuilder.newMandatoryColumn(PollAccount.PROPERTY_EMAIL, 
                                            PollAccount.PROPERTY_EMAIL, 
                                            EMAIL_PARSER);
            return modelBuilder.getColumnsForImport();
        }
    }

    @Override
    public List<PollAccount> execute(String url) throws FavoriteListImportException {

        List<PollAccount> results;
        Reader reader = null;
        try {
            reader = new FileReader(new File(url));

            Import<PollAccount> csvImport = Import.newImport(new FavoriteListImportModel(), reader);

            // Filter on empty account (without email or votingId)
            results = Lists.newArrayList(
                    Iterables.filter(csvImport,
                                     PollenServicePredicates.POLL_ACCOUNT_NOT_EMPTY)
            );

            if (log.isInfoEnabled()) {
                log.info(results.size() + " comptes importés.");
            }

        } catch (FileNotFoundException ex) {
            throw new PollenTechnicalException(ex);

        } catch (ImportRuntimeException ex) {
            if (log.isDebugEnabled()) {
                log.debug("Error during CSV import", ex);
            }
            throw new FavoriteListImportException("CSV", ex);

        } finally {
            IOUtils.closeQuietly(reader);
        }
        return results;
    }
    
//    @Override
//    public List<PollAccount> execute(String url) throws FavoriteListImportException {
//
//        List<PollAccount> results;
//        Reader reader = null;
//        try {
//            reader = new FileReader(new File(url));
//
//            // Définition de la stratégie de mapping
//            ColumnPositionMappingStrategy<PollAccountImpl> strat =
//                    new ColumnPositionMappingStrategy<PollAccountImpl>();
//
//            String[] columns = new String[] {
//                    PollAccount.PROPERTY_VOTING_ID,
//                    PollAccount.PROPERTY_EMAIL
//            };
//
//            strat.setType(PollAccountImpl.class);
//            strat.setColumnMapping(columns);
//
//            // Parsing du fichier CSV
//            CsvToBean<PollAccountImpl> csv = new CsvToBean<PollAccountImpl>();
//            List<PollAccount> importedData = Lists.<PollAccount>newArrayList(csv.parse(strat, reader));
//
//            // Suppression des comptes null
//            results = Lists.newArrayList(
//                    Iterables.filter(importedData,
//                                     PollenServicePredicates.POLL_ACCOUNT_NOT_EMPTY)
//            );
//
//            if (log.isInfoEnabled()) {
//                log.info(results.size() + " comptes importés.");
//            }
//
//        } catch (Exception ex) {
//            log.warn("Error during CSV import", ex);
//            throw new FavoriteListImportException(ex);
//
//        } finally {
//            IOUtils.closeQuietly(reader);
//        }
//        return results;
//    }
}
