/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataResultConverter;
import org.chorem.pollen.business.converters.EntityToVoteCountingDTO;
import org.chorem.pollen.business.converters.EnumController;
import org.chorem.pollen.business.dto.ResultDTO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.Result;
import org.chorem.pollen.business.utils.ContextUtil;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.votecounting.business.VoteCounting;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.services.ServiceVoteCounting;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des résultats.
 *
 * @author fdesbois
 * @version $Id: ServiceResultsImpl.java 2621 2009-07-03 15:09:23Z nrannou $
 */
public class ServiceResultsImpl implements ServiceResults {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private TopiaContext transaction = null;

    /** log. */
    private static final Log log = LogFactory.getLog(ServiceResultsImpl.class);

    public ServiceResultsImpl() {
    }

    @Override
    public List<ResultDTO> getAllResults(String pollId) {
        return getResults(pollId, null, false, false);
    }

    @Override
    public List<ResultDTO> getResultsByVoteCounting(String pollId,
            VoteCountingType voteCounting) {
        return getResults(pollId, voteCounting, false, false);
    }

    @Override
    public List<ResultDTO> getGroupResults(String pollId) {
        return getResults(pollId, null, true, true);
    }

    @Override
    public List<ResultDTO> getNormalResults(String pollId) {
        return getResults(pollId, null, true, false);
    }

    /**
     * Récupère les résultats d'un sondage avec le dépouillement normal sans
     * gestion des groupes.
     *
     * @param pollId le sondage
     * @param voteCounting type de dépouillement
     * @param byGroup résultats par groupe
     * @param groupOnly résultats uniquement par groupe
     * @return les résultats du sondage
     */
    private List<ResultDTO> getResults(String pollId,
            VoteCountingType voteCounting, boolean byGroup, boolean groupOnly) {
        try {
            transaction = rootContext.beginTransaction();

            PollDAO daoPoll = PollenModelDAOHelper.getPollDAO(transaction);
            Poll ePoll = daoPoll.findByPollId(pollId);

            // Exception Poll inexistant ??
            DataResultConverter converter = new DataResultConverter(transaction);
            if (log.isDebugEnabled()) {
                log.debug(ePoll.getPollId() + " (" + voteCounting
                        + ") has results: " + hasResults(ePoll, voteCounting));
            }

            // Dépouillement du sondage.
            // S'il n'existe pas de résultats ou si le sondage est encore ouvert.
            //if (!hasResults(ePoll, voteCounting) || !ePoll.getIsClosed()) {
            PollDTO dto = EntityToVoteCountingDTO
                    .createPollDTOForVoteCounting(ePoll);

            if (voteCounting != null) {
                dto.setTypeVoteCounting(voteCounting);
                // TODO gestion exception au moment du dépouillement si erreur
            }

            ServiceVoteCounting service = new VoteCounting();
            VoteCountingResultDTO result = null;

            if (EnumController.isGroupType(ePoll)) {
                result = service.executeGroupCounting(dto);
            } else {
                result = service.executeVoteCounting(dto);
            }

            ePoll.clearResult();
            converter.populateResultEntities(result);
            //}

            transaction.commitTransaction();

            // Conversion et trie des résultats
            List<ResultDTO> list = converter.createResultDTOs(ePoll);
            transaction.closeContext();

            if (voteCounting != null) {
                Iterator<ResultDTO> it = list.iterator();
                while (it.hasNext()) {
                    ResultDTO curr = it.next();
                    // Cas d'un dépouillement particulier
                    // Suppression des resultats qui ne sont pas de se dépouillement
                    if (voteCounting != null
                            && curr.getVoteCounting() != voteCounting) {
                        it.remove();
                    }
                    // Cas d'un sondage de type GROUP avec filtre
                    if (byGroup) {
                        // Filtre group : Suppression resultats non group
                        if (groupOnly && !curr.isByGroup()) {
                            it.remove();
                        }
                        // Filtre non group : Suppression resultats group
                        else if (!groupOnly && curr.isByGroup()) {
                            it.remove();
                        }
                        // Suppression des autres resultats de depouillements differents
                        if (curr.getVoteCounting() != EnumController
                                .getVoteCountingType(ePoll)) {
                            it.remove();
                        }
                    }
                }
            }

            return list;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public String exportPoll(String pollId) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public String importPoll(String filePath) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Retourne vrai si le sondage a des résultats.
     *
     * @param ePoll le sondage
     * @param type type de dépouillement
     * @return vrai si le sondage a des résultats
     */
    private boolean hasResults(Poll ePoll, VoteCountingType type) {
        Collection<Result> results = ePoll.getResult();
        if (results == null || results.isEmpty()) {
            return false;
        }
        if (!results.isEmpty() && type == null) {
            return true;
        }

        for (Result curr : results) {
            if (EnumController.getVoteCountingType(curr) == type) {
                return true;
            }
        }
        return false;
    }

    /**
     * Méthode exécutée lorsqu'une exception est détéctée.
     *
     * @param e l'exception
     */
    private void doCatch(TopiaException e) {

        // rollback de la transaction courante
        try {
            if (transaction != null) {
                transaction.rollbackTransaction();
                transaction.closeContext();
            }
        } catch (TopiaException ex) {
            ex.printStackTrace();
        }
        e.printStackTrace();
    }
}