/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataPreventRuleConverter;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.business.persistence.PreventRuleDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Gestion des règles de notification.
 *
 * @author rannou
 * @version $Id: ServicePreventRuleImpl.java 2621 2009-07-03 15:09:23Z nrannou $
 */
public class ServicePreventRuleImpl implements
        org.chorem.pollen.business.ServicePreventRule {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private TopiaContext transaction = null;
    private PreventRuleDAO preventRuleDAO = null;
    private DataPreventRuleConverter converter = new DataPreventRuleConverter();

    /** log. */
    private static final Log log = LogFactory
            .getLog(ServicePreventRuleImpl.class);

    public ServicePreventRuleImpl() {
    }

    @Override
    public String createPreventRule(PreventRuleDTO preventRule) {
        String topiaId = "";
        try {
            transaction = rootContext.beginTransaction();

            preventRuleDAO = PollenModelDAOHelper
                    .getPreventRuleDAO(transaction);

            PreventRule preventRuleEntity = preventRuleDAO.create();
            converter.setTransaction(transaction);
            converter.populatePreventRuleEntity(preventRule, preventRuleEntity);

            topiaId = preventRuleEntity.getTopiaId();
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity created: " + topiaId);
            }

            return topiaId;
        } catch (TopiaException e) {
            doCatch(e);
            return "";
        }
    }

    @Override
    public List<PreventRule> createPreventRules(
            List<PreventRuleDTO> preventRules, String pollId, TopiaContext trans) {
        if (preventRules.isEmpty() || pollId.length() == 0) {
            return null;
        }

        List<PreventRule> result = null;
        try {
            transaction = trans;

            preventRuleDAO = PollenModelDAOHelper.getPreventRuleDAO(trans);

            converter.setTransaction(trans);

            result = new ArrayList<PreventRule>();
            for (PreventRuleDTO preventRule : preventRules) {
                PreventRule preventRuleEntity = preventRuleDAO.create();
                preventRule.setId(preventRuleEntity.getTopiaId());
                preventRule.setPollId(pollId);
                converter.populatePreventRuleEntity(preventRule,
                        preventRuleEntity);
                result.add(preventRuleEntity);
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public boolean updatePreventRule(PreventRuleDTO preventRuleDTO) {
        boolean resu = false;
        try {
            transaction = rootContext.beginTransaction();

            preventRuleDAO = PollenModelDAOHelper
                    .getPreventRuleDAO(transaction);

            PreventRule preventRuleEntity = preventRuleDAO
                    .findByTopiaId(preventRuleDTO.getId());
            if (preventRuleEntity != null) {
                converter.setTransaction(transaction);
                converter.populatePreventRuleEntity(preventRuleDTO,
                        preventRuleEntity);

                preventRuleDAO.update(preventRuleEntity);
                transaction.commitTransaction();
                resu = true;
            }
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity updated: " + preventRuleDTO.getId());
            }

            return resu;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public boolean deletePreventRule(String preventRuleId) {
        try {
            transaction = rootContext.beginTransaction();

            preventRuleDAO = PollenModelDAOHelper
                    .getPreventRuleDAO(transaction);

            PreventRule preventRuleEntity = preventRuleDAO
                    .findByTopiaId(preventRuleId);

            preventRuleDAO.delete(preventRuleEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity deleted: " + preventRuleId);
            }

            return true;
        } catch (TopiaException e) {
            doCatch(e);
            return false;
        }
    }

    @Override
    public PreventRuleDTO findPreventRuleById(String preventRuleId) {
        PreventRuleDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            preventRuleDAO = PollenModelDAOHelper
                    .getPreventRuleDAO(transaction);

            PreventRule preventRuleEntity = preventRuleDAO
                    .findByTopiaId(preventRuleId);

            if (preventRuleEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createPreventRuleDTO(preventRuleEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    @Override
    public List<PreventRuleDTO> selectPreventRules(
            Map<String, Object> properties) {
        List<PreventRuleDTO> results = null;
        List<PreventRule> preventRuleEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            preventRuleDAO = PollenModelDAOHelper
                    .getPreventRuleDAO(transaction);

            if (properties == null) {
                preventRuleEntities = preventRuleDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : toutes les règles de notification ont été sélectionnées !");
                }
            } else {
                preventRuleEntities = preventRuleDAO
                        .findAllByProperties(properties);
            }
            converter.setTransaction(transaction);
            results = converter.createPreventRuleDTOs(preventRuleEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isInfoEnabled()) {
                log.info("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            doCatch(e);
            return null;
        }
    }

    /**
     * Méthode exécutée lorsqu'une exception est détéctée.
     *
     * @param e l'exception
     */
    private void doCatch(TopiaException e) {

        // rollback de la transaction courante
        try {
            if (transaction != null) {
                transaction.rollbackTransaction();
                transaction.closeContext();
            }
        } catch (TopiaException ex) {
            ex.printStackTrace();
        }
        e.printStackTrace();
    }
}