/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: ServiceFactory.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.2/vradi-services/src/main/java/com/jurismarches/vradi/services/ServiceFactory.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services;

import static org.nuiton.i18n.I18n._;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import com.jurismarches.vradi.entities.GroupFormsImpl;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.util.LocaleConverter;
import org.nuiton.util.converter.ConverterUtil;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyExtensionStorage;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceCached;
import org.nuiton.wikitty.WikittyServiceImpl;
import org.nuiton.wikitty.WikittyServiceNotifier;
import org.nuiton.wikitty.WikittyStorage;
import org.nuiton.wikitty.WikittyTreeNodeImpl;
import org.nuiton.wikitty.WikittyUserImpl;
import org.nuiton.wikitty.jdbc.WikittyExtensionStorageJDBC;
import org.nuiton.wikitty.jdbc.WikittyStorageJDBC;
import org.nuiton.wikitty.solr.WikittySearchEnginSolr;

import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.entities.ClientImpl;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.GroupImpl;
import com.jurismarches.vradi.entities.InfogeneImpl;
import com.jurismarches.vradi.entities.ModificationTagImpl;
import com.jurismarches.vradi.entities.QueryMakerImpl;
import com.jurismarches.vradi.entities.RootThesaurusImpl;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.SessionImpl;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.ThesaurusImpl;
import com.jurismarches.vradi.entities.UserImpl;
import com.jurismarches.vradi.entities.VradiUserImpl;
import com.jurismarches.vradi.entities.WebHarvestStreamImpl;
import com.jurismarches.vradi.entities.XmlFieldBindingImpl;
import com.jurismarches.vradi.entities.XmlStreamImpl;

/**
 * ServiceFactory.
 *
 * @author $Author: tchemit $
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 * @since 24 févr. 2010 22:18:17
 */
public class ServiceFactory {

    /** Log. */
    private static final Log log = LogFactory.getLog(ServiceFactory.class);

    /** Wikitty service (cached version) for remote access. */
    protected static WikittyService vradiWikittyService;

    /** Single instance of storage service. */
    protected static VradiStorageService vradiStorageService;

    /** Single instance of wikitty proxy. */
    protected static WikittyProxy wikittyProxy = null;

    
    static {
        // Dans un block static pour definition des variables 
        // d'environnement solr et jbossts
        // A voir pourquoi, les converters s'initialise mal durant les tests
        
        // FIXME XXX TODO WARNING this seams to cause a huge communication BUG with jgroups !

        try {
            ConverterUtil.initConverters();
            ConvertUtils.register(new LocaleConverter(), Locale.class);
            VradiServiceConfiguration config = VradiServiceConfiguration.getInstance();
            I18n.init(config.getLocale());
        } catch (Exception eee) {
            log.error("Cant initialize Service factory : ", eee);
        }
    }

    public static synchronized VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = new VradiStorageServiceImpl();
        }
        return vradiStorageService;
    }

    /**
     * Creates wikitty service (cached).
     * 
     * Getter for service, with default service configuration.
     *
     * @return a unique instance of WikittyService
     */
    public static synchronized WikittyService getWikittyService() {
        if (vradiWikittyService == null) {
            VradiServiceConfiguration configuration = VradiServiceConfiguration.getInstance();

            // Contains :
            //  - WikittyServiceNotifier.WIKITTY_EVENT_JGROUP_OPTION
            //  - WikittyServiceNotifier.WIKITTY_EVENT_PROPAGATE_OPTION
            //  - WikittyServiceNotifier.WIKITTY_EVENT_JGROUP_OPTION
            Properties flatProperties = configuration.getFlatOptions();

            getWikittyService(flatProperties);
        }

        return vradiWikittyService;
    }

    /**
     * Creates wikitty service (cached).
     *
     * Constructor added for UI embedded mode, with custom configuration.
     * 
     * @param properties custom properties
     * @return a unique instance of WikittyService
     */
    public static synchronized WikittyService getWikittyService(Properties properties) {
        if (vradiWikittyService == null) {
            
            WikittyExtensionStorage extensionStorage = new WikittyExtensionStorageJDBC(properties);
            WikittyStorage wikittyStorage = new WikittyStorageJDBC(extensionStorage, properties);
            WikittySearchEnginSolr searchEngin = new WikittySearchEnginSolr(extensionStorage);

            // declare service
            WikittyService service = new WikittyServiceImpl(extensionStorage, wikittyStorage, searchEngin);

            // add notification (on server, propagate option set to true)
            WikittyService serviceNotifier = new WikittyServiceNotifier(service, properties);

            // add cache
            // par defaut le cache ne se synchronise pas
            vradiWikittyService = new WikittyServiceCached(serviceNotifier, properties);
        }

        return vradiWikittyService;
    }

    /**
     * Creates wikitty proxy.
     *
     * @return a unique instance of WikittyProxy.
     */
    public static synchronized WikittyProxy getWikittyProxy() {
        if (wikittyProxy == null) {
            WikittyService serviceCache = getWikittyService();

            // init proxy on cached service
            wikittyProxy = new WikittyProxy(serviceCache);

            // post operation : register extensions
            updateExtensions(wikittyProxy);

            // post operation : reindex data if necessary
            reindexData(wikittyProxy);
        }

        return wikittyProxy;
    }

    /**
     * Register all extensions.
     * 
     * Already existing extensions with same version are skipped,
     * new extension are created and stored.
     * 
     * @param localWikittyProxy wikitty proxy
     */
    static void updateExtensions(WikittyProxy localWikittyProxy) {

        // easier if ordered
        // TODO EC20100616 don't work with required extensions :(
        List<WikittyExtension> extensions = Arrays.asList(
            ClientImpl.extensionClient,
            FormImpl.extensionForm,
            GroupImpl.extensionGroup,
            InfogeneImpl.extensionInfogene,
            ModificationTagImpl.extensionModificationTag,
            QueryMakerImpl.extensionQueryMaker,
            RootThesaurusImpl.extensionRootThesaurus,
            SendingImpl.extensionSending,
            SessionImpl.extensionSession,
            StatusImpl.extensionStatus,
            ThesaurusImpl.extensionThesaurus,
            WikittyTreeNodeImpl.extensionWikittyTreeNode,
            UserImpl.extensionUser,
            VradiUserImpl.extensionVradiUser,
            WebHarvestStreamImpl.extensionWebHarvestStream,
            WikittyUserImpl.extensionWikittyUser,
            XmlFieldBindingImpl.extensionXmlFieldBinding,
            XmlStreamImpl.extensionXmlStream,
            GroupFormsImpl.extensionGroupForms);

        localWikittyProxy.storeExtension(extensions);
    }

    /**
     * Check if version has changed and launch wikitty service reindex.
     * 
     * @param configuration configuration (containing version)
     * @param wikittyService wikitty service
     */
    protected static void reindexData(WikittyProxy localWikittyProxy) {
        try {
        
            VradiServiceConfiguration configuration = VradiServiceConfiguration.getInstance();

            String currentVersion = configuration.getApplicationVersion();
            String lastVersion = configuration.getServiceVersion();

            // different is enougth to reindex
            // currentVersion is never null, lastVersion could be
            if (!currentVersion.equals(lastVersion)) {

                // do version change migration
                ServiceMigration.versionChangeMigration(lastVersion, currentVersion, localWikittyProxy);

                if (log.isInfoEnabled()) {
                    log.info(_("Version change detected : %s. Reindexing data...", lastVersion));
                }
                long timeBefore = System.currentTimeMillis();

                // call syncEngin
                localWikittyProxy.syncEngin();

                long timeAfter = System.currentTimeMillis();

                if (log.isInfoEnabled()) {
                    log.info("Reindexing completed in " + (timeAfter - timeBefore) + " ms");
                }

                // save new version
                configuration.setServiceVersion(currentVersion);
                configuration.saveForUser();
            }
            else {
                if (log.isInfoEnabled()) {
                    log.info("No version change, skipping reindexing.");
                }
            }

            // post operation : register migration classes
            ServiceMigration.configureMigration();

        } catch (Exception eee) {
            log.error("Failled to reindex data : ", eee);
        }
    }
}
