/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.WikittyService;
import org.sharengo.wikitty.WikittyUtil;

import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.services.ServiceHelper;

/**
 * Class containing the metods to manage the form types :
 * - creation, update, retrieving
 * - cache filling
 *
 * @author schorlet
 * @date 2010-01-28 08:08:53
 * @version $Revision: 465 $ $Date: 2010-02-04 14:22:16 +0100 (jeu., 04 févr. 2010) $
 */
public class FormTypeManager {
    private static final Log log = LogFactory.getLog(FormTypeManager.class);

    private final WikittyProxy proxy;
    private Map<String, WikittyExtension> extensionCache = new HashMap<String, WikittyExtension>();

    public FormTypeManager(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    public FormTypeManager() {
        proxy = ServiceHelper.getWikittyProxy();
    }

    /**
     * Fills the cache map
     */
    private void fillExtensionCache() {
        if (extensionCache.isEmpty()) {
            WikittyService wikittyService = proxy.getWikittyService();
            List<WikittyExtension> allExtensions = wikittyService.getAllExtensions(true);
    
            for (WikittyExtension extension : allExtensions) {
                if (Form.EXT_FORM.equals(extension.getRequires())) {
                    extensionCache.put(extension.getName(), extension);
                }
            }
            
            if (log.isDebugEnabled()) {
                log.debug("FormType count : " + extensionCache.size());
            }
        }
    }

    /**
     * Retrieves all the form types
     *
     * @return a list containing all the form types
     */
    public List<WikittyExtension> getAllFormTypes() {
        if (log.isDebugEnabled()) {
            log.debug("getAllFormTypes()");
        }
        
        fillExtensionCache();
        List<WikittyExtension> list = new ArrayList<WikittyExtension>(extensionCache.values());
        return list;
    }

    /**
     * Retrieves all the form types associated with their name
     *
     * @return a map containing the form type names and the form types
     */
    public Map<String, WikittyExtension> getAllFormTypesMap() {
        if (log.isDebugEnabled()) {
            log.debug("getAllFormTypesMap()");
        }
        
        fillExtensionCache();
        Map<String, WikittyExtension> map = new HashMap<String, WikittyExtension>(extensionCache);
        return map;
    }

    /**
     * Retrieves the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve
     * @return the form type whose name is the paramater
     */
    public WikittyExtension getFormType(String name) {
        if (log.isDebugEnabled()) {
            log.debug("getFormType(" + name + ")");
        }
        
        Map<String, WikittyExtension> allFormTypes = getAllFormTypesMap();
        WikittyExtension wikittyExtension = allFormTypes.get(name);
        
        return wikittyExtension;
    }

    /**
     * Retrieves the fields of the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve the fields
     * @return a map containing the field names and their type
     */
    public Map<String, FieldType> getFormTypeFields(String name) {
        if (log.isDebugEnabled()) {
            log.debug("getFormTypeFields(" + name + ")");
        }
        
        WikittyExtension extension = getFormType(name);
        Map<String, FieldType> fields = new HashMap<String, FieldType>();
        
        for (String fieldName : extension.getFieldNames()) {
            fields.put(fieldName, extension.getFieldType(fieldName));
        }

        return fields;
    }

    /**
     * Updates the form type whose name is 'name'
     *
     * @param name the name of the form type to update
     * @param fields the new fields of the form type
     * @param requires the new requires of the form type
     * @param tagValues the new tag values of the form type
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(String name, Map<String, FieldType> fields,
           String requires, Map<String, String> tagValues) {
        
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + name + ", fields, requires, tagValues)");
        }
        
        if (name != null) {
            WikittyExtension lastVersion = getFormType(name);
            String newVersion = null;
            
            if (lastVersion != null) {
                newVersion = WikittyUtil.incrementMajorRevision(lastVersion.getVersion());
            } else {
                newVersion = WikittyUtil.DEFAULT_VERSION;
            }
            
            WikittyExtension extension = new WikittyExtension(name, newVersion,
                    requires, new LinkedHashMap<String, FieldType>(fields));
            
            if (tagValues != null) {
                for (Map.Entry<String, String> entry : tagValues.entrySet()) {
                    extension.addTagValue(entry.getKey(), entry.getValue());
                }
            }
            
            List<WikittyExtension> extensions = Arrays.asList(extension);
            proxy.getWikittyService().storeExtension(extensions);
            
            log.info(String.format("FormType named %s saved with id: %s and version: %s",
                    name, extension.getId(), newVersion));
            
            extensionCache.put(name, extension);
            return extension;
        }
        
        return null;
    }

    /**
     * Update the form type given in parameter
     *
     * @param extension the form type to update
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(WikittyExtension extension) {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + extension.toDefinition() + ")");
        }
        
        Map<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
        
        for (String fieldName : extension.getFieldNames()) {
            FieldType fieldType = extension.getFieldType(fieldName);
            fields.put(fieldName, fieldType);
        }

        return updateFormType(extension.getName(), fields,
                extension.getRequires(), extension.getTagValues());
    }

}
