/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.BusinessEntity;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.PagedResult;
import org.sharengo.wikitty.TreeNode;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyProxy;
import org.sharengo.wikitty.search.Search;

import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.XmlFieldBinding;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.dto.VradiFormPageDTO;
import com.jurismarches.vradi.services.dto.VradiSendingDTO;
import com.jurismarches.vradi.services.managers.BindingManager;
import com.jurismarches.vradi.services.managers.ClientManager;
import com.jurismarches.vradi.services.managers.FormManager;
import com.jurismarches.vradi.services.managers.FormTypeManager;
import com.jurismarches.vradi.services.managers.SearchManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;
import com.jurismarches.vradi.services.search.UnsupportedQueryException;

/**
 * @author morin
 */
public class VradiStorageServiceImpl implements VradiStorageService {

    static private final Log log = LogFactory.getLog(VradiStorageServiceImpl.class);

    /**
     * Proxy to store and retrieve data.
     */
    protected WikittyProxy proxy;
    
    protected ThesaurusManager thesaurusManager;
    protected BindingManager bindingManager;
    protected FormManager formManager;
    protected FormTypeManager formTypeManager;
    protected ClientManager clientManager;
    protected SearchManager searchManager;
    
    /**
     * Form Id date format
     */
    public static final SimpleDateFormat FORM_ID_DATE_FORMAT =
            FormManager.FORM_ID_DATE_FORMAT;
    
    public static final String DEFAULT_SENDING_PARAGRAPH =
            FormManager.DEFAULT_SENDING_PARAGRAPH;

    
    
    public VradiStorageServiceImpl() {
        proxy = ServiceHelper.getWikittyProxy();
        
        thesaurusManager = new ThesaurusManager(proxy);
        clientManager = new ClientManager(proxy);
        formTypeManager = new FormTypeManager(proxy);
        
        searchManager = new SearchManager(proxy, clientManager);
        formManager = new FormManager(proxy, clientManager, thesaurusManager, searchManager);

        bindingManager = new BindingManager(proxy, formTypeManager, formManager);
    }

    @Override
    public BusinessEntity getEntity(String id, Class<? extends BusinessEntity> clazz)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getEntity(" + id + ", " + clazz.getSimpleName() + ")");
        }
        
        BusinessEntity entity = proxy.restore(clazz, id);
        return entity;
    }

    @Override
    public BusinessEntity updateEntity(BusinessEntity entity)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateEntity(entity)");
        }
        
        if (entity != null) {
//            String incrementMajorRevision = WikittyUtil.incrementMajorRevision(
//                    entity.getWikittyVersion());
//            entity.setWikittyVersion(incrementMajorRevision);
            
            // updates the entity
            entity = proxy.store(entity);
            return entity;
        }
        
        return null;
    }

    @Override
    public void deleteEntity(BusinessEntity entity) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("deleteEntity(entity)");
        }
        
        if (entity != null) {
            proxy.delete(entity.getWikittyId());
        }
    }

    @Override
    public User getUser(String userId) throws TechnicalException {
        return clientManager.getUser(userId);
    }

    @Override
    public Client getClient(String clientId) throws TechnicalException {
        return clientManager.getClient(clientId);
    }

    @Override
    public Group getGroup(String groupId) throws TechnicalException {
        return clientManager.getGroup(groupId);
    }

    @Override
    public List<User> getGroupUsers(String groupId) throws TechnicalException {
        return clientManager.getGroupUsers(groupId);
    }

    @Override
    public List<User> getClientUsers(String clientId)
            throws TechnicalException {
        return clientManager.getClientUsers(clientId);
    }

    @Override
    public List<Client> getGroupClients(String groupId)
            throws TechnicalException {
        return clientManager.getGroupClients(groupId);
    }

    @Override
    public Client getClientByUserId(String userId) throws TechnicalException {
        return clientManager.getClientByUserId(userId);
    }

    @Override
    public List<Group> getGroupsByUserId(String userId)
            throws TechnicalException {
        return clientManager.getGroupsByUserId(userId);
    }

    @Override
    public List<Group> getGroupsByClientId(String clientId)
            throws TechnicalException {
        return clientManager.getGroupsByClientId(clientId);
    }

    @Override
    public List<Client> getAllClients() throws TechnicalException {
        return clientManager.getAllClients();
    }

    @Override
    public List<User> getAllUsers() throws TechnicalException {
        return clientManager.getAllUsers();
    }

    @Override
    public List<Group> getAllGroups() throws TechnicalException {
        return clientManager.getAllGroups();
    }

    @Override
    public Form updateForm(Form form) throws TechnicalException {
        return formManager.updateForm(form);
    }

    @Override
    public List<Form> updateForms(List<Form> forms) throws TechnicalException {
        return formManager.updateForms(forms);
    }

    @Override
    public void findForms(String query, VradiFormPageDTO formPageDTO) throws UnsupportedQueryException {
        findForms(query, null, null, null, null, null, formPageDTO);
    }

    @Override
    public void findForms(String query, WikittyExtension extension,
            String dateType, Date beginDate, Date endDate, List<String>[] thesaurus,
            VradiFormPageDTO formPageDTO) throws UnsupportedQueryException {

        searchManager.findForms(query, extension, dateType, beginDate, endDate,
                thesaurus, formPageDTO);
    }

    @Override
    public List<String> getQueriesReturningForm(Form form)
            throws TechnicalException {
        return searchManager.getQueriesReturningForm(form);
    }

    @Override
    public List<WikittyExtension> getAllFormTypes() {
        return formTypeManager.getAllFormTypes();
    }

    @Override
    public WikittyExtension getFormType(String name) {
        return formTypeManager.getFormType(name);
    }

    @Override
    public Map<String, FieldType> getFormTypeFields(String name) {
        return formTypeManager.getFormTypeFields(name);
    }

    @Override
    public WikittyExtension updateFormType(String name, Map<String, FieldType> fields,
           String requires, Map<String, String> tagValues) {
        
        return formTypeManager.updateFormType(name, fields, requires, tagValues);
    }

    @Override
    public WikittyExtension updateFormType(WikittyExtension extension) {
        return formTypeManager.updateFormType(extension);
    }

    @Override
    public Form getForm(String formId) {
        return formManager.getForm(formId);
    }

    @Override
    public void deleteForm(String formId) {
        formManager.deleteForm(formId);
    }
    
    @Override
    public List<Form> getForms(List<String> formIds) {
        return formManager.getForms(formIds);
    }

    @Override
    public List<Form> getAllForms() {
        return formManager.getAllForms();
    }

    @Override
    public List<XmlStream> getAllXmlStreams() {
        return bindingManager.getAllXmlStreams();
    }

    @Override
    public List<XmlFieldBinding> updateXmlFieldBindings(
            List<XmlFieldBinding> bindings) {
        return bindingManager.updateXmlFieldBindings(bindings);
    }

    @Override
    public XmlStream getXmlStream(String xmlStreamId)
            throws TechnicalException {
        return bindingManager.getXmlStream(xmlStreamId);
    }

    @Override
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        return bindingManager.getXmlFieldBinding(xmlFieldBindingId);
    }

    @Override
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        return bindingManager.getXmlFieldBindings(xmlStream);
    }

    @Override
    public void bindFormsToClients() {
        formManager.bindFormsToClients();
    }

    @Override
    public List<VradiSendingDTO> getFormsByClients(String dateType,
            Date beginDate, Date endDate, WikittyExtension extension,
            Boolean receptionProof, Boolean paragraph, int status)
            throws TechnicalException {

        return formManager.getFormsByClients(dateType, beginDate, endDate,
                extension, receptionProof, paragraph, status);
    }

    @Override
    public Map<Form, List<Client>> getClientsByForms(String dateType,
            Date beginDate, Date endDate, WikittyExtension extension) {
        return formManager.getClientsByForms(dateType, beginDate, endDate, extension);
    }

    @Override
    public Object[] getFormsFromXmlStream(XmlStream xmlStream, String lastItemRecorded,
            VradiUser vradiUser) throws TechnicalException {
        return bindingManager.getFormsFromXmlStream(xmlStream, lastItemRecorded, vradiUser);
    }

    @Override
    public VradiUser updateVradiUser(VradiUser vradiUser)
            throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateVradiUser(vradiUser)");
        }
        
        Search search = Search.query().eq(VradiUser.FQ_FIELD_NAME, vradiUser.getName());
        Criteria criteria = search.criteria();
        
        PagedResult<VradiUser> users = proxy.findAllByCriteria(VradiUser.class, criteria);
        List<VradiUser> result = users.getAll();
        
        if (result.size() > 0) {
            log.debug("user " + vradiUser + " already exists");
            return null;
            
        } else {
            return proxy.store(vradiUser);
        }
    }

    @Override
    public VradiUser logVradiUser(String vradiUserName,
                                  String vradiUserPassword) {
        if (log.isDebugEnabled()) {
            log.debug("logVradiUser(" + vradiUserName + ", " + vradiUserPassword + ")");
        }
        
        Criteria criteria = Search.query()
                .eq(VradiUser.FQ_FIELD_NAME, vradiUserName)
                .eq(VradiUser.FQ_FIELD_PASSWORD, vradiUserPassword)
                .criteria();
        
        VradiUser vradiUser = proxy.findByCriteria(VradiUser.class, criteria);
        return vradiUser;
    }

    @Override
    public void updateSendings(List<Sending> sendings) {
        if (log.isDebugEnabled()) {
            log.debug("updateSendings(sendings)");
        }
        
        proxy.store(sendings);
    }

    @Override
    public void importData(File file) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("importData(file)");
        }
        
        String toURI = file.toURI().toString();
        log.info("importing file: " + toURI);
        
        proxy.syncImportFromUri(toURI);
        log.info("import done");
    }

    @Override
    public String exportData() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("exportData()");
        }
        
        log.info("exporting all data");

        try {
            Search search = Search.query().keyword("*");
            Criteria criteria = search.criteria();

            String export = proxy.syncExportAllByCriteria(criteria);
            log.info("export done");
            return export;

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new TechnicalException(e);
        }
    }

    @Override
    public void reindexData() {
        if (log.isDebugEnabled()) {
            log.debug("reindexData()");
        }
        
        UpdateResponse response = proxy.getWikittyService().syncEngin();
        if (log.isDebugEnabled()) {
            log.debug(response.toString());
        }
    }

    @Override
    public void changeDataDir(String newDataDir, String oldDataDir) {
        if (log.isDebugEnabled()) {
            log.debug("changeDataDir(" + newDataDir + ", " + oldDataDir + ")");
        }
        
        proxy.changeDataDir(newDataDir, oldDataDir);
    }

    @Override
    public TreeNode getRootThesaurus() throws TechnicalException {
        return thesaurusManager.getRootThesaurus();
    }

    @Override
    public List<TreeNode> getAllThesaurus() throws TechnicalException {
        return thesaurusManager.getAllThesaurus();
    }

    @Override
    public TreeNode getThesaurus(String thesaurusId) throws TechnicalException {
        return thesaurusManager.getThesaurus(thesaurusId);
    }

    @Override
    public List<TreeNode> getChildrenThesaurus(String thesaurusId)
            throws TechnicalException {
        return thesaurusManager.getChildrenThesaurus(thesaurusId);
    }

    
    @Override
    public Map<TreeNode, Integer> getNbFormsByThesaurus()
            throws TechnicalException {
        return thesaurusManager.getNbFormsByThesaurus();
    }

    @Override
    public int getNbFormsForThesaurus(String thesaurusId)
            throws TechnicalException {
        return thesaurusManager.getNbFormsForThesaurus(thesaurusId);
    }

    @Override
    public Status getStatus(String statusId) throws TechnicalException {
        return formManager.getStatus(statusId);
    }

    @Override
    public List<Status> getStatuses(List<String> statusIds)
            throws TechnicalException {
        return formManager.getStatuses(statusIds);
    }

    @Override
    public List<Status> getAllStatuses() throws TechnicalException {
        return formManager.getAllStatuses();
    }

    @Override
    public Status updateStatus(Status status) throws TechnicalException {
        return formManager.updateStatus(status);
    }

    @Override
    public List<Status> updateStatuses(List<Status> statuses)
            throws TechnicalException {
        return formManager.updateStatuses(statuses);
    }

    @Override
    public void deleteStatus(String statusId) throws TechnicalException {
        List<String> statusIds = Arrays.asList(statusId);
        formManager.deleteStatuses(statusIds);
    }

    @Override
    public void deleteStatuses(List<String> statusIds) throws TechnicalException {
        formManager.deleteStatuses(statusIds);
    }

    @Override
    public Status getToTreatStatus() {
        return formManager.getToTreatStatus();
    }

    @Override
    public Status getValidatedStatus() {
        return formManager.getValidatedStatus();
    }

    public List<TreeNode> proposeThesaurus(Form form, List<TreeNode> thesaurus) throws TechnicalException {
        return thesaurusManager.proposeThesaurus(form, thesaurus);
    }
}
