package com.franciaflex.faxtomail.web.job;

/*
 * #%L
 * FaxToMail :: Web
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.FaxToMailApplicationContext;
import com.franciaflex.faxtomail.FaxToMailConfiguration;
import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.AttachmentImpl;
import com.franciaflex.faxtomail.persistence.entities.Client;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.EmailAccount;
import com.franciaflex.faxtomail.persistence.entities.EmailImpl;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailTopiaPersistenceContext;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.MailFilter;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.persistence.entities.OriginalEmail;
import com.franciaflex.faxtomail.services.DecoratorService;
import com.franciaflex.faxtomail.services.FaxToMailServiceContext;
import com.franciaflex.faxtomail.services.FaxToMailServiceUtils;
import com.franciaflex.faxtomail.services.service.ClientService;
import com.franciaflex.faxtomail.services.service.ConfigurationService;
import com.franciaflex.faxtomail.services.service.EmailService;
import com.franciaflex.faxtomail.services.service.MailFolderService;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.decorator.Decorator;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

import javax.activation.DataHandler;
import javax.mail.Address;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Header;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeUtility;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;

/**
 * @author Kevin Morin (Code Lutin)
 */
@DisallowConcurrentExecution
public class MailFilterJob extends AbstractFaxToMailJob {

    private static final Log log = LogFactory.getLog(MailFilterJob.class);

    protected FaxToMailApplicationContext applicationContext;

    protected FaxToMailConfiguration config;

    protected ConfigurationService configurationService;

    protected EmailService emailService;

    protected MailFolderService mailFolderService;

    protected ClientService clientService;

    protected DecoratorService decoratorService;

    @Override
    public void execute(JobExecutionContext jobExecutionContext) throws JobExecutionException {
        applicationContext = getApplicationContext(jobExecutionContext);
        FaxToMailTopiaPersistenceContext persistenceContext = null; 
        

        try {
            persistenceContext = applicationContext.newPersistenceContext();
            FaxToMailServiceContext serviceContext = applicationContext.newServiceContext(persistenceContext);
            config = serviceContext.getApplicationConfig();
        
            if (log.isInfoEnabled()) {
                log.info("Running MailFilterJob at " + serviceContext.getNow());
            }

            emailService = serviceContext.getEmailService();
            mailFolderService = serviceContext.getMailFolderService();
            configurationService = serviceContext.getConfigurationService();
            clientService = serviceContext.getClientService();
            decoratorService = serviceContext.getDecoratorService();

            Collection<EmailAccount> emailAccounts = configurationService.getEmailAccounts();
    
            for (EmailAccount account : emailAccounts) {
                checkEmails(account);
            }
            
            if (log.isDebugEnabled()) {
                log.debug("MailFilterJob ended at " + serviceContext.getNow());
            }
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't run quartz job", ex);
            }
        } finally {
            if (persistenceContext != null) {
                persistenceContext.close();
            }
        }
    }

    /**
     * Checks the emails of the account
     * @param account
     */
    public void checkEmails(EmailAccount account) {
        Properties properties = new Properties();
        // set the mail.mime.address.strict to false to avoid
        // javax.mail.internet.AddressException: Domain contains illegal character errors when recipients contains []
        properties.setProperty("mail.mime.address.strict", "false");
        
        switch (account.getProtocol()) {
        case IMAPS:
            properties.setProperty("mail.imap.ssl.enable", "true");
        case IMAP:
            properties.setProperty("mail.store.protocol", "imap");
            properties.setProperty("mail.imap.host", account.getHost());
            properties.setProperty("mail.imap.port", String.valueOf(account.getPort()));
            properties.setProperty("mail.imap.connectiontimeout", "2000");
            break;
        case POP3S:
            properties.setProperty("mail.pop3.ssl.enable", "true");
        case POP3:
            properties.setProperty("mail.store.protocol", "pop3");
            properties.setProperty("mail.pop3.host", account.getHost());
            properties.setProperty("mail.pop3.port", String.valueOf(account.getPort()));
            properties.setProperty("mail.pop3.connectiontimeout", "2000");
            break;
        }

        Session session = Session.getInstance(properties);
        Store store = null;
        Folder defaultFolder = null;
        Folder inbox = null;

        try {
            store = session.getStore();
            store.connect(account.getLogin(), account.getPassword());
            defaultFolder = store.getDefaultFolder();
            inbox = defaultFolder.getFolder("INBOX");
            checkEmailsOfFolder(account, inbox);

        } catch (Exception e) {
            log.error("Error while getting emails from the mailbox " + account.getLogin() + "@" + account.getHost(), e);

        } finally {
            close(inbox);
            close(defaultFolder);
            try {
                if (store != null && store.isConnected()) {
                    store.close();
                }
            } catch (MessagingException e) {
                log.error("Error while closing the store", e);
            }
        }

    }

    protected void close(Folder folder) {
        if (folder != null && folder.isOpen()) {
            try {
                boolean expunge = config.isMailExpunge();
                folder.close(expunge);
            } catch (Exception e) {
                log.error("Error while closing the folder", e);
            }
        }
    }

    /**
     * Check the emails of teh folder, create the emails in the database and delete the email in the folder.
     * 
     * @param emailAccount email account currently checked
     * @param folder the folder to check
     */
    protected void checkEmailsOfFolder(EmailAccount emailAccount, Folder folder) {
        
        int importedCount = 0;

        try {
            folder.open(Folder.READ_WRITE);

            int count = folder.getMessageCount();
            int unread = folder.getUnreadMessageCount();

            if (log.isDebugEnabled()) {
                log.debug(emailAccount.getLogin() + "@" +  emailAccount.getHost() + " : " + count + " messages, " + unread + " unread");
            }

            for (int i = 0 ; i < count ; i++) {
                try {
                    Email email = new EmailImpl();
                    email.setFax(emailAccount.isFaxAccountType());

                    int messageNumber = count - i;
                    Message message = folder.getMessage(messageNumber);

                    Charset charset = FaxToMailServiceUtils.getCharset(message);

                    Set<String> modifiedProperties = new HashSet<>();

                    if (log.isDebugEnabled()) {
                        log.debug(String.format("Message %d/%d : %s", i, count,  message.getSubject()));
                    }

                    List<Address> recipientAddresses = new ArrayList<>();
                    Address[] allRecipients = message.getAllRecipients();
                    if (allRecipients != null) {
                        recipientAddresses.addAll(Arrays.asList(allRecipients));
                    }
                    Set<String> recipients = new HashSet<String>(
                            Collections2.transform(recipientAddresses, new Function<Address, String>() {
                        @Override
                        public String apply(Address address) {
                            String recipient = address.toString();
                            // some recipient are like "toto tutu<toto.tutu73@gmail.com>"
                            // the regex is to extract email address from it
                            recipient = recipient.replaceAll("\\s", "").replaceFirst("^.*<(.*)>$", "$1");
                            recipient = recipient.toLowerCase();
                            return recipient;
                        }
                    }));

                    // try to find the real recipient, in case it is in the bcc
                    Enumeration allHeaders = message.getAllHeaders();
                    String regex = "^.*for<(.*)>.*$";

                    while (allHeaders.hasMoreElements()) {
                        Header header = (Header) allHeaders.nextElement();
                        if ("Received".equals(header.getName())) {
                            String forRecipient = StringUtils.removePattern(header.getValue(), "\\s");

                            if (forRecipient != null && forRecipient.matches(regex)) {
                                forRecipient = forRecipient.replaceFirst(regex, "$1");

                                if (StringUtils.isNotBlank(forRecipient) && recipients.add(forRecipient)) {
                                    if (log.isDebugEnabled()) {
                                        log.debug("recipient found in \"Received\" header: " + forRecipient);
                                    }
                                    break;
                                }
                            }
                        }
                    }

                    MailFilter filter = null;
                    for (String recipient : recipients) {
                        List<MailFilter> filters = mailFolderService.getFiltersForRecipient(recipient);

                        if (CollectionUtils.isNotEmpty(filters)) {
                            MailFilter mailFilter = filters.get(0);
                            // see #6161
                            if (filter == null || mailFilter.getPosition() < filter.getPosition()) {
                                filter = mailFilter;
                                email.setRecipient(recipient);
                                modifiedProperties.add(Email.PROPERTY_RECIPIENT);
                            }
                        }
                    }

                    if (filter == null) {
                        if (log.isDebugEnabled()) {
                            log.debug(" ==> No filter found for this message");
                            if (log.isTraceEnabled()) {
                                for (String recipient : recipients) {
                                    log.trace(" - for recipient " + recipient);
                                }
                            }
                        }

                        //  on garde le mail sur le serveur pour le traiter suite à l'ajout d'un futur filtre
                        continue;
                    }

                    // find company of the filter folder
                    MailFolder filterFolder = filter.getMailFolder();
                    while (!filterFolder.isUseCurrentLevelCompany() && filterFolder.getParent() != null) {
                        filterFolder = filterFolder.getParent();
                    }
                    String company = filterFolder.getCompany();
                    String brand = null;
                    // TODO kmorin 20150320 retirer cette rustine quand la gestion des marques par nom de domaine sera faite
                    if ("@groupecreal.com".equals(email.getRecipient().substring(email.getRecipient().lastIndexOf("@")))) {
                        brand = "creal";
                    }

                    Address[] addresses = message.getFrom();
                    if (addresses != null && addresses.length > 0) {
                        String sender = addresses[0].toString();
                        // some sender are like "toto tutu<toto.tutu73@gmail.com>"
                        // the regex is to extract email address from it
                        sender = sender.replaceFirst("^.*<(.*)>$", "$1");
                        sender = sender.toLowerCase();

                        // Identification du client en fonction du numéro de fax appelant ou de l'adresse e-mail émettrice
                        Client client = clientService.getClientForEmailAddress(sender, email, company, brand);
                        modifiedProperties.add(Email.PROPERTY_SENDER);

                        String object;
                        if (client != null) {
                            //TODO echatellier : a valider que l'on affiche bien le premier mail du client
                            String contact = sender;
                            if (email.isFax()) {
                                String faxNumber = contact.substring(0, contact.indexOf('@')).replaceAll(" ", "");
                                // NumberUtils.isNumber peut echouer (notation octal)
                                if (StringUtils.isNumeric(faxNumber)) {
                                    contact = StringUtils.leftPad(faxNumber, 10, '0');
                                }
                            }

                            List<String> objectItems = new ArrayList<>();
                            if (StringUtils.isNotEmpty(client.getCaracteristic1())) {
                                objectItems.add(client.getCaracteristic1());
                            }
                            if (StringUtils.isNotEmpty(client.getCaracteristic2())) {
                                objectItems.add(client.getCaracteristic2());
                            }
                            if (StringUtils.isNotEmpty(client.getCaracteristic3())) {
                                objectItems.add(client.getCaracteristic3());
                            }
                            if (StringUtils.isNotEmpty(client.getCode())) {
                                objectItems.add(client.getCode());
                            }
                            if (StringUtils.isNotEmpty(client.getName())) {
                                objectItems.add(client.getName());
                            }
                            objectItems.add(contact);
                            objectItems.add(DateFormat.getDateInstance(DateFormat.MEDIUM).format(new Date()));

                            object = StringUtils.join(objectItems, " / ");

                            modifiedProperties.add(Email.PROPERTY_CLIENT);

                        } else {
                            object = t("faxtomail.email.object.noClient");
                        }
                        email.setObject(object);
                        email.setClient(client);
                        modifiedProperties.add(Email.PROPERTY_OBJECT);

                        MailFolder mailFolder = null;
                        if (!filter.isFilterFolderPriority() && client != null) {
                            FaxToMailUser personInCharge = client.getPersonInCharge();
                            if (personInCharge != null) {
                                mailFolder = mailFolderService.getFolderForFaxToMailUser(personInCharge);
                            }
                        }

                        if (mailFolder == null) {
                            // to default folder
                            mailFolder = filter.getMailFolder();
                        }

                        email.setMailFolder(mailFolder);
                        modifiedProperties.add(Email.PROPERTY_MAIL_FOLDER);

                        // if client is null and folder
                        if (client == null && emailAccount.isRejectAllowed()) {
                            Boolean reject = null;
                            String rejectMessage = null;
                            String senderEmail = null;

                            // find reject conf
                            MailFolder rejectMailFolder = mailFolder;
                            while (!Boolean.FALSE.equals(reject) && rejectMailFolder != null
                                    && (reject == null
                                            || !rejectMailFolder.isUseCurrentLevelRejectResponseMessage()
                                            || !rejectMailFolder.isUseCurrentLevelRejectResponseMailAddress())) {

                                if (reject == null) {
                                    reject = rejectMailFolder.getRejectUnknownSender();
                                }
                                if (rejectMessage == null && rejectMailFolder.isUseCurrentLevelRejectResponseMessage()) {
                                    rejectMessage = rejectMailFolder.getRejectResponseMessage();
                                }
                                if (senderEmail == null && rejectMailFolder.isUseCurrentLevelRejectResponseMailAddress()) {
                                    senderEmail = rejectMailFolder.getRejectResponseMailAddress();
                                }
                                rejectMailFolder = rejectMailFolder.getParent();
                            }

                            if (Boolean.TRUE.equals(reject)) {

                                if (StringUtils.isAnyBlank(senderEmail, rejectMessage)) {
                                    if (log.isWarnEnabled()) {
                                        log.warn("Can't send reject message due to invalid configuration");
                                    }
                                } else {
                                    // unknown client -> message rejected
                                    String recipient = email.getSender();
                                    if (email.isFax()) {
                                        recipient = FaxToMailServiceUtils.addFaxDomainToFaxNumber(recipient, mailFolder);
                                    }
                                    emailService.rejectEmail(senderEmail, recipient, t("faxtomail.email.subject.re", message.getSubject()), rejectMessage);

                                    // important, delete mail
                                    deleteMail(message);
                                }

                                continue;
                            }
                        }

                    }

                    Date receivedDate = new Date();
                    email.setReceptionDate(receivedDate);
                    modifiedProperties.add(Email.PROPERTY_RECEPTION_DATE);

                    Date now = new Date();

                    Decorator<Date> dateDecorator = decoratorService.getDecoratorByType(Date.class, DecoratorService.DATE);
                    String projectRef = t("faxtomail.email.projectReference.default", dateDecorator.toString(now));
                    email.setProjectReference(projectRef);
                    modifiedProperties.add(Email.PROPERTY_PROJECT_REFERENCE);

                    email.setDemandStatus(DemandStatus.UNTREATED);
                    modifiedProperties.add(Email.PROPERTY_DEMAND_STATUS);

                    OriginalEmail originalEmail = emailService.originalEmailFromMessage((MimeMessage) message, charset);
                    email.setOriginalEmail(originalEmail);

                    List<Attachment> attachments = new ArrayList<>();
                    if (message.isMimeType("multipart/*")) {

                        // manage boundary id
                        List<String> htmlContent = emailService.decomposeMultipartEmail(attachments, message);
                        if (htmlContent != null)  {
                            if (log.isDebugEnabled()) {
                                log.debug("Converting html content to pdf : " + message.getSubject());
                            }
                            Attachment attachment = emailService.convertHTMLToPdf(attachments, htmlContent, t("faxtomail.email.content.attachment.htmlFileName"));
                            if (attachment != null) {
                                //remove text plain attachement if exists, to avoid having twice the mail content in the attachments
                                String plainTextFileName = t("faxtomail.email.content.attachment.plainFileName") + ".pdf";
                                for (Attachment a : attachments) {
                                    if (plainTextFileName.equals(a.getOriginalFileName())) {
                                        attachments.remove(a);
                                        break;
                                    }
                                }
                                attachments.add(attachment);
                            }
                        }

                    // text email
                    } else if (message.isMimeType("text/*")) {
                        // convertit le contenu texte en PDF
                        String content = IOUtils.toString(message.getInputStream(), charset);
                        if (StringUtils.isNotBlank(content)) {
                            Attachment attachment = emailService.convertTextToPdf(content, t("faxtomail.email.content.attachment.plainFileName"));
                            attachments.add(0, attachment);
                        }

                    //directly an attachment
                    } else {
                        String fileName = message.getFileName();

                        try {
                            fileName = MimeUtility.decodeText(fileName);

                        } catch (UnsupportedEncodingException ex) {
                            // don't care, use filename raw value
                        }

                        DataHandler dh = message.getDataHandler();

                        // create new attachment
                        Attachment attachment = new AttachmentImpl();
                        attachment.setAddedByUser(false);
                        AttachmentFile attachmentFile = emailService.getAttachmentFileFromStream(dh.getInputStream());
                        attachmentFile.setFilename(fileName);
                        attachment.setOriginalFile(attachmentFile);

                        // convert attachment if defined by admin
                        emailService.convertIfNecessary(attachment);

                        // save attachment
                        attachments.add(attachment);
                    }

                    emailService.saveEmail(email,
                            attachments,
                            null,
                            email.getClient() != null ? email.getClient().getCode() : null,
                            null,
                            modifiedProperties.toArray(new String[modifiedProperties.size()]));
                    importedCount++;

                    if (log.isDebugEnabled()) {
                        log.debug(" ==> Message placé dans le dossier " + email.getMailFolder().getName());
                    }

                    // important, delete mail
                    deleteMail(message);

                } catch (Exception e) {
                    log.error("Error while reading the email", e);
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("End of emails");
            }

        } catch (Exception e) {
            log.error("Error while reading the emails", e);
        }

        // usefull log info (do not remove)
        if (importedCount > 0 && log.isInfoEnabled()) {
            log.info(String.format("Imported %d mail for account %s@%s", importedCount, emailAccount.getLogin(), emailAccount.getHost()));
        }
    }

    /**
     * Supprime le mail sur le serveur (si nécéssaire).
     * 
     * @param message message to delete
     * @throws MessagingException 
     */
    protected void deleteMail(Message message) throws MessagingException {
        // suppression des mails sur le serveur distant (non actif par default)
        if (config.isMailDelete()) {
            message.setFlag(Flags.Flag.DELETED, true);
        }
    }

}
