/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import org.openscience.cdk.Atom;
import org.openscience.cdk.AtomRef;
import org.openscience.cdk.Bond;
import org.openscience.cdk.BondRef;
import org.openscience.cdk.ChemObject;
import org.openscience.cdk.SingleElectron;
import org.openscience.cdk.exception.NoSuchAtomException;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IChemObjectChangeEvent;
import org.openscience.cdk.interfaces.IChemObjectListener;
import org.openscience.cdk.interfaces.IElectronContainer;
import org.openscience.cdk.interfaces.ILonePair;
import org.openscience.cdk.interfaces.IPDBAtom;
import org.openscience.cdk.interfaces.IPseudoAtom;
import org.openscience.cdk.interfaces.ISingleElectron;
import org.openscience.cdk.interfaces.IStereoElement;
import org.openscience.cdk.isomorphism.matchers.IQueryAtom;
import org.openscience.cdk.isomorphism.matchers.IQueryBond;
import org.openscience.cdk.stereo.DoubleBondStereochemistry;
import org.openscience.cdk.stereo.ExtendedTetrahedral;
import org.openscience.cdk.stereo.TetrahedralChirality;
import org.openscience.cdk.tools.manipulator.SgroupManipulator;

final class AtomContainer2
extends ChemObject
implements IAtomContainer {
    private static final int DEFAULT_CAPACITY = 20;
    private BaseAtomRef[] atoms;
    private BaseBondRef[] bonds;
    private ILonePair[] lonepairs;
    private ISingleElectron[] electrons;
    private List<IStereoElement> stereo = new ArrayList<IStereoElement>();
    private int numAtoms;
    private int numBonds;
    private int numLonePairs;
    private int numSingleElectrons;

    AtomContainer2(int numAtoms, int numBonds, int numLonePairs, int numSingleElectrons) {
        this.atoms = new BaseAtomRef[numAtoms];
        this.bonds = new BaseBondRef[numBonds];
        this.lonepairs = new ILonePair[numLonePairs];
        this.electrons = new ISingleElectron[numSingleElectrons];
    }

    AtomContainer2() {
        this(0, 0, 0, 0);
    }

    AtomContainer2(IAtomContainer src) {
        this(src.getAtomCount(), src.getBondCount(), src.getLonePairCount(), src.getSingleElectronCount());
        for (IAtom atom : src.atoms()) {
            this.addAtom(atom);
        }
        for (IBond bond : src.bonds()) {
            this.addBond(bond);
        }
        for (ISingleElectron se : src.singleElectrons()) {
            this.addSingleElectron(se);
        }
        for (ILonePair lp : src.lonePairs()) {
            this.addLonePair(lp);
        }
        for (ISingleElectron se : src.stereoElements()) {
            this.addStereoElement((IStereoElement)se);
        }
    }

    private <T> T[] grow(T[] arr, int required) {
        int grow;
        int n = grow = arr.length == 0 ? 20 : arr.length + (arr.length >> 1);
        if (grow < required) {
            grow = required;
        }
        return Arrays.copyOf(arr, grow);
    }

    private void ensureAtomCapacity(int required) {
        if (required >= this.atoms.length) {
            this.atoms = this.grow(this.atoms, required);
        }
    }

    private void ensureBondCapacity(int required) {
        if (required >= this.bonds.length) {
            this.bonds = this.grow(this.bonds, required);
        }
    }

    private void ensureLonePairCapacity(int required) {
        if (required >= this.lonepairs.length) {
            this.lonepairs = this.grow(this.lonepairs, required);
        }
    }

    private void ensureElectronCapacity(int required) {
        if (required >= this.electrons.length) {
            this.electrons = this.grow(this.electrons, required);
        }
    }

    private static IAtom unbox(IAtom atom) {
        while (atom instanceof AtomRef) {
            atom = ((AtomRef)atom).deref();
        }
        return atom;
    }

    private static IBond unbox(IBond bond) {
        while (bond instanceof BondRef) {
            bond = ((BondRef)bond).deref();
        }
        return bond;
    }

    private BaseAtomRef getAtomRefUnsafe(IAtom atom) {
        if (atom.getContainer() == this && this.atoms[atom.getIndex()] == atom) {
            return (BaseAtomRef)atom;
        }
        atom = AtomContainer2.unbox(atom);
        for (int i = 0; i < this.numAtoms; ++i) {
            if (!Objects.equals((Object)this.atoms[i], atom)) continue;
            return this.atoms[i];
        }
        return null;
    }

    private BaseAtomRef getAtomRef(IAtom atom) {
        BaseAtomRef atomref = this.getAtomRefUnsafe(atom);
        if (atomref == null) {
            throw new NoSuchAtomException("Atom is not a member of this AtomContainer");
        }
        return atomref;
    }

    private BaseAtomRef newAtomRef(IAtom atom) {
        if (atom.getClass() == Atom.class) {
            return new BaseAtomRef(this, atom);
        }
        if ((atom = AtomContainer2.unbox(atom)).getClass() == Atom.class) {
            return new BaseAtomRef(this, atom);
        }
        if (atom instanceof IPseudoAtom) {
            return new PsuedoAtomRef((IAtomContainer)this, (IPseudoAtom)atom);
        }
        if (atom instanceof IQueryAtom) {
            return new QueryAtomRef((IAtomContainer)this, (IQueryAtom)atom);
        }
        if (atom instanceof IPDBAtom) {
            return new PdbAtomRef((IAtomContainer)this, (IPDBAtom)atom);
        }
        return new BaseAtomRef(this, atom);
    }

    private BondRef getBondRefUnsafe(IBond bond) {
        if (bond.getContainer() == this && this.bonds[bond.getIndex()] == bond) {
            return (BondRef)bond;
        }
        bond = AtomContainer2.unbox(bond);
        for (int i = 0; i < this.numBonds; ++i) {
            if (this.bonds[i].deref() != bond) continue;
            return this.bonds[i];
        }
        return null;
    }

    private BaseBondRef newBondRef(IBond bond) {
        BaseAtomRef end;
        BaseAtomRef beg = bond.getBegin() == null ? null : this.getAtomRef(bond.getBegin());
        BaseAtomRef baseAtomRef = end = bond.getEnd() == null ? null : this.getAtomRef(bond.getEnd());
        if (bond.getClass() == Bond.class) {
            return new BaseBondRef(this, bond, beg, end);
        }
        if ((bond = AtomContainer2.unbox(bond)) instanceof IQueryBond) {
            return new QueryBondRef(this, (IQueryBond)bond, beg, end);
        }
        return new BaseBondRef(this, bond, beg, end);
    }

    private void addToEndpoints(BaseBondRef bondref) {
        if (bondref.getAtomCount() == 2) {
            BaseAtomRef beg = this.getAtomRef((IAtom)bondref.getBegin());
            BaseAtomRef end = this.getAtomRef((IAtom)bondref.getEnd());
            beg.bonds.add(bondref);
            end.bonds.add(bondref);
        } else {
            for (int i = 0; i < bondref.getAtomCount(); ++i) {
                this.getAtomRef(bondref.getAtom(i)).bonds.add(bondref);
            }
        }
    }

    private void delFromEndpoints(BondRef bondref) {
        for (int i = 0; i < bondref.getAtomCount(); ++i) {
            BaseAtomRef aref = this.getAtomRefUnsafe(bondref.getAtom(i));
            if (aref == null) continue;
            aref.bonds.remove(bondref);
        }
    }

    public void addStereoElement(IStereoElement element) {
        this.stereo.add(element);
        this.notifyChanged();
    }

    public void setStereoElements(List<IStereoElement> elements) {
        this.stereo.clear();
        this.stereo.addAll(elements);
        this.notifyChanged();
    }

    public Iterable<IStereoElement> stereoElements() {
        return Collections.unmodifiableList(this.stereo);
    }

    private void clearAdjacency() {
        for (int i = 0; i < this.numAtoms; ++i) {
            this.atoms[i].bonds.clear();
        }
    }

    public void setAtoms(IAtom[] newatoms) {
        int i;
        this.ensureAtomCapacity(newatoms.length);
        boolean reindexBonds = false;
        for (i = 0; i < newatoms.length; ++i) {
            if (newatoms[i].getContainer() == this) {
                this.atoms[i] = (BaseAtomRef)newatoms[i];
                this.atoms[i].setIndex(i);
                continue;
            }
            if (this.atoms[i] != null) {
                this.atoms[i].removeListener((IChemObjectListener)this);
            }
            this.atoms[i] = this.newAtomRef(newatoms[i]);
            this.atoms[i].setIndex(i);
            this.atoms[i].addListener((IChemObjectListener)this);
            reindexBonds = true;
        }
        if (newatoms.length < this.numAtoms) {
            for (i = newatoms.length; i < this.numAtoms; ++i) {
                this.atoms[i].removeListener((IChemObjectListener)this);
            }
            Arrays.fill((Object[])this.atoms, newatoms.length, this.numAtoms, null);
        }
        this.numAtoms = newatoms.length;
        if (this.numBonds > 0 && reindexBonds) {
            this.clearAdjacency();
            for (i = 0; i < this.numBonds; ++i) {
                this.addToEndpoints(this.bonds[i]);
            }
        }
        this.notifyChanged();
    }

    public void setBonds(IBond[] newbonds) {
        int i;
        if (this.numBonds > 0) {
            this.clearAdjacency();
        }
        this.ensureBondCapacity(newbonds.length);
        for (i = 0; i < newbonds.length; ++i) {
            BaseBondRef bondRef = this.newBondRef(newbonds[i]);
            bondRef.setIndex(i);
            this.addToEndpoints(bondRef);
            if (this.bonds[i] != null) {
                this.bonds[i].removeListener((IChemObjectListener)this);
            }
            this.bonds[i] = bondRef;
            bondRef.addListener((IChemObjectListener)this);
        }
        if (newbonds.length < this.numBonds) {
            for (i = newbonds.length; i < this.numBonds; ++i) {
                this.bonds[i].removeListener((IChemObjectListener)this);
            }
            Arrays.fill((Object[])this.bonds, newbonds.length, this.numBonds, null);
        }
        this.numBonds = newbonds.length;
        this.notifyChanged();
    }

    public void setAtom(int idx, IAtom atom) {
        if (atom == null) {
            throw new NullPointerException("Null atom provided");
        }
        if (this.contains(atom)) {
            throw new IllegalArgumentException("Atom already in container at index: " + this.indexOf(atom));
        }
        if (idx < 0 || idx >= this.numAtoms) {
            throw new IndexOutOfBoundsException("No current atom at index: " + idx);
        }
        BaseAtomRef rep = this.newAtomRef(atom);
        BaseAtomRef org = this.atoms[idx];
        this.atoms[idx] = rep;
        this.atoms[idx].setIndex(idx);
        for (IBond bond : new ArrayList(org.bonds)) {
            if (bond.getBegin().equals((Object)org)) {
                bond.setAtom((IAtom)rep, 0);
                continue;
            }
            if (!bond.getEnd().equals((Object)org)) continue;
            bond.setAtom((IAtom)rep, 1);
        }
        for (ISingleElectron ec : this.singleElectrons()) {
            if (!org.equals(ec.getAtom())) continue;
            ec.setAtom((IAtom)rep);
        }
        for (ILonePair lp : this.lonePairs()) {
            if (!org.equals(lp.getAtom())) continue;
            lp.setAtom((IAtom)rep);
        }
        for (int i = 0; i < this.stereo.size(); ++i) {
            IStereoElement se = this.stereo.get(i);
            if (!se.contains((IAtom)org)) continue;
            Map<BaseAtomRef, BaseAtomRef> amap = Collections.singletonMap(org, rep);
            Map bmap = Collections.emptyMap();
            this.stereo.set(i, se.map(amap, bmap));
        }
        org.removeListener((IChemObjectListener)this);
        rep.addListener((IChemObjectListener)this);
        this.notifyChanged();
    }

    public IAtom getAtom(int idx) {
        if (idx >= this.numAtoms) {
            throw new IndexOutOfBoundsException("No atom at index: " + idx);
        }
        return this.atoms[idx];
    }

    public IBond getBond(int idx) {
        if (idx >= this.numBonds) {
            throw new IndexOutOfBoundsException("No bond at index: " + idx);
        }
        return this.bonds[idx];
    }

    public ILonePair getLonePair(int idx) {
        if (idx >= this.numLonePairs) {
            throw new NoSuchElementException("No lone pair at index: " + idx);
        }
        return this.lonepairs[idx];
    }

    public ISingleElectron getSingleElectron(int idx) {
        if (idx >= this.numSingleElectrons) {
            throw new NoSuchElementException("No electron at index: " + idx);
        }
        return this.electrons[idx];
    }

    public Iterable<IAtom> atoms() {
        return new Iterable<IAtom>(){

            @Override
            public Iterator<IAtom> iterator() {
                return new AtomIterator();
            }
        };
    }

    public Iterable<IBond> bonds() {
        return new Iterable<IBond>(){

            @Override
            public Iterator<IBond> iterator() {
                return new BondIterator();
            }
        };
    }

    public Iterable<ILonePair> lonePairs() {
        return new Iterable<ILonePair>(){

            @Override
            public Iterator<ILonePair> iterator() {
                return new LonePairIterator();
            }
        };
    }

    public Iterable<ISingleElectron> singleElectrons() {
        return new Iterable<ISingleElectron>(){

            @Override
            public Iterator<ISingleElectron> iterator() {
                return new SingleElectronIterator();
            }
        };
    }

    public Iterable<IElectronContainer> electronContainers() {
        return new Iterable<IElectronContainer>(){

            @Override
            public Iterator<IElectronContainer> iterator() {
                return new ElectronContainerIterator();
            }
        };
    }

    public IAtom getFirstAtom() {
        return this.numAtoms > 0 ? this.atoms[0] : null;
    }

    public IAtom getLastAtom() {
        return this.numAtoms > 0 ? this.atoms[this.numAtoms - 1] : null;
    }

    public int getAtomNumber(IAtom atom) {
        return this.indexOf(atom);
    }

    public int getBondNumber(IAtom beg, IAtom end) {
        return this.indexOf(this.getBond(beg, end));
    }

    public int getBondNumber(IBond bond) {
        return this.indexOf(bond);
    }

    public int getLonePairNumber(ILonePair lonePair) {
        return this.indexOf(lonePair);
    }

    public int getSingleElectronNumber(ISingleElectron singleElectron) {
        return this.indexOf(singleElectron);
    }

    public int indexOf(IAtom atom) {
        BaseAtomRef aref = this.getAtomRefUnsafe(atom);
        return aref == null ? -1 : aref.getIndex();
    }

    public int indexOf(IBond bond) {
        BondRef bref = this.getBondRefUnsafe(bond);
        return bref == null ? -1 : bref.getIndex();
    }

    public int indexOf(ISingleElectron electron) {
        for (int i = 0; i < this.numSingleElectrons; ++i) {
            if (this.electrons[i] != electron) continue;
            return i;
        }
        return -1;
    }

    public int indexOf(ILonePair pair) {
        for (int i = 0; i < this.numLonePairs; ++i) {
            if (this.lonepairs[i] != pair) continue;
            return i;
        }
        return -1;
    }

    public IElectronContainer getElectronContainer(int number) {
        if (number < this.numBonds) {
            return this.bonds[number];
        }
        if ((number -= this.numBonds) < this.numLonePairs) {
            return this.lonepairs[number];
        }
        if ((number -= this.numLonePairs) < this.numSingleElectrons) {
            return this.electrons[number];
        }
        return null;
    }

    public IBond getBond(IAtom beg, IAtom end) {
        BaseAtomRef begref = this.getAtomRefUnsafe(beg);
        return begref != null ? begref.getBond(end) : null;
    }

    public int getAtomCount() {
        return this.numAtoms;
    }

    public int getBondCount() {
        return this.numBonds;
    }

    public int getLonePairCount() {
        return this.numLonePairs;
    }

    public int getSingleElectronCount() {
        return this.numSingleElectrons;
    }

    public int getElectronContainerCount() {
        return this.numBonds + this.numSingleElectrons + this.numLonePairs;
    }

    public List<IAtom> getConnectedAtomsList(IAtom atom) {
        BaseAtomRef aref = this.getAtomRef(atom);
        ArrayList<IAtom> nbrs = new ArrayList<IAtom>(aref.getBondCount());
        for (IBond bond : aref.bonds()) {
            nbrs.add(bond.getOther(atom));
        }
        return nbrs;
    }

    public List<IBond> getConnectedBondsList(IAtom atom) {
        BaseAtomRef atomref = this.getAtomRef(atom);
        return new ArrayList<IBond>(atomref.bonds);
    }

    public List<ILonePair> getConnectedLonePairsList(IAtom atom) {
        this.getAtomRef(atom);
        ArrayList<ILonePair> lps = new ArrayList<ILonePair>();
        for (int i = 0; i < this.numLonePairs; ++i) {
            if (!this.lonepairs[i].contains(atom)) continue;
            lps.add(this.lonepairs[i]);
        }
        return lps;
    }

    public List<ISingleElectron> getConnectedSingleElectronsList(IAtom atom) {
        this.getAtomRef(atom);
        ArrayList<ISingleElectron> ses = new ArrayList<ISingleElectron>();
        for (int i = 0; i < this.numSingleElectrons; ++i) {
            if (!this.electrons[i].contains(atom)) continue;
            ses.add(this.electrons[i]);
        }
        return ses;
    }

    public List<IElectronContainer> getConnectedElectronContainersList(IAtom atom) {
        int i;
        ArrayList<IElectronContainer> ecs = new ArrayList<IElectronContainer>();
        BaseAtomRef aref = this.getAtomRef(atom);
        for (IBond bond : aref.bonds()) {
            ecs.add((IElectronContainer)bond);
        }
        for (i = 0; i < this.numLonePairs; ++i) {
            if (!this.lonepairs[i].contains(atom)) continue;
            ecs.add((IElectronContainer)this.lonepairs[i]);
        }
        for (i = 0; i < this.numSingleElectrons; ++i) {
            if (!this.electrons[i].contains(atom)) continue;
            ecs.add((IElectronContainer)this.electrons[i]);
        }
        return ecs;
    }

    public int getConnectedAtomsCount(IAtom atom) {
        return this.getConnectedBondsCount(atom);
    }

    public int getConnectedBondsCount(IAtom atom) {
        return this.getAtomRef(atom).getBondCount();
    }

    public int getConnectedBondsCount(int idx) {
        return this.getAtom(idx).getBondCount();
    }

    public int getConnectedLonePairsCount(IAtom atom) {
        this.getAtomRef(atom);
        int count = 0;
        for (int i = 0; i < this.numLonePairs; ++i) {
            if (!this.lonepairs[i].contains(atom)) continue;
            ++count;
        }
        return count;
    }

    public int getConnectedSingleElectronsCount(IAtom atom) {
        this.getAtomRef(atom);
        int count = 0;
        for (int i = 0; i < this.numSingleElectrons; ++i) {
            if (!this.electrons[i].contains(atom)) continue;
            ++count;
        }
        return count;
    }

    public double getBondOrderSum(IAtom atom) {
        double count = 0.0;
        for (IBond bond : this.getAtomRef(atom).bonds()) {
            IBond.Order order = bond.getOrder();
            if (order == null) continue;
            count += (double)order.numeric().intValue();
        }
        return count;
    }

    public IBond.Order getMaximumBondOrder(IAtom atom) {
        IBond.Order max = null;
        for (IBond bond : this.getAtomRef(atom).bonds()) {
            if (max != null && bond.getOrder().numeric() <= max.numeric()) continue;
            max = bond.getOrder();
        }
        if (max == null) {
            max = atom.getImplicitHydrogenCount() != null && atom.getImplicitHydrogenCount() > 0 ? IBond.Order.SINGLE : IBond.Order.UNSET;
        }
        return max;
    }

    public IBond.Order getMinimumBondOrder(IAtom atom) {
        IBond.Order min = null;
        for (IBond bond : this.getAtomRef(atom).bonds()) {
            if (min != null && bond.getOrder().numeric() >= min.numeric()) continue;
            min = bond.getOrder();
        }
        if (min == null) {
            min = atom.getImplicitHydrogenCount() != null && atom.getImplicitHydrogenCount() > 0 ? IBond.Order.SINGLE : IBond.Order.UNSET;
        }
        return min;
    }

    public void add(IAtomContainer that) {
        for (IAtom atom : that.atoms()) {
            atom.setFlag(16, false);
        }
        for (IBond bond : that.bonds()) {
            bond.setFlag(16, false);
        }
        for (IAtom atom : this.atoms()) {
            atom.setFlag(16, true);
        }
        for (IBond bond : this.bonds()) {
            bond.setFlag(16, true);
        }
        for (IStereoElement se : that.stereoElements()) {
            if (se instanceof TetrahedralChirality && !((TetrahedralChirality)se).getChiralAtom().getFlag(16)) {
                this.addStereoElement(se);
                continue;
            }
            if (se instanceof DoubleBondStereochemistry && !((DoubleBondStereochemistry)se).getStereoBond().getFlag(16)) {
                this.addStereoElement(se);
                continue;
            }
            if (!(se instanceof ExtendedTetrahedral) || ((ExtendedTetrahedral)se).focus().getFlag(16)) continue;
            this.addStereoElement(se);
        }
        for (IAtom atom : that.atoms()) {
            if (atom.getFlag(16)) continue;
            atom.setFlag(16, true);
            this.addAtom(atom);
        }
        for (IBond bond : that.bonds()) {
            if (bond.getFlag(16)) continue;
            bond.setFlag(16, true);
            this.addBond(bond);
        }
        for (IStereoElement se : that.singleElectrons()) {
            if (this.indexOf((ISingleElectron)se) >= 0) continue;
            this.addSingleElectron((ISingleElectron)se);
        }
        for (ILonePair lp : that.lonePairs()) {
            if (this.indexOf(lp) >= 0) continue;
            this.addLonePair(lp);
        }
        this.notifyChanged();
    }

    public void addAtom(IAtom atom) {
        if (this.contains(atom)) {
            return;
        }
        this.ensureAtomCapacity(this.numAtoms + 1);
        BaseAtomRef aref = this.newAtomRef(atom);
        aref.setIndex(this.numAtoms);
        this.atoms[this.numAtoms++] = aref;
        aref.addListener((IChemObjectListener)this);
        this.notifyChanged();
    }

    public void addBond(IBond bond) {
        this.ensureBondCapacity(this.numBonds + 1);
        BaseBondRef bref = this.newBondRef(bond);
        bref.setIndex(this.numBonds);
        this.addToEndpoints(bref);
        this.bonds[this.numBonds++] = bref;
        bref.addListener((IChemObjectListener)this);
        this.notifyChanged();
    }

    public void addLonePair(ILonePair lp) {
        this.ensureLonePairCapacity(this.numLonePairs + 1);
        this.lonepairs[this.numLonePairs++] = lp;
        lp.addListener((IChemObjectListener)this);
        this.notifyChanged();
    }

    public void addSingleElectron(ISingleElectron e) {
        this.ensureElectronCapacity(this.numSingleElectrons + 1);
        this.electrons[this.numSingleElectrons++] = e;
        e.addListener((IChemObjectListener)this);
        this.notifyChanged();
    }

    public void addElectronContainer(IElectronContainer ec) {
        if (ec instanceof IBond) {
            this.addBond((IBond)ec);
        }
        if (ec instanceof ILonePair) {
            this.addLonePair((ILonePair)ec);
        }
        if (ec instanceof ISingleElectron) {
            this.addSingleElectron((ISingleElectron)ec);
        }
        this.notifyChanged();
    }

    public void remove(IAtomContainer atomContainer) {
        int f;
        for (f = 0; f < atomContainer.getBondCount(); ++f) {
            this.removeBond(atomContainer.getBond(f));
        }
        for (f = 0; f < atomContainer.getLonePairCount(); ++f) {
            this.removeLonePair(atomContainer.getLonePair(f));
        }
        for (f = 0; f < atomContainer.getSingleElectronCount(); ++f) {
            this.removeSingleElectron(atomContainer.getSingleElectron(f));
        }
        for (f = 0; f < atomContainer.getAtomCount(); ++f) {
            this.removeAtom(atomContainer.getAtom(f));
        }
        this.notifyChanged();
    }

    public void removeAtomOnly(int idx) {
        if (idx >= 0 && idx < this.numAtoms) {
            --this.numAtoms;
            this.atoms[idx].removeListener((IChemObjectListener)this);
            for (int i = idx; i < this.numAtoms; ++i) {
                this.atoms[i] = this.atoms[i + 1];
                this.atoms[i].setIndex(i);
            }
            this.atoms[this.numAtoms] = null;
            this.notifyChanged();
        }
    }

    public void removeAtomOnly(IAtom atom) {
        this.removeAtomOnly(this.indexOf(atom));
    }

    public IBond removeBond(int idx) {
        BaseBondRef bond = null;
        if (idx >= 0 && idx < this.numBonds) {
            bond = this.bonds[idx];
            --this.numBonds;
            for (int i = idx; i < this.numBonds; ++i) {
                this.bonds[i] = this.bonds[i + 1];
                this.bonds[i].setIndex(i);
            }
            this.delFromEndpoints(bond);
            this.bonds[this.numBonds] = null;
            bond.removeListener((IChemObjectListener)this);
            this.notifyChanged();
        }
        return bond;
    }

    public IBond removeBond(IAtom beg, IAtom end) {
        return this.removeBond(this.indexOf(this.getBond(beg, end)));
    }

    public void removeBond(IBond bond) {
        this.removeBond(this.indexOf(bond));
    }

    public ILonePair removeLonePair(int idx) {
        ILonePair lonepair = null;
        if (idx >= 0 && idx < this.numLonePairs) {
            lonepair = this.lonepairs[idx];
            --this.numLonePairs;
            System.arraycopy(this.lonepairs, idx + 1, this.lonepairs, idx, this.numLonePairs - idx);
            this.lonepairs[this.numLonePairs] = null;
            lonepair.removeListener((IChemObjectListener)this);
            this.notifyChanged();
        }
        return lonepair;
    }

    public void removeLonePair(ILonePair lonePair) {
        this.removeLonePair(this.indexOf(lonePair));
    }

    public ISingleElectron removeSingleElectron(int idx) {
        ISingleElectron electron = null;
        if (idx >= 0 && idx < this.numSingleElectrons) {
            electron = this.electrons[idx];
            --this.numSingleElectrons;
            System.arraycopy(this.electrons, idx + 1, this.electrons, idx, this.numSingleElectrons - idx);
            this.electrons[this.numSingleElectrons] = null;
            electron.removeListener((IChemObjectListener)this);
            this.notifyChanged();
        }
        return electron;
    }

    public void removeSingleElectron(ISingleElectron electron) {
        this.removeSingleElectron(this.indexOf(electron));
    }

    public IElectronContainer removeElectronContainer(int number) {
        if (number < this.numBonds) {
            return this.removeBond(number);
        }
        if ((number -= this.numBonds) < this.numLonePairs) {
            return this.removeLonePair(number);
        }
        if ((number -= this.numLonePairs) < this.numSingleElectrons) {
            return this.removeSingleElectron(number);
        }
        return null;
    }

    public void removeElectronContainer(IElectronContainer ec) {
        if (ec instanceof IBond) {
            this.removeBond((IBond)ec);
        } else if (ec instanceof ILonePair) {
            this.removeLonePair((ILonePair)ec);
        } else if (ec instanceof ISingleElectron) {
            this.removeSingleElectron((ISingleElectron)ec);
        }
    }

    public void removeAtomAndConnectedElectronContainers(IAtom atom) {
        this.removeAtom(atom);
    }

    public void removeAtom(IAtom atom) {
        BaseAtomRef atomref = this.getAtomRefUnsafe(atom);
        if (atomref != null) {
            int i;
            if (atomref.getBondCount() > 0) {
                int newNumBonds = 0;
                for (i = 0; i < this.numBonds; ++i) {
                    if (!this.bonds[i].contains(atom)) {
                        this.bonds[newNumBonds] = this.bonds[i];
                        this.bonds[newNumBonds].setIndex(newNumBonds);
                        ++newNumBonds;
                        continue;
                    }
                    this.bonds[i].removeListener((IChemObjectListener)this);
                    this.delFromEndpoints(this.bonds[i]);
                }
                this.numBonds = newNumBonds;
            }
            int newNumSingleElectrons = 0;
            for (i = 0; i < this.numSingleElectrons; ++i) {
                if (!this.electrons[i].contains(atom)) {
                    this.electrons[newNumSingleElectrons] = this.electrons[i];
                    ++newNumSingleElectrons;
                    continue;
                }
                this.electrons[i].removeListener((IChemObjectListener)this);
            }
            this.numSingleElectrons = newNumSingleElectrons;
            int newNumLonePairs = 0;
            for (int i2 = 0; i2 < this.numLonePairs; ++i2) {
                if (!this.lonepairs[i2].contains(atom)) {
                    this.lonepairs[newNumLonePairs] = this.lonepairs[i2];
                    ++newNumLonePairs;
                    continue;
                }
                this.lonepairs[i2].removeListener((IChemObjectListener)this);
            }
            this.numLonePairs = newNumLonePairs;
            ArrayList<IStereoElement> atomElements = new ArrayList<IStereoElement>();
            for (IStereoElement element : this.stereo) {
                if (!element.contains(atom)) continue;
                atomElements.add(element);
            }
            this.stereo.removeAll(atomElements);
            this.removeAtomOnly(atomref.getIndex());
        }
    }

    public void removeAtom(int pos) {
        this.removeAtom(this.getAtom(pos));
    }

    public void removeAllElements() {
        this.removeAllElectronContainers();
        this.atoms = new BaseAtomRef[0];
        this.numAtoms = 0;
        this.stereo.clear();
    }

    public void removeAllElectronContainers() {
        this.removeAllBonds();
        this.lonepairs = new ILonePair[0];
        this.electrons = new ISingleElectron[0];
        this.numLonePairs = 0;
        this.numSingleElectrons = 0;
    }

    public void removeAllBonds() {
        this.bonds = new BaseBondRef[0];
        this.numBonds = 0;
        this.clearAdjacency();
        this.notifyChanged();
    }

    public void addBond(int beg, int end, IBond.Order order, IBond.Stereo stereo) {
        IBond bond = this.getBuilder().newBond();
        bond.setAtoms(new IAtom[]{this.getAtom(beg), this.getAtom(end)});
        bond.setOrder(order);
        bond.setStereo(stereo);
        this.addBond(bond);
    }

    public void addBond(int beg, int end, IBond.Order order) {
        this.addBond(beg, end, order, IBond.Stereo.NONE);
    }

    public void addLonePair(int idx) {
        ILonePair lp = (ILonePair)this.getBuilder().newInstance(ILonePair.class, new Object[0]);
        lp.setAtom(this.getAtom(idx));
        this.addLonePair(lp);
    }

    public void addSingleElectron(int idx) {
        ISingleElectron electron = (ISingleElectron)this.getBuilder().newInstance(ISingleElectron.class, new Object[0]);
        electron.setAtom(this.getAtom(idx));
        this.addSingleElectron(electron);
    }

    public boolean contains(IAtom atom) {
        return this.indexOf(atom) >= 0;
    }

    public boolean contains(IBond bond) {
        return this.indexOf(bond) >= 0;
    }

    public boolean contains(ILonePair lonepair) {
        return this.indexOf(lonepair) >= 0;
    }

    public boolean contains(ISingleElectron electron) {
        return this.indexOf(electron) >= 0;
    }

    public boolean contains(IElectronContainer electronContainer) {
        if (electronContainer instanceof IBond) {
            return this.contains((IBond)electronContainer);
        }
        if (electronContainer instanceof ILonePair) {
            return this.contains((ILonePair)electronContainer);
        }
        if (electronContainer instanceof ISingleElectron) {
            return this.contains((SingleElectron)electronContainer);
        }
        return false;
    }

    public String getTitle() {
        return (String)this.getProperty("cdk:Title");
    }

    public void setTitle(String title) {
        this.setProperty("cdk:Title", title);
    }

    public String toString() {
        int i;
        StringBuilder sb = new StringBuilder(64);
        sb.append("AtomContainer(");
        sb.append(this.hashCode());
        if (this.getAtomCount() > 0) {
            sb.append(", #A:").append(this.getAtomCount());
            for (i = 0; i < this.getAtomCount(); ++i) {
                sb.append(", ").append(this.getAtom(i).toString());
            }
        }
        if (this.getBondCount() > 0) {
            sb.append(", #B:").append(this.getBondCount());
            for (i = 0; i < this.getBondCount(); ++i) {
                sb.append(", ").append(this.getBond(i).toString());
            }
        }
        if (this.getLonePairCount() > 0) {
            sb.append(", #LP:").append(this.getLonePairCount());
            for (i = 0; i < this.getLonePairCount(); ++i) {
                sb.append(", ").append(this.getLonePair(i).toString());
            }
        }
        if (this.getSingleElectronCount() > 0) {
            sb.append(", #SE:").append(this.getSingleElectronCount());
            for (i = 0; i < this.getSingleElectronCount(); ++i) {
                sb.append(", ").append(this.getSingleElectron(i).toString());
            }
        }
        if (this.stereo.size() > 0) {
            sb.append(", ST:[#").append(this.stereo.size());
            for (IStereoElement elements : this.stereo) {
                sb.append(", ").append(elements.toString());
            }
            sb.append(']');
        }
        sb.append(')');
        return sb.toString();
    }

    public IAtomContainer shallowCopy() {
        return new AtomContainer2(this);
    }

    public IAtomContainer clone() throws CloneNotSupportedException {
        BaseBondRef original;
        int i;
        int i2;
        AtomContainer2 clone = (AtomContainer2)super.clone();
        clone.numAtoms = 0;
        clone.numBonds = 0;
        clone.numSingleElectrons = 0;
        clone.numLonePairs = 0;
        clone.atoms = new BaseAtomRef[0];
        clone.bonds = new BaseBondRef[0];
        clone.electrons = new ISingleElectron[0];
        clone.lonepairs = new ILonePair[0];
        clone.stereo = new ArrayList<IStereoElement>();
        clone.removeAllElements();
        HashMap<BaseAtomRef, IAtom> atomMap = new HashMap<BaseAtomRef, IAtom>(this.numAtoms + (this.numAtoms >> 1));
        HashMap<BaseBondRef, IBond> bondMap = new HashMap<BaseBondRef, IBond>(this.numBonds + (this.numBonds >> 1));
        IAtom[] atoms = new IAtom[this.numAtoms];
        for (i2 = 0; i2 < atoms.length; ++i2) {
            atoms[i2] = this.atoms[i2].deref().clone();
        }
        clone.setAtoms(atoms);
        for (i2 = 0; i2 < atoms.length; ++i2) {
            atomMap.put(this.atoms[i2], clone.getAtom(i2));
        }
        IBond[] bonds = new IBond[this.numBonds];
        for (i = 0; i < bonds.length; ++i) {
            original = this.bonds[i];
            IBond bond = original.deref().clone();
            int n = bond.getAtomCount();
            IAtom[] members = new IAtom[n];
            for (int j = 0; j < n; ++j) {
                members[j] = (IAtom)atomMap.get(original.getAtom(j));
            }
            bond.setAtoms(members);
            bondMap.put(this.bonds[i], bond);
            bonds[i] = bond;
        }
        clone.setBonds(bonds);
        for (i = 0; i < bonds.length; ++i) {
            bondMap.put(this.bonds[i], clone.getBond(i));
        }
        for (i = 0; i < this.numLonePairs; ++i) {
            original = this.lonepairs[i];
            ILonePair pair = (ILonePair)original.clone();
            if (pair.getAtom() != null) {
                pair.setAtom((IAtom)atomMap.get(original.getAtom()));
            }
            clone.addLonePair(pair);
        }
        for (i = 0; i < this.numSingleElectrons; ++i) {
            original = this.electrons[i];
            ISingleElectron electron = (ISingleElectron)original.clone();
            if (electron.getAtom() != null) {
                electron.setAtom((IAtom)atomMap.get(original.getAtom()));
            }
            clone.addSingleElectron(electron);
        }
        for (IStereoElement element : this.stereo) {
            clone.addStereoElement(element.map(atomMap, bondMap));
        }
        Collection sgroups = (Collection)this.getProperty("cdk:CtabSgroups");
        if (sgroups != null) {
            HashMap<Object, Object> replace = new HashMap<Object, Object>();
            replace.putAll(atomMap);
            replace.putAll(bondMap);
            clone.setProperty("cdk:CtabSgroups", SgroupManipulator.copy((Collection)sgroups, replace));
        }
        return clone;
    }

    public void stateChanged(IChemObjectChangeEvent event) {
        this.notifyChanged(event);
    }

    public boolean isEmpty() {
        return this.numAtoms == 0;
    }

    private static final class QueryBondRef
    extends BaseBondRef
    implements IQueryBond {
        public QueryBondRef(AtomContainer2 mol, IQueryBond bond, BaseAtomRef beg, BaseAtomRef end) {
            super(mol, (IBond)bond, beg, end);
        }

        public boolean matches(IBond bond) {
            return ((IQueryBond)this.deref()).matches(bond);
        }
    }

    private static class BaseBondRef
    extends BondRef {
        private int idx;
        private final AtomContainer2 mol;
        private BaseAtomRef beg;
        private BaseAtomRef end;

        private BaseBondRef(AtomContainer2 mol, IBond bond, BaseAtomRef beg, BaseAtomRef end) {
            super(bond);
            this.mol = mol;
            this.beg = beg;
            this.end = end;
        }

        public int getIndex() {
            return this.idx;
        }

        public void setIndex(int idx) {
            this.idx = idx;
        }

        public IAtomContainer getContainer() {
            return this.mol;
        }

        public BaseAtomRef getBegin() {
            return this.beg;
        }

        public BaseAtomRef getEnd() {
            return this.end;
        }

        public IAtom getAtom(int idx) {
            switch (idx) {
                case 0: {
                    return this.getBegin();
                }
                case 1: {
                    return this.getEnd();
                }
            }
            return this.mol.getAtomRef(super.getAtom(idx));
        }

        public AtomRef getOther(IAtom atom) {
            if (atom == this.beg) {
                return this.end;
            }
            if (atom == this.end) {
                return this.beg;
            }
            if ((atom = AtomContainer2.unbox(atom)) == this.beg.deref()) {
                return this.end;
            }
            if (atom == this.end.deref()) {
                return this.beg;
            }
            return null;
        }

        public void setAtoms(IAtom[] atoms) {
            assert (atoms.length == 2);
            super.setAtoms(atoms);
            if (Objects.equals(atoms[0], (Object)this.end) && Objects.equals(atoms[1], (Object)this.beg)) {
                BaseAtomRef tmp = this.beg;
                this.beg = this.end;
                this.end = tmp;
                return;
            }
            if (this.beg != null) {
                this.beg.bonds.remove((Object)this);
            }
            if (this.end != null) {
                this.end.bonds.remove((Object)this);
            }
            this.beg = this.mol.getAtomRef(atoms[0]);
            this.end = this.mol.getAtomRef(atoms[1]);
            this.beg.bonds.add(this);
            this.end.bonds.add(this);
        }

        public void setAtom(IAtom atom, int idx) {
            super.setAtom(atom, idx);
            if (idx == 0) {
                if (this.beg != null) {
                    this.beg.bonds.remove((Object)this);
                }
                this.beg = this.mol.getAtomRef(atom);
                this.beg.bonds.add(this);
            } else if (idx == 1) {
                if (this.end != null) {
                    this.end.bonds.remove((Object)this);
                }
                this.end = this.mol.getAtomRef(atom);
                this.end.bonds.add(this);
            }
        }

        public int hashCode() {
            return this.deref().hashCode();
        }

        public boolean equals(Object obj) {
            if (obj instanceof BondRef) {
                return this.deref().equals(((BondRef)obj).deref());
            }
            return this.deref().equals(obj);
        }
    }

    private static final class PdbAtomRef
    extends BaseAtomRef
    implements IPDBAtom {
        private final IPDBAtom pdbAtom;

        private PdbAtomRef(IAtomContainer mol, IPDBAtom atom) {
            super(mol, (IAtom)atom);
            this.pdbAtom = atom;
        }

        public String getRecord() {
            return this.pdbAtom.getRecord();
        }

        public void setRecord(String newRecord) {
            this.pdbAtom.setRecord(newRecord);
        }

        public Double getTempFactor() {
            return this.pdbAtom.getTempFactor();
        }

        public void setTempFactor(Double newTempFactor) {
            this.pdbAtom.setTempFactor(newTempFactor);
        }

        public String getResName() {
            return this.pdbAtom.getResName();
        }

        public void setResName(String newResName) {
            this.pdbAtom.setResName(newResName);
        }

        public String getICode() {
            return this.pdbAtom.getICode();
        }

        public void setICode(String newICode) {
            this.pdbAtom.setICode(newICode);
        }

        public String getName() {
            return this.pdbAtom.getName();
        }

        public void setName(String newName) {
            this.pdbAtom.setName(newName);
        }

        public String getChainID() {
            return this.pdbAtom.getChainID();
        }

        public void setChainID(String newChainID) {
            this.pdbAtom.setChainID(newChainID);
        }

        public String getAltLoc() {
            return this.pdbAtom.getAltLoc();
        }

        public void setAltLoc(String newAltLoc) {
            this.pdbAtom.setAltLoc(newAltLoc);
        }

        public String getSegID() {
            return this.pdbAtom.getSegID();
        }

        public void setSegID(String newSegID) {
            this.pdbAtom.setSegID(newSegID);
        }

        public Integer getSerial() {
            return this.pdbAtom.getSerial();
        }

        public void setSerial(Integer newSerial) {
            this.pdbAtom.setSerial(newSerial);
        }

        public String getResSeq() {
            return this.pdbAtom.getResSeq();
        }

        public void setResSeq(String newResSeq) {
            this.pdbAtom.setResSeq(newResSeq);
        }

        public Boolean getOxt() {
            return this.pdbAtom.getOxt();
        }

        public void setOxt(Boolean newOxt) {
            this.pdbAtom.setOxt(newOxt);
        }

        public Boolean getHetAtom() {
            return this.pdbAtom.getHetAtom();
        }

        public void setHetAtom(Boolean newHetAtom) {
            this.pdbAtom.setHetAtom(newHetAtom);
        }

        public Double getOccupancy() {
            return this.pdbAtom.getOccupancy();
        }

        public void setOccupancy(Double newOccupancy) {
            this.pdbAtom.setOccupancy(newOccupancy);
        }
    }

    private static final class QueryAtomRef
    extends BaseAtomRef
    implements IQueryAtom {
        private final IQueryAtom qatom;

        private QueryAtomRef(IAtomContainer mol, IQueryAtom atom) {
            super(mol, (IAtom)atom);
            this.qatom = atom;
        }

        public boolean matches(IAtom atom) {
            return this.qatom.matches(atom);
        }
    }

    private static final class PsuedoAtomRef
    extends BaseAtomRef
    implements IPseudoAtom {
        private final IPseudoAtom pseudo;

        private PsuedoAtomRef(IAtomContainer mol, IPseudoAtom atom) {
            super(mol, (IAtom)atom);
            this.pseudo = atom;
        }

        public String getLabel() {
            return this.pseudo.getLabel();
        }

        public void setLabel(String label) {
            this.pseudo.setLabel(label);
        }

        public int getAttachPointNum() {
            return this.pseudo.getAttachPointNum();
        }

        public void setAttachPointNum(int ap) {
            this.pseudo.setAttachPointNum(ap);
        }

        @Override
        public int hashCode() {
            return this.deref().hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (obj instanceof BondRef) {
                return this.deref().equals(((BondRef)obj).deref());
            }
            return this.deref().equals(obj);
        }

        public IPseudoAtom clone() throws CloneNotSupportedException {
            return (IPseudoAtom)super.clone();
        }
    }

    private static class BaseAtomRef
    extends AtomRef {
        private int idx = -1;
        private final IAtomContainer mol;
        private final List<IBond> bonds = new ArrayList<IBond>(4);

        private BaseAtomRef(IAtomContainer mol, IAtom atom) {
            super(atom);
            this.mol = mol;
        }

        public final int getIndex() {
            return this.idx;
        }

        public final void setIndex(int idx) {
            this.idx = idx;
        }

        public final IAtomContainer getContainer() {
            return this.mol;
        }

        public final int getBondCount() {
            return this.bonds.size();
        }

        public final Iterable<IBond> bonds() {
            return this.bonds;
        }

        public IBond getBond(IAtom atom) {
            for (IBond bond : this.bonds) {
                if (!bond.getOther((IAtom)this).equals(atom)) continue;
                return bond;
            }
            return null;
        }

        public int hashCode() {
            return this.deref().hashCode();
        }

        public boolean equals(Object obj) {
            if (obj instanceof AtomRef) {
                return this.deref().equals(((AtomRef)obj).deref());
            }
            return this.deref().equals(obj);
        }
    }

    private class ElectronContainerIterator
    implements Iterator<IElectronContainer> {
        private int idx = 0;

        private ElectronContainerIterator() {
        }

        @Override
        public boolean hasNext() {
            return this.idx < AtomContainer2.this.numBonds + AtomContainer2.this.numLonePairs + AtomContainer2.this.numSingleElectrons;
        }

        @Override
        public IElectronContainer next() {
            if (this.idx < AtomContainer2.this.numBonds) {
                return AtomContainer2.this.bonds[this.idx++];
            }
            if (this.idx < AtomContainer2.this.numBonds + AtomContainer2.this.numLonePairs) {
                return AtomContainer2.this.lonepairs[this.idx++ - AtomContainer2.this.numBonds];
            }
            if (this.idx < AtomContainer2.this.numBonds + AtomContainer2.this.numLonePairs + AtomContainer2.this.numSingleElectrons) {
                return AtomContainer2.this.electrons[this.idx++ - (AtomContainer2.this.numBonds + AtomContainer2.this.numLonePairs)];
            }
            return null;
        }

        @Override
        public void remove() {
            if (this.idx <= AtomContainer2.this.numBonds) {
                AtomContainer2.this.removeBond(--this.idx);
            } else if (this.idx <= AtomContainer2.this.numBonds + AtomContainer2.this.numLonePairs) {
                AtomContainer2.this.removeLonePair(--this.idx - AtomContainer2.this.numBonds);
            } else if (this.idx <= AtomContainer2.this.numBonds + AtomContainer2.this.numLonePairs + AtomContainer2.this.numSingleElectrons) {
                AtomContainer2.this.removeSingleElectron(--this.idx - (AtomContainer2.this.numBonds - AtomContainer2.this.numLonePairs));
            }
        }
    }

    private class SingleElectronIterator
    implements Iterator<ISingleElectron> {
        private int idx = 0;

        private SingleElectronIterator() {
        }

        @Override
        public boolean hasNext() {
            return this.idx < AtomContainer2.this.numSingleElectrons;
        }

        @Override
        public ISingleElectron next() {
            return AtomContainer2.this.electrons[this.idx++];
        }

        @Override
        public void remove() {
            AtomContainer2.this.removeSingleElectron(--this.idx);
        }
    }

    private class LonePairIterator
    implements Iterator<ILonePair> {
        private int idx = 0;

        private LonePairIterator() {
        }

        @Override
        public boolean hasNext() {
            return this.idx < AtomContainer2.this.numLonePairs;
        }

        @Override
        public ILonePair next() {
            return AtomContainer2.this.lonepairs[this.idx++];
        }

        @Override
        public void remove() {
            AtomContainer2.this.removeLonePair(--this.idx);
        }
    }

    private class BondIterator
    implements Iterator<IBond> {
        private int idx = 0;

        private BondIterator() {
        }

        @Override
        public boolean hasNext() {
            return this.idx < AtomContainer2.this.numBonds;
        }

        @Override
        public IBond next() {
            return AtomContainer2.this.bonds[this.idx++];
        }

        @Override
        public void remove() {
            AtomContainer2.this.removeBond(--this.idx);
        }
    }

    private class AtomIterator
    implements Iterator<IAtom> {
        private int idx = 0;

        private AtomIterator() {
        }

        @Override
        public boolean hasNext() {
            return this.idx < AtomContainer2.this.numAtoms;
        }

        @Override
        public IAtom next() {
            return AtomContainer2.this.atoms[this.idx++];
        }

        @Override
        public void remove() {
            AtomContainer2.this.removeAtomOnly(--this.idx);
        }
    }
}

