/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: PublishAttachmentsMojo.java 157 2010-10-08 10:23:16Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.1.4/maven-jredmine-plugin/src/main/java/org/nuiton/jredmine/plugin/PublishAttachmentsMojo.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;


import org.apache.maven.plugin.MojoExecutionException;
import org.nuiton.helper.plugin.CollectFilesMojo;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * Publish files for a given project and version on redmine server.
 *
 * @author tchemit
 * @goal publish-attachments
 * @since 1.0.0
 */
public class PublishAttachmentsMojo extends AbstractRedmineMojo {
    /**
     * Flag to know if anonymùous connexion to redmine server is required.
     * <p/>
     * <b>Note:</b> If set to {@code false}, you should fill {@link #username}
     * and {@link #password} properties.
     *
     * @parameter expression="${redmine.anonymous}" default-value="false"
     * @since 1.1.3
     */
    protected boolean anonymous;

    /**
     * The path of a properties files where files to deploy are descriped.
     * <p/>
     * The key of a property is the sortor key
     * <p/>
     * The value of a property is the path of the file to upload.
     * <p/>
     * If no Set - will not use this source
     *
     * @parameter expression="${redmine.filesFromProperties}"
     * @since 1.0.0
     */
    protected File filesFromProperties;

    /**
     * A list of files to deploy.
     * <p/>
     * If no Set - will not use this source
     *
     * @parameter expression="${redmine.files}"
     * @since 1.0.0
     */
    protected File[] files;

    /**
     * A flag to skip the goal.
     *
     * @parameter expression="${redmine.skipCollectReleaseAttachments}" default-value="false"
     * @since 1.0.0
     */
    protected boolean skipPublishAttachments;

    /**
     * A flag to test plugin but send nothing to redmine.
     *
     * @parameter expression="${redmine.dryRun}" default-value="false"
     * @since 1.0.0
     */
    protected boolean dryRun;

    /**
     * A flag to restirct only one run in a build (for multi-module context).
     *
     * @parameter expression="${redmine.runOnce}" default-value="true"
     * @since 1.0.0
     */
    protected boolean runOnce;

    /** files to deploy */
    protected List<File> allFiles;

    public PublishAttachmentsMojo() {
        super(true, true, true);
    }

    @Override
    public boolean isAnonymous() {
        return anonymous;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }

    @Override
    protected boolean isGoalSkip() {
        return skipPublishAttachments;
    }

    @Override
    protected boolean isRunOnce() {
        return runOnce;
    }

    @Override
    protected boolean checkRunOnceDone() {
        return isRunOnce() && !isExecutionRoot();
    }

    @Override
    protected void init() throws Exception {

        versionId = PluginHelper.removeSnapshotSuffix(versionId);

        if (filesFromProperties != null && !filesFromProperties.exists()) {
            // no file to publish...
            getLog().warn("could not find the properties file " + filesFromProperties);
            getLog().warn("will not use this source");
            filesFromProperties = null;
        }

        if (filesFromProperties != null && files != null && files.length > 0) {
            throw new MojoExecutionException("can not use both sources files and filesFromProperties");
        }

        if (filesFromProperties != null) {

            allFiles = getFilesToDeploy();

        } else {
            allFiles = Arrays.asList(files);
        }

        if (allFiles == null || allFiles.isEmpty()) {
            return;
        }

        runOnceDone = false;

        if (isRunOnce()) {
            runOnceDone = checkRunOnceDone();
            if (runOnceDone) {
                return;
            }
        }

        super.init();
    }

    @Override
    protected boolean checkSkip() {
        if (allFiles == null || allFiles.isEmpty()) {
            getLog().warn("No attachment to publish, will skip the goal.");
            return false;
        }

        boolean b = super.checkSkip();
        return b;
    }

    @Override
    protected void doAction() throws Exception {

        if (dryRun) {
            getLog().info("\n  dryRun flag is on, no data will be send!\n");
        }

        String basedir = project.getBasedir().getAbsolutePath();
        int basedirLength = basedir.length() + 1;
        for (File f : allFiles) {

            if (!f.exists()) {
                getLog().warn("can not upload a non existing file " + f);
                continue;
            }

            if (f.getAbsolutePath().startsWith(basedir)) {
                getLog().info("upload " + f.getAbsolutePath().substring(basedirLength));
            } else {
                getLog().info("upload " + f.getAbsolutePath());
            }

            if (dryRun) {
                continue;
            }

            // upload the file

            long t0 = System.nanoTime();
            Attachment a = new Attachment();
            a.setFilename(f.getName());
            a.setAuthorId(releaseUser.getId());
            a.setContainerId(releaseVersion.getId());

            a.setToUpload(f);

            // do upload
            Attachment result = service.addAttachment(projectId, releaseVersion.getName(), a);
            long t1 = System.nanoTime();
            if (verbose) {
                getLog().info("done in  " + PluginHelper.convertTime(t1 - t0) + ". attachment id : " + result.getId() + ", size : " + PluginHelper.convertMemory(result.getFilesize()));
            }
        }
    }

    protected List<File> getFilesToDeploy() throws MojoExecutionException {

        // load definition file
        List<File> incoming;

        try {
            incoming = new CollectFilesMojo().getFiles(filesFromProperties);
            getLog().info("Loaded " + filesFromProperties);
        } catch (IOException ex) {
            throw new MojoExecutionException("could not load file " + filesFromProperties, ex);
        }

        // sort attachments by file names 

        Collections.sort(incoming, new Comparator<File>() {

            @Override
            public int compare(File o1, File o2) {

                return o1.getName().compareTo(o2.getName());
            }
        });

        return incoming;
    }
}

