/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: PublishAttachmentsMojo.java 364 2012-10-13 21:50:04Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.5/jredmine-maven-plugin/src/main/java/org/nuiton/jredmine/plugin/PublishAttachmentsMojo.java $
 * %%
 * Copyright (C) 2009 - 2012 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;


import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.nuiton.helper.plugin.CollectFilesMojo;
import org.nuiton.jredmine.model.Attachment;
import org.nuiton.plugin.PluginHelper;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * Publish files for a given project and version on redmine server.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0.0
 */
@Mojo(name = "publish-attachments", requiresOnline = true, requiresProject = true)
public class PublishAttachmentsMojo extends AbstractRedmineMojoWithProjectAndVersion implements DryRunAware {

    ///////////////////////////////////////////////////////////////////////////
    /// Mojo parameters
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Flag to know if anonymous connexion to redmine server is required.
     * <p/>
     * <b>Note:</b> If set to {@code false}, you should fill {@link #username}
     * and {@link #password} properties.
     *
     * @since 1.1.3
     */
    @Parameter(property = "redmine.anonymous", defaultValue = "false")
    protected boolean anonymous;

    /**
     * The path of a properties files where files to deploy are descriped.
     * <p/>
     * The key of a property is the sortor key
     * <p/>
     * The value of a property is the path of the file to upload.
     * <p/>
     * If no Set - will not use this source
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.filesFromProperties")
    protected File filesFromProperties;

    /**
     * A list of files to deploy.
     * <p/>
     * If no Set - will not use this source
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.files")
    protected File[] files;

    /**
     * A flag to skip the goal.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.skipCollectReleaseAttachments", defaultValue = "false")
    protected boolean skipPublishAttachments;

    /**
     * A flag to test plugin but send nothing to redmine.
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.dryRun", defaultValue = "false")
    protected boolean dryRun;

    /**
     * A flag to restrict only one run in a build (for multi-module context).
     *
     * @since 1.0.0
     */
    @Parameter(property = "redmine.runOnce", defaultValue = "true")
    protected boolean runOnce;

    ///////////////////////////////////////////////////////////////////////////
    /// Mojo internal attributes
    ///////////////////////////////////////////////////////////////////////////

    /** files to deploy */
    protected List<File> allFiles;

    public PublishAttachmentsMojo() {
        super(true, true);
    }

    ///////////////////////////////////////////////////////////////////////////
    /// RedmineClientConfiguration
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public boolean isAnonymous() {
        return anonymous;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// DryRunAware
    ///////////////////////////////////////////////////////////////////////////

    @Override
    public boolean isDryRun() {
        return dryRun;
    }

    @Override
    public void setDryRun(boolean dryRun) {
        this.dryRun = dryRun;
    }

    ///////////////////////////////////////////////////////////////////////////
    /// AbstractRedmineMojo
    ///////////////////////////////////////////////////////////////////////////

    @Override
    protected boolean isGoalSkip() {
        return skipPublishAttachments;
    }

    @Override
    protected boolean isRunOnce() {
        return runOnce;
    }

    @Override
    protected boolean checkRunOnceDone() {
        return isRunOnce() && !isExecutionRoot();
    }

    ///////////////////////////////////////////////////////////////////////////
    /// AbstractPlugin
    ///////////////////////////////////////////////////////////////////////////

    @Override
    protected void init() throws Exception {

        versionId = PluginHelper.removeSnapshotSuffix(versionId);

        if (filesFromProperties != null && !filesFromProperties.exists()) {
            // no file to publish...
            getLog().warn("could not find the properties file " + filesFromProperties);
            getLog().warn("will not use this source");
            filesFromProperties = null;
        }

        if (filesFromProperties != null && files != null && files.length > 0) {
            throw new MojoExecutionException("can not use both sources files and filesFromProperties");
        }

        if (filesFromProperties != null) {

            allFiles = getFilesToDeploy();

        } else {
            allFiles = Arrays.asList(files);
        }

        if (allFiles == null || allFiles.isEmpty()) {
            return;
        }

        runOnceDone = false;

        if (isRunOnce()) {
            runOnceDone = checkRunOnceDone();
            if (runOnceDone) {
                return;
            }
        }

        super.init();
    }

    @Override
    protected boolean checkSkip() {
        if (allFiles == null || allFiles.isEmpty()) {
            getLog().warn("No attachment to publish, will skip the goal.");
            return false;
        }

        boolean b = super.checkSkip();
        return b;
    }

    @Override
    protected void doAction() throws Exception {

        if (dryRun) {
            getLog().info("\n  dryRun flag is on, no data will be send!\n");
        }

        String basedir = project.getBasedir().getAbsolutePath();
        int basedirLength = basedir.length() + 1;
        for (File f : allFiles) {

            if (!f.exists()) {
                getLog().warn("can not upload a non existing file " + f);
                continue;
            }

            if (f.getAbsolutePath().startsWith(basedir)) {
                getLog().info("upload " + f.getAbsolutePath().substring(basedirLength));
            } else {
                getLog().info("upload " + f.getAbsolutePath());
            }

            if (dryRun) {
                continue;
            }

            // upload the file

            long t0 = System.nanoTime();
            Attachment a = new Attachment();
            a.setFilename(f.getName());
            a.setAuthorId(releaseUser.getId());
            a.setContainerId(releaseVersion.getId());

            a.setToUpload(f);

            // do upload
            Attachment result = service.addAttachment(projectId, releaseVersion.getName(), a);
            long t1 = System.nanoTime();
            if (isVerbose()) {
                getLog().info("done in  " + PluginHelper.convertTime(t1 - t0) + ". attachment id : " + result.getId() + ", size : " + PluginHelper.convertMemory(result.getFilesize()));
            }
        }
    }

    ///////////////////////////////////////////////////////////////////////////
    /// Others
    ///////////////////////////////////////////////////////////////////////////

    protected List<File> getFilesToDeploy() throws MojoExecutionException {

        // load definition file
        List<File> incoming;

        try {
            incoming = new CollectFilesMojo().getFiles(filesFromProperties);
            getLog().info("Loaded " + filesFromProperties);
        } catch (IOException ex) {
            throw new MojoExecutionException("could not load file " + filesFromProperties, ex);
        }

        // sort attachments by file names 

        Collections.sort(incoming, new Comparator<File>() {

            @Override
            public int compare(File o1, File o2) {

                return o1.getName().compareTo(o2.getName());
            }
        });

        return incoming;
    }
}

