/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: ParserGWTJavaMojo.java 1971 2012-08-29 19:38:52Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.5.2/i18n-maven-plugin/src/main/java/org/nuiton/i18n/plugin/parser/impl/ParserGWTJavaMojo.java $
 * %%
 * Copyright (C) 2011 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.i18n.plugin.parser.impl;

import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.nuiton.i18n.plugin.parser.AbstractFileParser;
import org.nuiton.i18n.plugin.parser.AbstractI18nParserMojo;
import org.nuiton.i18n.plugin.parser.FileParser;
import org.nuiton.i18n.plugin.parser.I18nSourceEntry;
import org.nuiton.i18n.plugin.parser.ParserException;
import org.nuiton.i18n.plugin.parser.SourceEntry;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.MirroredFileUpdater;
import org.nuiton.io.SortedProperties;
import org.nuiton.processor.ProcessorUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * To detect from GWT java files new I18n keys, says content of patterns :
 * <ul>
 * <li>{@code @Key("XXX")}</li>
 * <li>{@code @LocalizableResource.Key("XXX")}</li>
 * <li>{@code @com.google.gwt.i18n.client.LocalizableResource.Key("XXX")}</li>
 * </ul>
 * <b>Note: </b> this goal must always be invoked before the {@code process-resources}
 * phase, otherwise all files will be considered as uptodate.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.1
 */
@Mojo(name = "parserGWTJava", defaultPhase = LifecyclePhase.GENERATE_RESOURCES)
public class ParserGWTJavaMojo extends AbstractI18nParserMojo {

    public static final String DEFAULT_INCLUDES = "**/*.java";

    /** Root directory of the default entry. */
    @Parameter(property = "i18n.defaultBasedir", defaultValue = "${basedir}/src/main/java")
    protected File defaultBasedir;

    /**
     * Default included files to process (ant-like expression).
     * <p/>
     * <strong>Note:</strong> default value is **\/*.java
     */
    @Parameter(property = "i18n.defaultIncludes", defaultValue = DEFAULT_INCLUDES, required = true)
    protected String defaultIncludes;

    /**
     * Defines the file name of the getter where to put detected i18n keys
     * while getter phase.
     *
     * @since 2.0
     */
    @Parameter(property = "i18n.outputGetter", defaultValue = "gwt-java.getter")
    protected String outputGetter;

    /**
     * Where to generated temporary processed files.
     *
     * @since 2.0
     */
    @Parameter(property = "i18n.workdir", defaultValue = "${basedir}/target/i18n-workdir")
    protected File workdir;

    protected MirroredFileUpdater entryUpdater;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    @Override
    protected boolean onEnterEntry(I18nSourceEntry entry) {
        boolean b = super.onEnterEntry(entry);
        if (!b) {

            // no skipped entry
            // keep the file updater
            entryUpdater = (MirroredFileUpdater) entry.getUpdater();
        }
        return b;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return new MirroredFileUpdater("", "", entry.getBasedir(), workdir) {

            @Override
            public File getMirrorFile(File f) {
                String file =
                        f.getAbsolutePath().substring(prefixSourceDirecotory);
                return new File(destinationDirectory + File.separator + file);
            }
        };
    }

    @Override
    protected String getOutGetter() {
        return outputGetter;
    }

    @Override
    public FileParser newFileParser(Pattern acceptPattern) {

        return new GWTJavaFileParser(getLog(),
                                     encoding,
                                     oldParser,
                                     acceptPattern,
                                     isShowTouchedFiles()
        );
    }

    protected class GWTJavaFileParser extends AbstractFileParser {


        public GWTJavaFileParser(Log log,
                                 String encoding,
                                 SortedProperties oldParser,
                                 Pattern acceptKeyPattern,
                                 boolean showTouchedFiles) {
            super(log, encoding, oldParser, acceptKeyPattern,showTouchedFiles);
        }


        protected File prepareFile(File file) throws IOException {

            // clean the jsp to make it xml

            File result = entryUpdater.getMirrorFile(file);
            if (isVerbose()) {
                getLog().info("Will process [" + file + "] to " + result);
            }
            try {
                createDirectoryIfNecessary(result.getParentFile());
            } catch (IOException e) {
                // don't care about it...
            }
            GWTJavaProcessor processor = new GWTJavaProcessor();
            processor.setVerbose(isVerbose());
            processor.process(file, result, getEncoding());
            return result;
        }

        @Override
        public void parseFile(File file) throws IOException {

            // clean the jsp to make it xml

            File processedFile = prepareFile(file);

            // process file to obtain
            String line = null;
            LineNumberReader lnr = new LineNumberReader(new InputStreamReader(
                    new FileInputStream(processedFile), getEncoding()));
            try {
                while ((line = lnr.readLine()) != null) {
                    parseLine(processedFile, line);
                }
            } catch (Exception e) {
                if (line != null) {
                    getLog().error(
                            "could not parse line (" + lnr.getLineNumber() + ") '"
                            + line + "' of file " + file);
                }
                throw new ParserException(e);
            } finally {
                lnr.close();
            }
        }

        @Override
        public void parseLine(File file, String line) throws IOException {

            String key = line.trim();

            if (key.isEmpty()) {
                // no key detected on this line
                return;
            }

//            // one key found in file, so file is marked as touched
//            setTouched(true);
            // Found a i18n key, register it.
            if (getLog().isDebugEnabled()) {
                getLog().debug(file.getName() + " detected key = " + key);
            }
            registerKey(key);
        }
    }

    /**
     * To Extract i18n keys from GWT java files.
     * <p/>
     * Says keep content only of those patterns :
     * <ul>
     * <li>{@code @Key("XXX")}</li>
     * <li>{@code @LocalizableResource.Key("XXX")}</li>
     * <li>{@code @com.google.gwt.i18n.client.LocalizableResource.Key("XXX")}</li>
     * </ul>
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.1
     */
    public static class GWTJavaProcessor extends ProcessorHelper.AbstractParserProcessor {

        protected ProcessorUtil.FragmentExtractor filter1 =
                new ProcessorUtil.FragmentExtractor(
                        "@Key(\"",
                        "\")"
                );

        protected ProcessorUtil.FragmentExtractor filter2 =
                new ProcessorUtil.FragmentExtractor(
                        "@LocalizableResource.Key(\"",
                        "\")"
                );

        protected ProcessorUtil.FragmentExtractor filter3 =
                new ProcessorUtil.FragmentExtractor(
                        "@com.google.gwt.i18n.client.LocalizableResource.Key(\"",
                        "\")"
                );


        @Override
        public void process(File filein,
                            File fileout,
                            String encoding) throws IOException {

            Set<String> keys = new HashSet<String>();
            extractKeys(filter1, filein, encoding, verbose, keys);
            extractKeys(filter2, filein, encoding, verbose, keys);
            extractKeys(filter3, filein, encoding, verbose, keys);
            saveKeysToFile(fileout, keys);
        }

    }
}