/*
 * #%L
 * EUGene :: Maven plugin
 * 
 * $Id: EugenePlugin.java 906 2010-05-16 12:19:55Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/eugene/tags/eugene-2.0.2/maven-eugene-plugin/src/main/java/org/nuiton/eugene/plugin/EugenePlugin.java $
 * %%
 * Copyright (C) 2006 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.eugene.plugin;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.nuiton.eugene.DefaultTemplateConfiguration;
import org.nuiton.eugene.ModelReader;
import org.nuiton.eugene.Template;
import org.nuiton.eugene.models.Model;
import org.nuiton.plugin.PluginHelper;
import org.nuiton.plugin.PluginIOContext;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.*;
import java.util.Map.Entry;

/**
 * Effectue toutes les générations et copie les fichiers générés
 * dans le répertoire de compilation
 *
 * @author ruchaud
 * @version $Revision: 906 $
 *          <p/>
 *          Last update: $Date: 2010-05-16 14:19:55 +0200 (dim., 16 mai 2010) $
 *          by : * @goal generate
 * @requiresProject true
 * @deprecated since 2.0.0, use now the {@link SmartGenerateMojo}.
 */
@Deprecated
public class EugenePlugin extends EugeneAbstractMojo {

    /**
     * Les entrées sorties du plugin.
     * <p/>
     * <p/>
     * <p/>
     * En entrée on demande des répertoires où chercher les fichiers
     * objectmodel a convertir.
     * <p/>
     * En sortie on demande le répertoire ou generer les classes java.
     * <p/>
     * Par défaut on a les valeurs suivantes :
     * </p>
     * <pre>
     * &lt;generateResources&gt;
     * </p>
     *   &lt;input&gt;target/generated-sources/models&lt;/input&gt;
     * </p>
     *   &lt;output&gt;target/generated-sources/java&lt;/output&gt;
     * </p>
     * &lt;/generateResources&gt;
     * </pre>
     * </p>
     * <p/>
     * Note: si {@link #testPhase} est activée, les valeurs par défaut sont :
     * </p>
     * <pre>
     * &lt;generateResources&gt;
     * </p>
     *   &lt;input&gt;target/generated-sources/test-models&lt;/input&gt;
     * </p>
     *   &lt;output&gt;target/generated-sources/test-java&lt;/output&gt;
     * </p>
     * &lt;/generateResources&gt;
     * </pre>
     *
     * @parameter
     * @since 1.0.0-rc-8
     */
    protected PluginIOContext generateResources;
    /**
     * Templates à utiliser, séparés par des virgules.
     *
     * @parameter expression="${eugene.templates}"
     * @required
     * @since 0.50
     */
    protected String templates;
    /**
     * Templates à ne pas utiliser.
     *
     * @parameter expression="${eugene.excludeTemplates}"
     * @since 0.63
     */
    protected String[] excludeTemplates;
    /**
     * Fichier à inclure.
     *
     * @parameter expression="${eugene.includes}" default-value="*.*model"
     * @since 0.50
     */
    protected String includes;
    /**
     * Nom par défaut du paquetage généré.
     *
     * @parameter expression="${eugene.defaultPackage}" default-value="${project.groupId}.${project.artifactId}"
     * @since 0.50
     */
    protected String defaultPackage;
    /**
     * An extra directory to be added to the classpath.
     *
     * @parameter expression="${eugene.extraClassPathDirectory}"
     * @since 0.63
     */
    protected File extraClassPathDirectory;
    /**
     * List of packages to generate (comma separated).
     * <p/>
     * If the parameter is not filled, will generate all packages.
     *
     * @parameter expression="${eugene.generatedPackages}"
     * @since 1.0.0-rc-8
     */
    protected String generatedPackages;

    /**
     * Reader for transform input files in a Model to generate
     *
     * @parameter expression="${eugene.reader}"
     * @since 1.1.0
     */
    protected String reader;


    @Override
    public void doAction() throws Exception {

        getLog().info("Generating java sources from models");
        getLog().info(" reader               : " + reader);
        getLog().info(" includes             : " + includes);
        getLog().info(" using template       : " + templates);
        getLog().info(" using defaultPackage : " + defaultPackage);

        File[] modelFiles = getModelFiles();
        if (modelFiles.length == 0) {
            // can skip
            getLog().warn("no model to treate");
            return;
        }
        if (verbose) {
            for (File m : modelFiles) {
                getLog().info("will treate model file : " + m);
            }
        }
        ModelReader<?> modelReader = getReader();
        generate(modelFiles, modelReader);

        fixCompileSourceRoots();
    }

    protected <M extends Model> void generate(
            File[] modelFiles, ModelReader<M> modelReader) throws Exception {

        if (modelReader == null) {
            // can skip
            getLog().warn("no reader to use");
            return;
        }
        M model = modelReader.read(modelFiles);

        if (StringUtils.isEmpty(generatedPackages)) {
            getLog().info(" generating all packages");
        } else {
            getLog().info(" generating only for packages " + generatedPackages);
        }

        List<Template<M>> templatesList = getTemplates(modelReader);

        for (Template<M> template : templatesList) {
            getLog().info("Apply " + template.getClass().getSimpleName() +
                          " generator");

            template.applyTemplate(model, generateResources.getOutput());
        }
    }

    @Override
    protected PluginIOContext getResources() {
        return generateResources;
    }

    @Override
    protected PluginIOContext initResources() {

        File defaultIn = getFileFromBasedir("target", "generated-sources",
                                            "models");
        File defaultOut = getFileFromBasedir("target", "generated-sources",
                                             "java");

        File defaultTestIn = getFileFromBasedir("target", "generated-sources",
                                                "test-models");
        File defaultTestOut = getFileFromBasedir("target", "generated-sources",
                                                 "test-java");

        generateResources = initResources(
                defaultIn,
                defaultOut,
                defaultTestIn,
                defaultTestOut
        );

        return generateResources;
    }

    /**
     * Recuperation de la liste des fichiers de modele a traite.
     *
     * @return la liste des modeles a utiliser
     */
    protected File[] getModelFiles() {
        List<File> modelFiles = new ArrayList<File>();
        String[] includePatterns = includes.split(",");

        for (File srcDirGen : generateResources.getInputs()) {

            if (verbose) {
                getLog().info("Search for " + Arrays.toString(includePatterns)
                              + " in " + srcDirGen.getAbsolutePath());
            }
            List<File> currentFiles = PluginHelper.getIncludedFiles(
                    srcDirGen,
                    includePatterns,
                    null
            );
            modelFiles.addAll(currentFiles);
        }
        return modelFiles.toArray(new File[modelFiles.size()]);
    }

    protected ModelReader<?> getReader() throws MojoFailureException,
                                                MojoExecutionException {
        ModelReader<?> modelReader;
        try {
            ClassLoader fixedClassLoader = fixClassLoader();
            modelReader = (ModelReader<?>) Class.forName(
                    reader,
                    true,
                    fixedClassLoader).newInstance();
        } catch (InstantiationException eee) {
            throw new MojoFailureException("Can't instantiate reader : " +
                                           reader, eee);
        } catch (IllegalAccessException eee) {
            throw new MojoFailureException("Can't access reader : " + reader,
                                           eee);
        } catch (ClassNotFoundException eee) {
            throw new MojoFailureException("Can't found reader : " + reader,
                                           eee);
        }
        return modelReader;
    }

    @SuppressWarnings("unchecked")
    protected <M extends Model> List<Template<M>> getTemplates(
            ModelReader<M> modelReader)
            throws MojoFailureException, MojoExecutionException {
        // init generators
        Properties templateProperties = new Properties();
        templateProperties.setProperty(Template.PROP_DEFAULT_PACKAGE,
                                       defaultPackage);
        templateProperties.setProperty(Template.PROP_OVERWRITE,
                                       String.valueOf(overwrite));
        templateProperties.setProperty(Template.PROP_ENCODING, encoding);
        templateProperties.setProperty(
                Template.PROP_LAST_MODIFIED_SOURCE,
                String.valueOf(modelReader.getLastModifiedSource()));
        if (generatedPackages != null) {
            templateProperties.setProperty(Template.PROP_GENERATED_PACKAGES,
                                           generatedPackages);
        }
        templateProperties.setProperty(Template.PROP_EXCLUDE_TEMPLATES,
                                       getExcludeTemplatesAsString());

        List<Template<M>> templatesList = new ArrayList<Template<M>>();
        String[] templatesNames = templates.split(",");
        ClassLoader fixedClassLoader = fixClassLoader();
        for (String templateName : templatesNames) {
            // remove trailing spaces
            templateName = templateName.trim();
            try {
                Template<M> template = (Template<M>) Class.forName(templateName,
                        true, fixedClassLoader).newInstance();
                // configuration
                template.setConfiguration(new DefaultTemplateConfiguration(templateProperties));
//                template.setProperties(templateProperties);
                templatesList.add(template);
            } catch (InstantiationException e) {
                throw new MojoFailureException("Can't instantiate generator : "
                                               + templateName, e);
            } catch (IllegalAccessException e) {
                throw new MojoFailureException("Can't access generator : " +
                                               templateName, e);
            } catch (ClassNotFoundException e) {
                throw new MojoFailureException("Can't found generator : " +
                                               templateName, e);
            }
        }
        return templatesList;
    }

    protected String getExcludeTemplatesAsString() {
        String result = "";
        for (int i = 0; i < excludeTemplates.length; i++) {
            result += excludeTemplates[i];
            if (i != excludeTemplates.length - 1) {
                result += ",";
            }
        }
        return result;
    }

    /**
     * permet d'ajout le répertoire de génération des fichiers java dans les
     * répertoires de compilation du projet Maven.
     */
    protected void fixCompileSourceRoots() {

        if (project == null) {
            // no project defined, can not fix anything
            // this case could appears if we wanted to do some tests of the
            // plugin
            return;
        }

        File destDirGen = generateResources.getOutput();

        //TODO-TC20091016 should use AbstractPlugin api
        if (testPhase) {
            if (!project.getTestCompileSourceRoots().contains(
                    destDirGen.getPath())) {
                getLog().info("Add test compile source root : " + destDirGen);
                project.addTestCompileSourceRoot(destDirGen.getPath());
                Resource resources = new Resource();
                resources.setDirectory(destDirGen.getAbsolutePath());
                resources.setExcludes(Arrays.asList("**/*.java"));
                getLog().info("Add test resource root :" + resources);
                project.addTestResource(resources);
            }
        } else {
            if (!project.getCompileSourceRoots().contains(
                    destDirGen.getPath())) {
                getLog().info("Add compile source root : " + destDirGen);
                project.addCompileSourceRoot(destDirGen.getPath());
                Resource resources = new Resource();
                resources.setDirectory(destDirGen.getAbsolutePath());
                resources.setExcludes(Arrays.asList("**/*.java"));
                getLog().info("Add resource root :" + resources);
                project.addResource(resources);
            }
        }
    }

    /**
     * Prepare le classLoader a utiliser dans le generateur.
     * <p/>
     * Si un {@link #extraClassPathDirectory} a été renseigné, il est rajouté.
     * <p/>
     * Si des références à des sibling modules, ils seront rajoutés aussi.
     *
     * @return le class loader modifie
     * @throws MojoExecutionException if any pb
     */
    protected ClassLoader fixClassLoader() throws MojoExecutionException {
        List<URL> urls = new ArrayList<URL>();
        Set<String> urlsAsString = new HashSet<String>();
        try {
            ClassLoader loader = null;
            if (extraClassPathDirectory != null) {
                if (verbose) {
                    getLog().info("Add extra directory in generator's" +
                                  " classLoader : " + extraClassPathDirectory);
                }
                addDirectoryToUrlsList(
                        extraClassPathDirectory,
                        urls,
                        urlsAsString
                );
            }
            if (project.getProjectReferences() != null) {
                // this case is for multi-module when calling from a parent
                //  module
                for (Object o : project.getProjectReferences().entrySet()) {
                    Entry<?, ?> entry = (Entry<?, ?>) o;
                    MavenProject relatedProject =
                            (MavenProject) entry.getValue();
                    if (verbose) {
                        getLog().info("Add project reference in " +
                                      "generator's classLoader : '" +
                                      relatedProject.getArtifact() + "'");
                    }
                    //TODO il faudrait peut-etre aussi ajouter les dependances ?
                    addDirectoryToUrlsList(
                            relatedProject.getArtifact().getFile(),
                            urls,
                            urlsAsString
                    );
                }
            }
            if (!project.getArtifacts().isEmpty()) {
                // this is a special case when artifacts were resolved (for
                // example in site phase)
                if (verbose) {
                    getLog().info("Use resolved artifacts to build class-path");
                }
                for (Object o : project.getArtifacts()) {
                    Artifact a = (Artifact) o;
                    if (!a.getScope().equals("provided")) {
                        addDirectoryToUrlsList(a.getFile(), urls, urlsAsString);
                    }
                }
            }
            // we ask to add the directory in classloader
            loader = getClass().getClassLoader();
            if (!urls.isEmpty()) {
                loader = new URLClassLoader(urls.toArray(new URL[urls.size()]),
                        loader);
            }
            return loader;
        } catch (MalformedURLException e) {
            throw new MojoExecutionException(e.getMessage());
        } finally {
            urls.clear();
            urlsAsString.clear();
        }

    }
}
