/* *##% Plugin maven Generator
 * Copyright (C) 2006 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/
package org.nuiton.eugene.plugin;

import org.nuiton.util.PluginIOContext;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Map.Entry;

import java.util.Set;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.nuiton.eugene.Generator;
import org.nuiton.util.PluginHelper;

/**
 * Effectue toutes les générations et copie les fichiers générés
 * dans le répertoire de compilation
 *
 * @author ruchaud
 * @version $Revision: 620 $
 * 
 * Last update: $Date: 2009-08-24 22:50:41 +0200 (lun., 24 août 2009) $
 * by : $Author: tchemit $
 * 
 * @goal generate
 * @projectRequired true
 */
public class EugenePlugin extends EugeneAbstractMojo {

    /**
     * Les entrées sorties du plugin.
     *
     * <p/>
     *
     * En entrée on demande des répertoires où chercher les fichiers objectmodel a convertir.
     * <p/>
     * En sortie on demande le répertoire ou generer les classes java.
     * <p/>
     * Par défaut on a les valeurs suivantes :
     * </p>
     * <code>
     * &lt;generateResources&gt;
     * </p>
     *   &lt;input&gt;target/generated-sources/models&lt;/input&gt;
     * </p>
     *   &lt;output&gt;target/generated-sources/java&lt;/output&gt;
     * </p>
     * &lt;/generateResources&gt;
     * </code>
     * </p>
     *
     * Note: si {@link #testPhase} est activée, les valeurs par défaut sont :
     * </p>
     * <code>
     * &lt;generateResources&gt;
     * </p>
     *   &lt;input&gt;target/generated-sources/test-models&lt;/input&gt;
     * </p>
     *   &lt;output&gt;target/generated-sources/test-java&lt;/output&gt;
     * </p>
     * &lt;/generateResources&gt;
     * </code>
     *
     * @parameter
     * @since 1.0.0-rc-8
     */
    protected PluginIOContext generateResources;
    /**
     * Templates à utiliser, séparés par des virgules.
     *
     * @parameter expression="${eugene.templates}"
     * @required
     * @since 0.50
     */
    protected String templates;
    /**
     * Templates à ne pas utiliser.
     *
     * @parameter expression="${eugene.excludeTemplates}"
     * @since 0.63
     */
    protected String[] excludeTemplates;
    /**
     * Fichier à inclure.
     *
     * @parameter expression="${eugene.includes}" default-value="*.*model"
     * @since 0.50
     */
    protected String includes;
    /**
     * Nom par défaut du paquetage généré.
     *
     * @parameter expression="${eugene.defaultPackage}" default-value="${project.groupId}.${project.artifactId}"
     * @since 0.50
     */
    protected String defaultPackage;
    /**
     * An extra directory to be added to the classpath.
     *
     * @parameter expression="${eugene.extraClassPathDirectory}"
     * @since 0.63
     */
    protected File extraClassPathDirectory;
    /**
     * List of packages to generate (comma separated).
     *
     * If the parameter is not filled, will generate all packages.
     * 
     * @parameter expression="${eugene.generatedPackages}"
     * @since 1.0.0-rc-8
     */
    protected String generatedPackages;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        getLog().info("Generating java sources from models");
        getLog().info(" includes             : " + includes);
        getLog().info(" using template       : " + templates);
        getLog().info(" using defaultPackage : " + defaultPackage);
        super.execute();

        File[] modelFiles = getModelFiles();
        if (modelFiles.length == 0) {
            // can skip
            getLog().warn("no model to treate");
            return;
        }
        if (verbose) {
            for (File m : modelFiles) {
                getLog().info("will treate model file : " + m);
            }
        }

        List<String> packages = getPackagesToGenerate();

        if (packages == null) {
            getLog().info(" generating all packages");
        } else {
            getLog().info(" generating only for packages " + packages);
        }

        List<Generator> generators = getGenerators(packages);

        for (Generator generator : generators) {
            getLog().info("Apply " + generator.getClass().getSimpleName() + " generator");

            //TC-20090829 fix when loading more than one model together...
            generator.generate(modelFiles, generateResources.getOutput());
//            for (File modelFile : modelFiles) {
//                getLog().debug(" on " + modelFile.getAbsolutePath());
//
//                // generation
//                generator.generate(modelFile, destDirGen);
//            }
        }

        fixCompileSourceRoots();
    }

    @Override
    protected PluginIOContext getResources() {
        return generateResources;
    }

    @Override
    protected PluginIOContext initResources() {

        File defaultIn = getFile("target", "generated-sources", "models");
        File defaultOut = getFile("target", "generated-sources", "java");

        File defaultTestIn = getFile("target", "generated-sources", "test-models");
        File defaultTestOut = getFile("target", "generated-sources", "test-java");

        generateResources = initResources(defaultIn, defaultOut, defaultTestIn, defaultTestOut);

        return generateResources;
    }

    protected List<String> getPackagesToGenerate() {
        List<String> generatedPackagesAsList = null;
        if (generatedPackages != null && !generatedPackages.isEmpty()) {
            generatedPackagesAsList = new ArrayList<String>();
            for (String s : generatedPackages.split(",")) {
                generatedPackagesAsList.add(s);
            }
        }
        return generatedPackagesAsList;
    }

    /**
     * Recuperation de la liste des fichiers de modele a traite.
     * 
     * @return la liste des modeles a utiliser
     */
    protected File[] getModelFiles() {
        List<File> modelFiles = new ArrayList<File>();
        String[] includePatterns = includes.split(",");

        for (File srcDirGen : generateResources.getInputs()) {

            if (verbose) {
                getLog().info("Search for " + Arrays.toString(includePatterns) + " in " + srcDirGen.getAbsolutePath());
            }
            List<File> currentFiles = PluginHelper.getIncludedFiles(srcDirGen, includePatterns, null);
            modelFiles.addAll(currentFiles);
        }
        return modelFiles.toArray(new File[modelFiles.size()]);
    }
    
//    protected File[] getModelFiles() {
//        // get file to generate
//        // TODO improve this loop
//        // TODO too strange code
//        //TODO TC-20090820 use a DirectoryScanner which understand ant-like regex patterns :)
//        List<File> modelFiles = new ArrayList<File>();
//        String[] includePatterns = includes.split(",");
//        for (File srcDirGen : generateResources.getInputs()) {
//            for (String includePattern : includePatterns) {
//                includePattern = includePattern.trim();
//                boolean recursive = false;
//                if (includePattern.startsWith("**/")) {
//                    recursive = true;
//                    includePattern = includePattern.substring(3);
//                }
//                // transform pattern in java regex
//                includePattern = includePattern.replaceAll("\\.", "\\\\.");
//                includePattern = includePattern.replaceAll("\\*", ".*");
//                // log java regex
//                if (verbose) {
//                    getLog().info("Search for " + includePattern + " in " + srcDirGen.getAbsolutePath());
//                }
//                List<File> currentFiles = FileUtil.find(srcDirGen, includePattern, recursive);
//                modelFiles.addAll(currentFiles);
//            }
//        }
//        return modelFiles.toArray(new File[modelFiles.size()]);
//    }

    protected List<Generator> getGenerators(List<String> generatedPackagesAsList) throws MojoFailureException, MojoExecutionException {
        // init generators
        Properties generatorProperties = new Properties();
        generatorProperties.setProperty("defaultPackage", defaultPackage);
        List<Generator> generators = new ArrayList<Generator>();
        String[] templatesNames = templates.split(",");
        ClassLoader fixedClassLoader = fixClassLoader();
        for (String templateName : templatesNames) {
            // remove trailing spaces
            templateName = templateName.trim();
            try {
                Generator generator = (Generator) Class.forName(templateName,
                        true, fixedClassLoader).newInstance();
                // configuration
                generator.setExcludeTemplates(Arrays.asList(excludeTemplates));
                generator.setProperties(generatorProperties);
                generator.setOverwrite(overwrite);
                generator.setEncoding(encoding);
                generator.setGeneratedPackages(generatedPackagesAsList);
                generators.add(generator);
            } catch (InstantiationException e) {
                throw new MojoFailureException("Can't instantiate generator : " + templateName, e);
            } catch (IllegalAccessException e) {
                throw new MojoFailureException("Can't access generator : " + templateName, e);
            } catch (ClassNotFoundException e) {
                throw new MojoFailureException("Can't found generator : " + templateName, e);
            }
        }
        return generators;
    }

    /**
     * permet d'ajout le répertoire de génération des fichiers java dans les répertoires
     * de compilation du projet Maven.
     */
    protected void fixCompileSourceRoots() {

        if (project == null) {
            // no project defined, can not fix anything
            // this case could appears if we wanted to do some tests of the plugin
            return;
        }

        File destDirGen = generateResources.getOutput();

        if (testPhase) {
            if (!project.getTestCompileSourceRoots().contains(
                    destDirGen.getPath())) {
                getLog().info("Add test compile source root : " + destDirGen);
                project.addTestCompileSourceRoot(destDirGen.getPath());
                Resource resources = new Resource();
                resources.setDirectory(destDirGen.getAbsolutePath());
                resources.setExcludes(Arrays.asList("**/*.java"));
                getLog().info("Add test resource root :" + resources);
                project.addTestResource(resources);
            }
        } else {
            if (!project.getCompileSourceRoots().contains(destDirGen.getPath())) {
                getLog().info("Add compile source root : " + destDirGen);
                project.addCompileSourceRoot(destDirGen.getPath());
                Resource resources = new Resource();
                resources.setDirectory(destDirGen.getAbsolutePath());
                resources.setExcludes(Arrays.asList("**/*.java"));
                getLog().info("Add resource root :" + resources);
                project.addResource(resources);
            }
        }
    }

    /**
     * Prepare le classLoader a utiliser dans le generateur.
     * <p/>
     * Si un {@link #extraClassPathDirectory} a été renseigné, il est rajouté.
     * <p/>
     * Si des références à des sibling modules, ils seront rajoutés aussi.
     *
     * @return le class loader modifie
     * @throws MojoExecutionException if any pb
     */
    protected ClassLoader fixClassLoader() throws MojoExecutionException {
        List<URL> urls = new ArrayList<URL>();
        Set<String> urlsAsString = new HashSet<String>();
        try {
            ClassLoader loader = null;
            if (extraClassPathDirectory != null) {
                if (verbose) {
                    getLog().info("Add extra directory in generator's classLoader : " + extraClassPathDirectory);
                }
                urls.add(extraClassPathDirectory.toURI().toURL());
            }
            if (project.getProjectReferences() != null) {
                // this case is for multi-module when calling from a parent module
                for (Object o : project.getProjectReferences().entrySet()) {
                    Entry<?, ?> entry = (Entry<?, ?>) o;
                    MavenProject relatedProject = (MavenProject) entry.getValue();
                    if (verbose) {
                        getLog().info("Add project reference in generator's classLoader : '" + relatedProject.getArtifact() + "'");
                    }
                    //TODO il faudrait peut-etre aussi ajouter les dependances ?
                    addUrl(relatedProject.getArtifact().getFile().toURI().toURL(), urls, urlsAsString);
                }
            }
            if (!project.getArtifacts().isEmpty()) {
                // this is a special case when artifacts were resolved (for example in site phase)
                if (verbose) {
                    getLog().info("Use resolved artifacts to build class-path");
                }
                for (Object o : project.getArtifacts()) {
                    Artifact a = (Artifact) o;
                    if (!a.getScope().equals("provided")) {
                        addUrl(a.getFile().toURI().toURL(), urls, urlsAsString);
                    }
                }
            }
            // we ask to add the directory in classloader
            loader = getClass().getClassLoader();
            if (!urls.isEmpty()) {
                loader = new URLClassLoader(urls.toArray(new URL[urls.size()]),
                        loader);
            }
            return loader;
        } catch (MalformedURLException e) {
            throw new MojoExecutionException(e.getMessage());
        } finally {
            urls.clear();
            urlsAsString.clear();
        }

    }
}
