package org.immutables.value.processor.meta;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;
import com.google.common.base.Verify;
import com.google.common.base.Preconditions;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;

/**
 * InjectManyAnnotationsMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.annotate.InjectManyAnnotations}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.meta.AnnotationInjections.InjectManyAnnotations} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class InjectManyAnnotationsMirror implements AnnotationInjections.InjectManyAnnotations {
  public static final String QUALIFIED_NAME = "org.immutables.annotate.InjectManyAnnotations";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.meta.AnnotationInjections.InjectManyAnnotations";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "InjectManyAnnotations";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code InjectManyAnnotationsMirror}, present if this annotation found
   */
  public static Optional<InjectManyAnnotationsMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code InjectManyAnnotationsMirror}, present if this annotation found
   */
  public static Optional<InjectManyAnnotationsMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new InjectManyAnnotationsMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code InjectManyAnnotationsMirror}s
   */
  public static ImmutableList<InjectManyAnnotationsMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<InjectManyAnnotationsMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new InjectManyAnnotationsMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code InjectManyAnnotationsMirror}
   */
  public static InjectManyAnnotationsMirror from(TypeElement element) {
    return new InjectManyAnnotationsMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code InjectManyAnnotationsMirror}, present if mirror matched this annotation type
   */
  public static Optional<InjectManyAnnotationsMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final AnnotationMirror[] value;
  private final InjectAnnotationMirror[] valueMirror;

  private InjectManyAnnotationsMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    AnnotationMirror[] value = null;
    InjectAnnotationMirror[] valueMirror = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @InjectManyAnnotations");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        valueMirror = valueExtractor$.mirror();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "default attribute 'value'");
    this.valueMirror = Preconditions.checkNotNull(valueMirror, "default attribute 'value'");
  }

  private InjectManyAnnotationsMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    AnnotationMirror[] value = null;
    InjectAnnotationMirror[] valueMirror = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("value".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'value' attribute of @InjectManyAnnotations");
        }
        ValueExtractor valueExtractor$ = new ValueExtractor();
        annotationValue$.accept(valueExtractor$, null);

        value = valueExtractor$.get();
        valueMirror = valueExtractor$.mirror();
        continue;
      }
    }
    this.value = Preconditions.checkNotNull(value, "value for 'value'");
    this.valueMirror = Preconditions.checkNotNull(valueMirror, "value for 'value'");
  }

  /**
   * @return annotation mirror for value of attribute {@code value}
   */
  public AnnotationMirror[] valueMirror() {
    return value.clone();
  }

  /**
   * @return reified annotation mirror for {@link org.immutables.value.processor.meta.AnnotationInjections.InjectAnnotation}
   */
  @Override
  public InjectAnnotationMirror[] value() {
    return valueMirror;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code InjectManyAnnotations.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return AnnotationInjections.InjectManyAnnotations.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "value".hashCode() ^ Arrays.hashCode(value);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof InjectManyAnnotationsMirror) {
      InjectManyAnnotationsMirror otherMirror = (InjectManyAnnotationsMirror) other;
      return Arrays.equals(value, otherMirror.value);
    }
    return false;
  }

  @Override
  public String toString() {
    return "InjectManyAnnotationsMirror:" + annotationMirror;
  }

  private static class ValueExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    AnnotationMirror[] values;
    int position;

    @Override
    public Void visitAnnotation(AnnotationMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new AnnotationMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    AnnotationMirror[] get() {
      return values;
    }

    public InjectAnnotationMirror[] mirror() {
      InjectAnnotationMirror[] mirrors = new InjectAnnotationMirror[values.length];
      for (int i = 0; i < values.length; i++) {
        mirrors[i] = InjectAnnotationMirror.from(values[i]).get();
      }
      return mirrors;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'value' in @" + QUALIFIED_NAME);
    }
  }
}
