/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.util.Date;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.ValidationException;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.chenillekit.tapestry.core.components.DateTimeField;
import org.chorem.pollen.business.ServiceList;
import org.chorem.pollen.business.ServicePoll;
import org.chorem.pollen.business.ServicePollAccount;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.UserDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.utils.MD5;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.data.PollStep;

/**
 * Classe de la page de modification d'un sondage.
 * 
 * @author kmorin
 * @author rannou
 * @version $Id: PollModification.java 2630 2009-07-08 15:23:00Z nrannou $
 */
@IncludeStylesheet("context:css/pollcreation.css")
public class PollModification {

    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /** Étape courante du formulaire */
    @Persist
    private PollStep step;

    /**
     * Objet de session représentant l'utilisateur identifié.
     */
    @SessionState
    private UserDTO user;
    @Property
    private boolean userExists;

    /** Composants DateTimeField pour le début des ajouts de choix */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField beginChoiceDate;

    /** Composants DateTimeField pour le début et la fin du sondage */
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField beginDate;
    @SuppressWarnings("unused")
    @Component(parameters = { "timePicker=true", "timePickerAdjacent=true" })
    private DateTimeField endDate;

    @Property
    @Persist
    private PollDTO poll;

    @Property
    @Persist
    private PollAccountDTO pollAccount;

    /** Indique si l'utilisateur est authorisé */
    @Property
    private boolean userAllowed = false;

    /** Date de début du sondage (utilisée pour la validation). */
    private Date beginDateValidation;

    @InjectComponent
    private Zone pollCreationZone;

    @InjectPage
    private ClosePoll closePoll;

    /** variable utilisée pour l'affichage de la notificationCheckBox */
    @Property
    @Persist
    private boolean notificationCheckBox;

    /** Règle de notification */
    @Property
    @Persist
    private PreventRuleDTO newRule;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des votants
     */
    @SuppressWarnings("unused")
    @Property
    private VotingListDTO votingList;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des votants
     */
    @SuppressWarnings("unused")
    @Property
    private PollAccountDTO votingListPerson;

    /**
     * Objet utilisé dans la boucle de parcours de la liste des choix
     */
    @Property
    private ChoiceDTO choice;

    @InjectPage
    private ModificationValidation modificationValidation;

    @Inject
    private Messages messages;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceList serviceList;

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape suivante de la
     * modification de sondage.
     */
    Object onSuccessFromPollCreationForm() {
        switch (step) {
        case POLL:
            step = PollStep.OPTIONS;
            break;
        case OPTIONS:
            if (poll.isContinuousResults()) {
                poll.setPublicResults(true);
            }

            if (isFreePoll()) {
                step = PollStep.CHOICES;
            } else {
                step = PollStep.LISTS;
            }
            break;
        case LISTS:
            step = PollStep.CHOICES;
            break;
        case CHOICES:

            // Règles de notification
            PreventRuleDTO oldRule = null;
            for (PreventRuleDTO rule : poll.getPreventRuleDTOs()) {
                if (rule.getId().equals(newRule.getId())) {
                    oldRule = rule;
                }
            }
            if (oldRule != null) { // remplacement de oldRule par newRule
                poll.getPreventRuleDTOs().remove(oldRule);
                if (notificationCheckBox) {
                    poll.getPreventRuleDTOs().add(newRule);
                }
            } else { // création d'une nouvelle règle
                poll.getPreventRuleDTOs().add(newRule);
            }

            // mise à jour des listes de votants du sondage
            servicePollAccount.updatePollAccount(pollAccount);            
            for (VotingListDTO votingList : poll.getVotingListDTOs()) {
                for (PollAccountDTO account : votingList.getPollAccountDTOs()) {
                    servicePollAccount.updatePollAccount(account);
                }
                serviceList.updateVotingList(votingList);
            }

            if (poll.isContinuousResults()) {
                poll.setPublicResults(true);
            }

            for (ChoiceDTO choice : poll.getChoiceDTOs()) {
                choice.setValidate(true);
            }
            servicePoll.updatePoll(poll);
            modificationValidation.setPoll(poll);
            return modificationValidation;
        default:
            step = PollStep.POLL;
            break;
        }
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée que le formulaire soit valide ou non. Il est nécessaire
     * de la redéfinir pour qu'en cas d'erreur de validation, la zone soit tout
     * de même mise à jour pour afficher l'erreur.
     */
    Object onSubmitFromPollCreationForm() {
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée lorsqu'on souhaite accéder à l'étape précédente de la
     * création de sondage.
     */
    Object onPrevious() {
        switch (step) {
        case OPTIONS:
            step = PollStep.POLL;
            break;
        case LISTS:
            step = PollStep.OPTIONS;
            break;
        case CHOICES:
            if (isFreePoll()) {
                step = PollStep.OPTIONS;
            } else {
                step = PollStep.LISTS;
            }
            break;
        default:
            step = PollStep.POLL;
            break;
        }
        return pollCreationZone.getBody();
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * endDate.
     * 
     * @throws ValidationException
     */
    void onValidateFromEndDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }

        if (value != null && value.before(beginDateValidation)) {
            throw new ValidationException(messages.get("endDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation du champs
     * beginChoiceDate.
     * 
     * @throws ValidationException
     */
    void onValidateFromBeginChoiceDate(Date value) throws ValidationException {
        if (beginDateValidation == null) {
            beginDateValidation = new Date();
        }

        if (value != null && value.after(beginDateValidation)) {
            throw new ValidationException(messages
                    .get("beginChoiceDate-validate"));
        }
    }

    /**
     * Méthode appelée lors de la validation du formulaire. Validation des
     * listes de votants.
     * 
     * @throws ValidationException
     */
    void onValidateFormFromPollCreationForm() throws ValidationException {
        int nbEqual = 0;
        int nbNotNull = 0;

        // Validation des votants
        if (step == PollStep.LISTS) {

            // Repérage des doublons
            for (VotingListDTO votingList : poll.getVotingListDTOs()) {
                for (PollAccountDTO account1 : votingList.getPollAccountDTOs()) {
                    if (account1.getVotingId() != null) {
                        nbNotNull++;
                        for (PollAccountDTO account2 : votingList
                                .getPollAccountDTOs()) {
                            if (account1.getVotingId().equals(
                                    account2.getVotingId())) {
                                nbEqual++;
                            }
                        }
                    }
                }
            }

            if (nbEqual > nbNotNull) {
                throw new ValidationException(messages.get("list-validate"));
            }
        }
    }

    Object onActionFromCloseButton() {
        closePoll.setPoll(poll);
        return closePoll;
    }

    public String getChoiceDateDisplay() {
        return poll.isChoiceAddAllowed() ? "display: block;" : "display: none;";
    }

    public String getNotificationDisplay() {
        return notificationCheckBox ? "display: block;" : "display: none;";
    }

    /** Retourne la date correspondant au choix courant */
    public Date getChoiceNameAsDate() {
        return new Date(Long.valueOf(choice.getName()));
    }

    /** Retourne la chaîne correspondant à l'étape courante */
    public String getStepLegend() {
        Integer index = step.getIndex();

        // corrections selon le type de sondage
        if (poll.getPollType() == PollType.FREE) {
            if (step == PollStep.CHOICES) {
                index--;
            }
        }

        // mise en forme du message
        switch (step) {
        case POLL:
            return messages.format("pollLegend", index);
        case OPTIONS:
            return messages.format("optionsLegend", index);
        case LISTS:
            return messages.format("listsLegend", index);
        case CHOICES:
            return messages.format("choicesLegend", index);
        default:
            return "";
        }
    }

    public boolean isInPoll() {
        return step == PollStep.POLL;
    }

    public boolean isInLists() {
        return step == PollStep.LISTS;
    }

    public boolean isInChoices() {
        return step == PollStep.CHOICES;
    }

    public boolean isInOptions() {
        return step == PollStep.OPTIONS;
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }

    public boolean isTextChoices() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateChoices() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImgChoices() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    Object onActionFromPrevOptions() {
        step = PollStep.POLL;
        return pollCreationZone.getBody();
    }

    Object onActionFromPrevLists() {
        step = PollStep.OPTIONS;
        return pollCreationZone.getBody();
    }

    Object onActionFromPrevChoices() {
        if (isFreePoll()) {
            step = PollStep.OPTIONS;
        } else {
            step = PollStep.LISTS;
        }
        return pollCreationZone.getBody();
    }

    void onActivate(String id) {
        if (id != null && !"".equals(id) && id.split(":", 2).length == 2) {
            String pollId = id.split(":", 2)[0];
            String creatorId = id.split(":", 2)[1];
            poll = servicePoll.findPollByPollId(pollId);

            if (poll != null) {
                if (creatorId.equals(MD5.encode(poll.getCreatorId()))) {
                    pollAccount = servicePollAccount.findPollAccountById(poll
                            .getCreatorId());
                    userAllowed = true;
                }

                // Règles de notification
                notificationCheckBox = false;
                newRule = new PreventRuleDTO("vote", 0, true,
                        PreventRuleManager.EMAIL_ACTION);
                newRule.setPollId(poll.getId());
                for (PreventRuleDTO rule : poll.getPreventRuleDTOs()) {
                    if ("vote".equals(rule.getScope())) {
                        notificationCheckBox = true;
                        newRule = rule;
                    }
                }
            }
        }

        // Affichage des erreurs
        if (poll == null) {
            feedback.addError(messages.get("pollNotFound"));
        } else if (poll.isClosed()) {
            feedback.addError(messages.get("pollClosed"));
        } else if (!userAllowed) {
            feedback.addError(messages.get("userNotAllowed"));
        }
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
        step = PollStep.POLL;
    }
}
