/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: VoteService.java 3328 2012-04-27 10:24:54Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-services/src/main/java/org/chorem/pollen/services/impl/VoteService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListDAO;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.business.persistence.VoteDAO;
import org.chorem.pollen.business.persistence.VoteToChoice;
import org.chorem.pollen.business.persistence.VoteToChoiceDAO;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.entities.PollenBinderHelper;
import org.chorem.pollen.entities.PollenDAOHelper;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.PollAccountNotFound;
import org.chorem.pollen.services.exceptions.VoteNotFoundException;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaFilterPagerUtil;

import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;

public class VoteService extends PollenServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(VoteService.class);
    
    public Vote getNewVote(Poll poll, PollAccount account) {
        
        Preconditions.checkNotNull(poll);
        
        VoteDAO voteDAO = getDAO(Vote.class);
        VoteToChoiceDAO voteToChoiceDAO = getDAO(VoteToChoice.class);
        
        Vote result = newInstance(voteDAO);
        result.setPollAccount(account);
        result.setWeight(1.);
        
        // Prepare the List of VoteToChoice with Poll's choices
        for (Choice choice : poll.getChoice()) {
            VoteToChoice element = newInstance(voteToChoiceDAO);
            element.setChoice(choice);
            result.addChoiceVoteToChoice(element);
        }
        
        return result;        
    }
    
    public Vote getVoteEditable(Poll poll, PollAccount accountEditable) {
        
        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(accountEditable);

        VoteDAO dao = getDAO(Vote.class);
        Vote voteLoaded;
        try {
            voteLoaded = dao.findByAccountId(accountEditable.getAccountId());
        } catch (TopiaException e) {
            throw new PollenTechnicalException(e);
        }
        
        Vote result;
        if (voteLoaded != null) {

            result = newInstance(dao);
            PollenBinderHelper.simpleCopy(voteLoaded, result, true);

            // Attach the given accountEditable instead of the loaded one
            result.setPollAccount(accountEditable);

            // For an existing Vote, ensure that there is as many voteToChoice as poll choices
            VoteToChoiceDAO voteToChoiceDAO = getDAO(VoteToChoice.class);
            List<VoteToChoice> voteToChoices = Lists.newArrayList();

            // Add all VoteToChoice even they have empty value to match the size of Poll Choice List
            List<Choice> choices = poll.getChoice();
            for (int i = 0; i < choices.size(); i++) {

                Choice choice = choices.get(i);

                VoteToChoice voteToChoice = newInstance(voteToChoiceDAO);
                voteToChoice.setChoice(choice);

                // Retrieve existing value from loaded vote
                VoteToChoice voteToChoiceLoaded = voteLoaded.getChoiceVoteToChoice(choice);
                if (voteToChoiceLoaded != null) {
                    voteToChoice.setVoteValue(voteToChoiceLoaded.getVoteValue());
                }

                voteToChoices.add(i, voteToChoice);
            }
            result.setChoiceVoteToChoice(voteToChoices);
            
        } else {             
            result = getNewVote(poll, accountEditable);
        }

        // Retrieve weight for Restricted Poll with existing account
        if (poll.getPollType() != PollType.FREE && accountEditable.getTopiaId() != null) {
            PersonToListDAO personToListDAO = getDAO(PersonToList.class);
            PersonToList personToList = personToListDAO.findByPollAndAccount(poll, accountEditable);
            result.setWeight(personToList.getWeight());
        }
        return result;
    }

    /**
     * Retrieve the URL to update a vote based on {@link PollUri}.
     * 
     * @param pollUri Uri with pollId and accountId
     * @return the URL that allow the voting person to update his vote
     */
    public String getUpdateVoteUrl(PollUri pollUri) {
        Preconditions.checkNotNull(pollUri.getAccountId());
        URL baseUrl = serviceContext.getApplicationURL();
        String result =
                String.format("%s/poll/votefor/%s", baseUrl, pollUri.getUri());
        return result;
    }
    
    public Vote createVote(Vote vote) throws PollAccountNotFound {

        VoteDAO voteDAO = getDAO(Vote.class);

        Vote result = create(voteDAO);
        result.setWeight(vote.getWeight());
        result.setAnonymous(vote.isAnonymous());

        // -- PollAccount -- //
        PollAccount pollAccount = vote.getPollAccount();
        String pollAccountId = pollAccount.getTopiaId();
        PollAccount pollAccountLoaded;

        // Load existing account (restricted poll)
        if (pollAccountId != null) {
            pollAccountLoaded = getEntityById(PollAccount.class, pollAccountId);
            
            // Create new account
        } else {
            PollAccountDAO pollAccountDAO = getDAO(PollAccount.class);
            pollAccountLoaded = create(pollAccountDAO);
            pollAccountLoaded.setAccountId(pollAccount.getAccountId());
        }

        // Update user data if not anonymous
        if (!vote.isAnonymous()) {

            pollAccountLoaded.setVotingId(pollAccount.getVotingId());
            pollAccountLoaded.setEmail(pollAccount.getEmail());

            UserAccount userAccount = pollAccount.getUserAccount();
            if (userAccount != null) {
                UserAccount userAccountLoaded = getEntityById(UserAccount.class, userAccount.getTopiaId());
                pollAccountLoaded.setUserAccount(userAccountLoaded);
            }
        }
        // TODO Manage anonymous for existing account ??? problem with restricted and email
        result.setPollAccount(pollAccountLoaded);

        // -- List of VoteToChoice -- //
        for (VoteToChoice input : vote.getChoiceVoteToChoice()) {

            Integer value = input.getVoteValue();
            if (value != null) {
                createVoteToChoice(result, input);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Entity created: " + result.getTopiaId());
        }

        commitTransaction("Could not create vote");

        return result;
    }

    public Vote updateVote(Vote vote) throws VoteNotFoundException {

        Vote result = getEntityById(Vote.class, vote.getTopiaId());

        if (result == null) {
            throw new VoteNotFoundException();
        }

        // -- PollAccount -- //
        PollAccount voteAccount = vote.getPollAccount();
        PollAccount pollAccountEntity = result.getPollAccount();
        if (vote.isAnonymous()) {
            pollAccountEntity.setVotingId(null);
            pollAccountEntity.setEmail(null);
            pollAccountEntity.setUserAccount(null);

        } else {
            pollAccountEntity.setVotingId(voteAccount.getVotingId());
            pollAccountEntity.setEmail(voteAccount.getEmail());
            pollAccountEntity.setUserAccount(voteAccount.getUserAccount());
        }
        result.setAnonymous(vote.isAnonymous());

        // -- List of VoteToChoice -- //
        for (VoteToChoice input : vote.getChoiceVoteToChoice()) {

            Integer value = input.getVoteValue();

            VoteToChoice voteToChoiceEntity = result.getChoiceVoteToChoice(input.getChoice());
            if (value != null) {

                if (voteToChoiceEntity != null) {
                    voteToChoiceEntity.setVoteValue(value);
                    
                } else {
                    createVoteToChoice(result, input);
                }

            } else if (voteToChoiceEntity != null) {
                deleteVoteToChoice(result, voteToChoiceEntity);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Entity updated: " + result.getTopiaId());
        }

        commitTransaction("Could not update vote");

        return result;
    }
    
    protected VoteToChoice createVoteToChoice(Vote vote, VoteToChoice source) {
        
        VoteToChoiceDAO voteToChoiceDao = getDAO(VoteToChoice.class);
        VoteToChoice result = create(voteToChoiceDao);
        
        String choiceId = source.getChoice().getTopiaId();
        Choice choiceLoaded = getEntityById(Choice.class, choiceId);
        result.setChoice(choiceLoaded);
        
        result.setVote(vote);           
        result.setVoteValue(source.getVoteValue());
        
        vote.addChoiceVoteToChoice(result);
        
        return result;
    }
    
    protected void deleteVoteToChoice(Vote vote, VoteToChoice entity) {

        VoteToChoiceDAO voteToChoiceDao = getDAO(VoteToChoice.class);
        vote.removeChoiceVoteToChoice(entity);
        delete(voteToChoiceDao, entity);
    }

    /**
     * Delete vote referenced by {@code voteId}. This will also delete all
     * associated {@link VoteToChoice}. The {@link PollAccount} could also be
     * deleted for a {@link PollType#FREE} poll where vote will be removed. This
     * remove implied an new execution of poll vote counting if continuous results
     * is activated.
     * 
     * @param voteId TopiaId of the vote
     * @throws VoteNotFoundException if vote doesn't exist
     */
    public void deleteVote(String voteId) throws VoteNotFoundException {

        Vote entityToDelete = getEntityById(Vote.class, voteId);

        if (entityToDelete == null) {
            throw new VoteNotFoundException();
        }

        VoteDAO dao = getDAO(Vote.class);
        VoteToChoiceDAO voteToChoiceDao = getDAO(VoteToChoice.class);        

        // Delete all VoteToChoice
        List<VoteToChoice> voteToChoices = ImmutableList.copyOf(entityToDelete.getChoiceVoteToChoice());
        for (VoteToChoice voteToChoice : voteToChoices) {
            entityToDelete.removeChoiceVoteToChoice(voteToChoice);
            delete(voteToChoiceDao, voteToChoice);
        }

        Poll poll = entityToDelete.getPoll();

        if (PollType.FREE == poll.getPollType()) {

            // Delete vote PollAccount if the Poll is free and account is not the creator
            PollAccount voteAccount = entityToDelete.getPollAccount();
            if (!voteAccount.equals(poll.getCreator())) {
                PollAccountDAO accountDAO = getDAO(PollAccount.class);
                delete(accountDAO, voteAccount);
            }

            // Update pollAccount hasVoted flag
        } else {
            PersonToList personToList = poll.getPersonToListByVote(entityToDelete);
            personToList.setHasVoted(false);
        }

        poll.removeVote(entityToDelete);
        delete(dao, entityToDelete);

        // TODO calculate results if continuous

        if (log.isDebugEnabled()) {
            log.debug("Entity deleted: " + voteId);
        }

        commitTransaction("Could not delete vote");
    }

    public List<Vote> getVotesByPoll(Poll poll,
                                     TopiaFilterPagerUtil.FilterPagerBean pager) {
        try {
            VoteDAO dao = getDAO(Vote.class);
            List<Vote> results = dao.getVotesByPoll(poll, pager);
            return results;

        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not obtain votes", e);
        }
    }

    /**
     * Vote is allowed if {@code poll} is running and {@code pollAccount} is 
     * defined in the {@code poll} restricted list if it's not a {@link PollType#FREE} 
     * poll. The account must be defined previously using
     * {@link PollService#getPollAccountEditable(String, UserAccount, Poll)} to
     * have a proper link between userAccount and pollAccount even if not already
     * created in dabase. The poll creator can't vote. The token is just use
     * for moderate purpose.
     * 
     * @param poll Poll
     * @param accountEditable Account to check
     * @return true if vote is allowed, false otherwise
     */
    public boolean isVoteAllowed(Poll poll, PollAccount accountEditable) {
        
        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(accountEditable);
                
        Date now = serviceContext.getCurrentTime();

        boolean result;
        if (poll.getCreator().equals(accountEditable)) {

            // The creator user can't vote
            result = false;
        } else {

            // The poll must be running and account allowed for restricted poll
            result = poll.isRunning(now);
            if (poll.getPollType() != PollType.FREE) {

                PersonToListDAO personToListDAO = getDAO(PersonToList.class);
                PersonToList personToList = personToListDAO.findByPollAndAccount(poll, accountEditable);

                result &= personToList != null;
            }
        }
        return result;
    }
    
    public boolean isUpdateAllowed(Poll poll, String voteId, String accountId, UserAccount userConnected) {
        
        Date now = serviceContext.getCurrentTime();
        
        boolean result = false;

        Vote vote = poll.getVoteByTopiaId(voteId);

        // can only modify a vote if poll is running.
        if (vote != null && poll.isRunning(now)) {
            PollAccount votePollAccount = vote.getPollAccount();

            // si le votant du vote correspond au votant actuel (pollAccountId)
            if (accountId != null
                && accountId.equals(votePollAccount.getAccountId())) {
                result = true;
            }

            // si l'utilisateur du vote correspond à l'utilisateur actuel (user)
            if (userConnected != null) {
                UserAccount voteUserAccount = votePollAccount.getUserAccount();
                result = userConnected.equals(voteUserAccount);
            }
        }
        return result;
    }

    public boolean hasAlreadyVoted(String votingId, Poll poll) {
        try {
            VoteDAO dao = getDAO(Vote.class);
            boolean result = dao.hasAlreadyVoted(votingId, poll);
            return result;
        } catch (Exception e) {
            throw new PollenTechnicalException(
                    "Unable test vote existing for account with votingId = " +
                    votingId + " and poll with uid = " + poll.getPollId(), e);
        }
    }

    public List<Vote> selectVotes(Map<String, Object> properties) {
        TopiaContext transaction = getTransaction();
        try {
            VoteDAO voteDAO = PollenDAOHelper.getVoteDAO(transaction);

            List<Vote> voteEntities;

            if (properties == null) {
                voteEntities = voteDAO.findAll();
                if (log.isWarnEnabled()) {
                    log.warn("Attention : tous les votes ont été sélectionnés !");
                }
            } else {
                voteEntities = voteDAO.findAllByProperties(properties);
            }

            List<Vote> results = Lists.newArrayList(voteEntities);

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                          + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (Exception e) {
            throw new PollenTechnicalException("Could not obtain votes", e);
        }
    }

} //voteservice
