/*
 * #%L
 * Pollen :: Services
 * 
 * $Id: PollenNotifierWorker.java 3332 2012-04-30 15:52:29Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3/pollen-services/src/main/java/org/chorem/pollen/services/PollenNotifierWorker.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services;

import com.google.common.base.Preconditions;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.services.impl.PreventRuleService;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

/**
 * This service run as a daemon seek to send some notification from running poll
 * store in {@link PreventRule}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenNotifierWorker implements Runnable, Closeable {

    /** Service context used to create other services. */
    protected final PollenServiceContext serviceContext;

    /** Timer gérant les threads. */
    private final Timer timer;

    /** Topia root context to pollen db. */
    private final TopiaContext rootContext;

    public PollenNotifierWorker(PollenServiceContext serviceContext,
                                TopiaContext rootContext) {
        Preconditions.checkNotNull(serviceContext);
        Preconditions.checkNotNull(rootContext);
        Preconditions.checkState(!rootContext.isClosed());
        this.rootContext = rootContext;
        this.serviceContext = serviceContext;
        timer = new Timer();
    }

    @Override
    public void run() {

        // starts it
        timer.schedule(new TimerTask() {

            @Override
            public void run() {
                try {
                    TopiaContext transaction = rootContext.beginTransaction();
                    try {
                        serviceContext.setTransaction(transaction);

                        PollService pollService =
                                serviceContext.newService(PollService.class);
                        PreventRuleService preventRuleService =
                                serviceContext.newService(PreventRuleService.class);

                        // Récupération des sondages en cours
                        List<Poll> polls = pollService.getRunningPolls(true);

                        for (Poll poll : polls) {

                            PollUrl pollVoteUrl = pollService.getPollVoteUrl(poll, false);

                            preventRuleService.onPollToRemind(poll, pollVoteUrl);

                        }

                    } finally {

                        serviceContext.setTransaction(null);

                        transaction.closeContext();
                    }
                } catch (TopiaException e) {
                    throw new PollenTechnicalException("Topia problem", e);
                }
            }
        }, 600000, 600000);
    }

    @Override
    public void close() throws IOException {
        timer.cancel();
    }

}
