/*
 * This file is part of the OWL API.
 *
 * The contents of this file are subject to the LGPL License, Version 3.0.
 *
 * Copyright (C) 2011, The University of Manchester
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 *
 * Alternatively, the contents of this file may be used under the terms of the Apache License, Version 2.0
 * in which case, the provisions of the Apache License Version 2.0 are applicable instead of those above.
 *
 * Copyright 2011, University of Manchester
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.coode.owlapi.rdf.model;

import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLAnonymousIndividual;
import org.semanticweb.owlapi.model.OWLLiteral;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyManager;

/**
 * @author Matthew Horridge, The University Of Manchester, Bio-Health Informatics
 *         Group, Date: 06-Dec-2006
 */
public class RDFTranslator
        extends
        AbstractTranslator<RDFNode, RDFResourceNode, RDFResourceNode, RDFLiteralNode> {

    private RDFGraph graph;

    /**
     * @param manager
     *        the manager
     * @param ontology
     *        the ontology
     * @param useStrongTyping
     *        true if strong typing is required
     */
    public RDFTranslator(OWLOntologyManager manager, OWLOntology ontology,
            boolean useStrongTyping) {
        super(manager, ontology, useStrongTyping);
        graph = new RDFGraph();
    }

    /** @return the graph */
    public RDFGraph getGraph() {
        return graph;
    }

    @Override
    protected void addTriple(RDFResourceNode subject, RDFResourceNode pred,
            RDFNode object) {
        graph.addTriple(new RDFTriple(subject, pred, object));
    }

    @Override
    protected RDFResourceNode getAnonymousNode(Object key) {
        if (key instanceof OWLAnonymousIndividual) {
            String id = ((OWLAnonymousIndividual) key).getID().getID();
            RDFResourceNode toReturn = new RDFResourceNode(
                    System.identityHashCode(id));
            return toReturn;
        }
        RDFResourceNode toReturn = new RDFResourceNode(
                System.identityHashCode(key));
        return toReturn;
    }

    @Override
    protected RDFLiteralNode getLiteralNode(OWLLiteral literal) {
        return translateLiteralNode(literal);
    }

    /**
     * @param literal
     *        literal to translate
     * @return translated literal
     */
    public static RDFLiteralNode translateLiteralNode(OWLLiteral literal) {
        if (!literal.isRDFPlainLiteral()) {
            return new RDFLiteralNode(literal.getLiteral(), literal
                    .getDatatype().getIRI());
        } else {
            return new RDFLiteralNode(literal.getLiteral(),
                    literal.hasLang() ? literal.getLang() : null);
        }
    }

    @Override
    protected RDFResourceNode getPredicateNode(IRI uri) {
        return new RDFResourceNode(uri);
    }

    @Override
    protected RDFResourceNode getResourceNode(IRI uri) {
        return new RDFResourceNode(uri);
    }

    /** clear the graph */
    public void reset() {
        graph = new RDFGraph();
    }
}
