/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.admin.content;

import static org.nuiton.i18n.I18n._;

import java.awt.Color;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.io.File;
import java.util.List;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JColorChooser;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTreeTable;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.TreeNode;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.VradiHelper;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.RootThesaurus;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.ThesaurusImpl;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.ui.admin.AdminHandler;
import com.jurismarches.vradi.ui.admin.helpers.ThesaurusTreeTableHelper;
import com.jurismarches.vradi.ui.admin.popups.ThesaurusConfirmDialogUI;
import com.jurismarches.vradi.ui.admin.popups.ThesaurusEditUI;
import com.jurismarches.vradi.ui.admin.popups.ThesaurusPathChooserUI;
import com.jurismarches.vradi.ui.admin.popups.ThesaurusQueryChangesHandler;
import com.jurismarches.vradi.ui.admin.popups.ThesaurusQueryChangesUI;
import com.jurismarches.vradi.ui.admin.query.ThesaurusChangesTreeTableModel;
import com.jurismarches.vradi.ui.helpers.ThesaurusDataHelper;
import com.jurismarches.vradi.ui.helpers.UIHelper;
import com.jurismarches.vradi.ui.offer.thesaurus.ThesaurusCartographyUI;
import com.jurismarches.vradi.ui.renderers.ColorsMap;
import com.jurismarches.vradi.ui.tree.VradiTreeTableNode;
import com.sun.star.uno.RuntimeException;

/**
 * @author sletellier <letellier@codelutin.com>
 **/
public class AdminThesaurusHandler {   
    /**
     * to use log facility, just put in your code: log.info(\"...\");
     */
    static private Log log = LogFactory.getLog(AdminThesaurusHandler.class);

    public WikittyProxy getProxy() {
        return VradiService.getWikittyProxy();
    }

    protected AdminThesaurusUI getUI(JAXXContext context) {
        return (AdminThesaurusUI)context;
    }
    
    public Thesaurus getThesaurus(VradiTreeTableNode parentNode) {
        Class<?> type = parentNode.getInternalClass();
        if (!type.equals(Thesaurus.class)) {
            return null;
        }
        String id = parentNode.getId();
        return getThesaurus(id);
    }

    public Thesaurus getThesaurus(String id) {
        return ThesaurusDataHelper.restoreThesaurus(id);
    }

    public RootThesaurus getRootThesaurus(VradiTreeTableNode parentNode) {
        Class<?> type = parentNode.getInternalClass();
        if (!type.equals(RootThesaurus.class)) {
            return null;
        }
        String id = parentNode.getId();
        return getRootThesaurus(id);
    }

    public RootThesaurus getRootThesaurus(String id) {
        return ThesaurusDataHelper.restoreRootThesaurus(id);
    }

    /**
     * Return helpers selected node
     *
     * @param helper thesaurus tree table helper
     * @return selected node
     */
    protected VradiTreeTableNode getSelectedNode(ThesaurusTreeTableHelper helper) {

        // Get selected thesaurus
        VradiTreeTableNode selectedNode = helper.getSelectedNode();
        if (selectedNode == null) {
            selectedNode = helper.getRootNode();
        }
        return selectedNode;
    }

    /**
     * Creation d'un thesaurus root.
     * 
     * Le root thesaurus sert de requetage solr, il ne doit contenir que
     * des caraceres alphanumerique.
     * 
     * @param context parent ui
     */
    public void createRootThesaurus(JAXXContext context) {
        String rootThesaurusName = JOptionPane.showInputDialog(getUI(context),
                        _("vradi.adminThesaurus.askRootThesaurusName"),
                        _("vradi.adminThesaurus.askRootThesaurusTitle"),
                        JOptionPane.QUESTION_MESSAGE);

        try {
            if (rootThesaurusName != null) {
    
                // test des caracteres alphanumerique
                if (!rootThesaurusName.matches("\\w+")) {
                    JOptionPane.showMessageDialog(getUI(context),
                            _("vradi.adminThesaurus.rootThesaurusInvalidName"),
                            _("Error"), JOptionPane.ERROR_MESSAGE);
                }
                
                else {
                    int n = JOptionPane.showConfirmDialog(getUI(context),
                                _("vradi.adminThesaurus.askCreateRootThesaurus", rootThesaurusName),
                                _("vradi.admin.saveTitle"),
                                JOptionPane.YES_OPTION);
        
                    if (n == JOptionPane.YES_OPTION) {
        
                        // Check existing
                        if (VradiService.getVradiDataService().isRootThesaurusNameExists(rootThesaurusName)) {
        
                            // Ask retry
                            int n2 = JOptionPane.showConfirmDialog(getUI(context),
                                    _("vradi.adminThesaurus.warnNameIsExisting", rootThesaurusName),
                                    _("vradi.admin.warnTitle"),
                                    JOptionPane.OK_CANCEL_OPTION);
        
                            if (n2 == JOptionPane.OK_OPTION) {
                                createRootThesaurus(context);
                            }
                            return;
                        }
        
                        // Store
                        RootThesaurus newRootThesaurus = VradiService.getVradiDataService().createRootThesaurus(rootThesaurusName);
        
                        // Select node created
                        ThesaurusTreeTableHelper helper = context.getContextValue(ThesaurusTreeTableHelper.class);
                        UIHelper.selectNodeLater(helper, newRootThesaurus.getWikittyId());
                    }
                }
            }
        } catch (VradiException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't create root thesaurus", ex);
            }
            throw new RuntimeException("Can't create root thesaurus", ex);
        }
    }

    public void createThesaurus(JAXXContext context, RootThesaurus parentRootThesaurus) {

        // Create child
        Thesaurus thesaurusToCreate = new ThesaurusImpl();

        // Register parent and root thesaurus
        String rootId = parentRootThesaurus.getWikittyId();
        thesaurusToCreate.setParent(rootId);
        thesaurusToCreate.setRootThesaurus(rootId);

        // Open edit ui
        ThesaurusEditUI edit = new ThesaurusEditUI(context);

        // Init edit ui
        edit.init(thesaurusToCreate, _("vradi.adminThesaurus.creationTitle"), true);

        // If is to save
        if (edit.isSave()) {

            // If name is not empty
            String name = thesaurusToCreate.getName();
            if (name != null && !name.isEmpty()) {

                // Check existing
                RootThesaurus rootThesaurus = ThesaurusDataHelper.restoreRootThesaurus(rootId);
                if (VradiService.getVradiDataService().isThesaurusNameExistsInRootThesaurus(rootThesaurus, name)) {

                    // Ask retry
                    int n = JOptionPane.showConfirmDialog(getUI(context),
                            _("vradi.adminThesaurus.warnNameIsExisting", name),
                            _("vradi.adminThesaurus.warnTitle"),
                            JOptionPane.OK_CANCEL_OPTION);

                    if (n == JOptionPane.OK_OPTION) {
                        // reouvre la fenetre de creation
                        createThesaurus(context, parentRootThesaurus);
                    }
                    return;
                }

                // Store
                getProxy().store(thesaurusToCreate);

                if (log.isDebugEnabled()) {
                    log.debug("Adding thesaurus " + thesaurusToCreate.getName()
                            + " to parent " + parentRootThesaurus.getName());
                }

                // Select node created
                ThesaurusTreeTableHelper helper = context.getContextValue(ThesaurusTreeTableHelper.class);
                UIHelper.selectNodeLater(helper, thesaurusToCreate.getWikittyId());
            }
        }
    }

    public void createThesaurus(JAXXContext context, Thesaurus parentThesaurus) {

        // Create child
        Thesaurus thesaurusToCreate = new ThesaurusImpl();

        if (parentThesaurus == null) {
            return;
        }

        // Register parent and root thesaurus
        thesaurusToCreate.setParent(parentThesaurus.getWikittyId());
        thesaurusToCreate.setRootThesaurus(parentThesaurus.getRootThesaurus());

        // Open edit ui
        ThesaurusEditUI edit = new ThesaurusEditUI(context);

        // Init edit ui
        edit.init(thesaurusToCreate, _("vradi.adminThesaurus.creationTitle"), true);

        // If is to save
        if (edit.isSave()) {

            // If name is not empty
            String name = thesaurusToCreate.getName();
            if (name != null && !name.isEmpty()) {
                // Get root
                RootThesaurus rootThesaurus = ThesaurusDataHelper
                        .restoreRootThesaurus(thesaurusToCreate.getRootThesaurus());

                // Check existing
                if (VradiService.getVradiDataService().isThesaurusNameExistsInRootThesaurus(rootThesaurus, name)) {

                    // Ask retry
                    int n = JOptionPane.showConfirmDialog(getUI(context),
                            _("vradi.adminThesaurus.warnNameIsExisting", name),
                            _("vradi.admin.warnTitle"),
                            JOptionPane.OK_CANCEL_OPTION);

                    if (n == JOptionPane.OK_OPTION) {
                        createThesaurus(context, parentThesaurus);
                    }
                    return;
                }

                // Store
                getProxy().store(thesaurusToCreate);

                if (log.isDebugEnabled()) {
                    log.debug("Adding thesaurus " + thesaurusToCreate.getName()
                            + " to parent " + parentThesaurus.getName());
                }
                
                // Select node created
                ThesaurusTreeTableHelper helper = context.getContextValue(ThesaurusTreeTableHelper.class);
                UIHelper.selectNodeLater(helper, thesaurusToCreate.getWikittyId());
            }
        }
    }

    public void editRootThesaurus(JAXXContext context, RootThesaurus rootThesaurus) {

        // Keep oldName
        String oldName = rootThesaurus.getName();

        // Ask name
        String name = JOptionPane
                .showInputDialog(getUI(context),
                        _("vradi.adminThesaurus.askRootThesaurusEditName",
                            oldName));

        if (name != null && !name.isEmpty()) {

            // Check existing
            if (VradiService.getVradiDataService().isRootThesaurusNameExists(rootThesaurus.getName(),
                    rootThesaurus.getWikittyId())) {

                // Ask retry
                int n = JOptionPane.showConfirmDialog(getUI(context),
                        _("vradi.adminThesaurus.warnNameIsExisting", name),
                        _("vradi.admin.warnTitle"),
                        JOptionPane.OK_CANCEL_OPTION);

                if (n == JOptionPane.OK_OPTION) {
                    editRootThesaurus(context, rootThesaurus);
                }
                return;
            }

            // Confirm and view apply
            int n = JOptionPane.showConfirmDialog(getUI(context),
                        _("vradi.adminThesaurus.askEditRootThesaurus",
                                oldName, name),
                        _("vradi.admin.saveTitle"),
                        JOptionPane.OK_CANCEL_OPTION);

            if (n == JOptionPane.OK_OPTION) {

                // Impact group request
                modifyQueries(context, oldName, name, null, null);

                // Set name
                rootThesaurus.setName(name);

                // Store
                getProxy().store(rootThesaurus);

                // Select node created
                ThesaurusTreeTableHelper helper = context.getContextValue(ThesaurusTreeTableHelper.class);
                UIHelper.selectNodeLater(helper, rootThesaurus.getWikittyId());
            }
        }
    }

    public void editThesaurus(JAXXContext context, Thesaurus thesaurus) {

        // To prevent wikity obselete
        thesaurus = ThesaurusDataHelper.restoreThesaurus(thesaurus.getWikittyId());

        // Open edit ui
        ThesaurusEditUI edit = new ThesaurusEditUI(context);

        // Get old name
        String oldName = thesaurus.getName();

        // Open edit UI
        edit.init(thesaurus, _("vradi.adminThesaurus.editTitle", oldName), false);

        if (edit.isSave()) {

            // If name is not empty
            if (oldName != null && !oldName.isEmpty()) {

                // Check existing
                String rootId = thesaurus.getRootThesaurus();
                RootThesaurus rootThesaurus = getRootThesaurus(rootId);
                if (VradiService.getVradiDataService().isThesaurusNameExistsInRootThesaurus(rootThesaurus,
                        thesaurus.getName(), thesaurus.getWikittyId())) {

                    // Ask retry
                    int n = JOptionPane.showConfirmDialog(getUI(context),
                            _("vradi.adminThesaurus.warnNameIsExisting", oldName),
                            _("vradi.admin.warnTitle"),
                            JOptionPane.OK_CANCEL_OPTION);

                    if (n == JOptionPane.OK_OPTION) {
                        editThesaurus(context, thesaurus);
                    }
                    return;
                }

                String newName = thesaurus.getName();

                boolean responce = true;

                if (!oldName.equals(newName)) {

                    thesaurus.setName(oldName);

                    // Confirm
                    responce = showThesaurusConfirmDialog(context, thesaurus,
                                _("vradi.adminThesaurus.askEditThesaurus", oldName,
                                        ThesaurusDataHelper.getNbFormsForThesaurus(thesaurus)),
                                _("vradi.admin.saveTitle"));
                }

                if (responce) {  

                    // Set name
                    thesaurus.setName(newName);

                    // Impact user request
                    modifyQueries(context, rootThesaurus.getName(), rootThesaurus.getName(), oldName, thesaurus.getName());

                    // Store
                    getProxy().store(thesaurus);

                    // Select node created
                    ThesaurusTreeTableHelper helper = context.getContextValue(ThesaurusTreeTableHelper.class);
                    UIHelper.selectNodeLater(helper, thesaurus.getWikittyId());
                }
            }
        }
    }

    public void moveThesaurus(JAXXContext context, Thesaurus thesaurus) {

        // To prevent wikity obselete
        thesaurus = ThesaurusDataHelper.restoreThesaurus(thesaurus.getWikittyId());

        // Get root thesaurus of selected
        String rootThesaurusId = thesaurus.getRootThesaurus();
        RootThesaurus rootThesaurus = getRootThesaurus(rootThesaurusId);

        // Create new helpers
        ThesaurusTreeTableHelper selectHelper = new ThesaurusTreeTableHelper(rootThesaurus);

        // Create select ui
        ThesaurusPathChooserUI selectUI = new ThesaurusPathChooserUI(context,
                selectHelper);

        selectUI.setVisible(true);

        // If select
        if (selectUI.isSaved()) {

            // Confirm
            boolean responce = showThesaurusConfirmDialog(context, thesaurus,
                        _("vradi.adminThesaurus.askMoveThesaurus", thesaurus.getName()),
                        _("vradi.adminThesaurus.moveTitle"));

            if (responce) {

                // Get new parent selected
                VradiTreeTableNode selectedNode = getSelectedNode(selectHelper);

                // Apply move
                String id = selectedNode.getId();
                if (!id.equals(thesaurus.getParent())) {
                    
                    thesaurus.setParent(id);

                    // Store
                    getProxy().store(thesaurus);

                    // Select node created
                    ThesaurusTreeTableHelper helper = context.getContextValue(ThesaurusTreeTableHelper.class);
                    UIHelper.selectNodeLater(helper, thesaurus.getWikittyId());
                }
            }
        }
    }

    public void deleteRootThesaurus(JAXXContext context, RootThesaurus rootThesaurus) {

        // Confirm
        int n = JOptionPane.showConfirmDialog(getUI(context),
                    _("vradi.adminThesaurus.askDeleteRootThesaurus", rootThesaurus.getName()),
                    _("vradi.admin.deleteTitle"),
                    JOptionPane.OK_CANCEL_OPTION);

        if (n == JOptionPane.OK_OPTION) {

            // TODO SLE 19/09/10 : impact query for root deletion ?
            // Impact user request
            // modifyQueries(context, rootThesaurus);

            // delete all thesaurus and subnodes recursively
            try {
                VradiService.getVradiDataService().deleteThesaurus(rootThesaurus.getWikittyId());
            } catch (VradiException ex) {
                throw new RuntimeException("Can't delete thesaurus", ex);
            }
        }
    }

    public boolean deleteThesaurus(JAXXContext context, Thesaurus thesaurus) {

        // Confirm
        boolean response = showThesaurusConfirmDialog(context, thesaurus,
                    _("vradi.adminThesaurus.askDeleteThesaurus", thesaurus.getName(),
                            ThesaurusDataHelper.getNbFormsForThesaurus(thesaurus)),
                    _("vradi.admin.deleteTitle"));

        if (response) {

            // Keep parent id
            String parentId = thesaurus.getParent();

            // delete all thesaurus and subnodes recursively
            try {
                List<String> childrenList = VradiService.getVradiDataService().deleteThesaurus(thesaurus.getWikittyId());
                
                // Get root thesaurus
                RootThesaurus rootThesaurus = ThesaurusDataHelper.restoreRootThesaurus(thesaurus.getRootThesaurus());
                
                modifyDeleteRequest(context, rootThesaurus, thesaurus, childrenList);
                
                // Select node created
                ThesaurusTreeTableHelper helper = context.getContextValue(ThesaurusTreeTableHelper.class);
                UIHelper.selectNodeLater(helper, parentId);
                
            } catch (VradiException ex) {
                throw new RuntimeException("Can't delete thesaurus", ex);
            }

            return true;
        }
        return false;
    }

    protected void modifyDeleteRequest(JAXXContext context, RootThesaurus rootThesaurus, Thesaurus thesaurus, List<String> childrenList) {
        // Impact group request
        // before has been deleted, so the replacement would be the parent node
        for (String childId : childrenList) {
            Thesaurus child = ThesaurusDataHelper.restoreThesaurus(childId);
            if (ThesaurusDataHelper.isFirstChild(child)) {
                modifyQueries(context, rootThesaurus.getName(), rootThesaurus.getName(), child.getName(), null);
                return;
            }

            Thesaurus parent = null;

            // Get first parent not deleted
            String parentId = thesaurus.getParent();
            while (childrenList.contains(parentId)) {

                parent = ThesaurusDataHelper.restoreThesaurus(parentId);

                if (ThesaurusDataHelper.isFirstChild(parent)) {
                    modifyQueries(context, rootThesaurus.getName(), rootThesaurus.getName(), child.getName(), null);
                    return;
                }
                parentId = parent.getParent();
            }

            // Propose parent
            modifyQueries(context, rootThesaurus.getName(), rootThesaurus.getName(), child.getName(), parent == null ? null : parent.getName());
        }
    }

    public void addChildThesaurusToSelected(JAXXContext context, ThesaurusTreeTableHelper helper) {
        VradiTreeTableNode selectedNode = getSelectedNode(helper);
        RootThesaurus rootThesaurus = getRootThesaurus(selectedNode);
        if (rootThesaurus != null) {
            createThesaurus(context, rootThesaurus);
            return;
        }
        createThesaurus(context, getThesaurus(selectedNode));
    }

    public void editThesaurusSelected(JAXXContext context, ThesaurusTreeTableHelper helper) {
        VradiTreeTableNode selectedNode = getSelectedNode(helper);
        RootThesaurus rootThesaurus = getRootThesaurus(selectedNode);
        if (rootThesaurus != null) {
            editRootThesaurus(context, rootThesaurus);
            return;
        }
        editThesaurus(context, getThesaurus(selectedNode));
    }

    public void deleteThesaurusSelected(JAXXContext context, ThesaurusTreeTableHelper helper) {
        VradiTreeTableNode selectedNode = getSelectedNode(helper);
        RootThesaurus rootThesaurus = getRootThesaurus(selectedNode);
        if (rootThesaurus != null) {
            deleteRootThesaurus(context, rootThesaurus);
            return;
        }
        deleteThesaurus(context, getThesaurus(selectedNode));
    }

    public void moveThesaurusSelected(JAXXContext context, ThesaurusTreeTableHelper helper) {
        VradiTreeTableNode selectedNode = getSelectedNode(helper);

        Thesaurus thesaurus = getThesaurus(selectedNode);

        moveThesaurus(context, thesaurus);
    }

    public boolean showThesaurusConfirmDialog(JAXXContext context, Thesaurus concerned, String title, String msg){
        ThesaurusConfirmDialogUI condirmDialogUI = new ThesaurusConfirmDialogUI(context);
        condirmDialogUI.init(concerned, title, msg);
        return condirmDialogUI.isSave();
    }

    public void showApply(JAXXContext context, Thesaurus concerned){
        UIHelper.getHandler(context, AdminHandler.class).openRequest(context, concerned);
    }
//
//    public void modifyAllQuery(JAXXContext context, RootThesaurus rootThesaurusModified) {
//
//        RootThesaurus beforeModified = ThesaurusDataHelper.restoreRootThesaurus(rootThesaurusModified.getWikittyId());
//
//        // modify clients queries
//        modifyQueries(context, beforeModified.getName(), rootThesaurusModified.getName());
//    }
//
//    public void modifyAllQuery(JAXXContext context, Thesaurus thesaurusModified, boolean isToDelete) {
//
//        if (isToDelete) {
//            // before has been deleted, so the replacement would be the parent node
//            while (!ThesaurusDataHelper.isFirstChild(thesaurusModified) && isToDelete) {
//                String parentId = thesaurusModified.getParent();
//                thesaurusModified = ThesaurusDataHelper.restoreThesaurus(parentId);
//            }
//        }
//
//        Thesaurus beforeModified = ThesaurusDataHelper.restoreThesaurus(thesaurusModified.getWikittyId());
//
//        modifyQueries(context, beforeModified.getName(),
//                ThesaurusDataHelper.getRequestName(thesaurusModified));
//    }

    protected void modifyQueries(JAXXContext context,
                                 String rootThesaurusBefore,
                                 String rootThesaurusAfter,
                                 String thesaurusBefore,
                                 String thesaurusAfter) {
        Map<Group, java.util.List<QueryBean>> queryMakers = VradiService
                .getVradiStorageService()
                .getQueriesToModifyAfterThesaurusModification(rootThesaurusBefore, thesaurusBefore);

        if (queryMakers.isEmpty()) {
            // the modified thesaurus is not part of a client query
            if (log.isDebugEnabled()) {
                log.debug("[ModifyQueries] Query makers are empty for thesaurus : " + rootThesaurusBefore +
                        ":" + thesaurusBefore);
            }
            return;
        }

        if (log.isDebugEnabled()) {
            log.debug("[ModifyQueries] Query makers are not empty for thesaurus : " + rootThesaurusBefore +
                    ":" + thesaurusBefore + " will show modification UI");
        }
        // show window with modified queries according thesaurus nodes changes
        ThesaurusQueryChangesHandler handler = new ThesaurusQueryChangesHandler();
        ThesaurusQueryChangesUI thesaurusChangesUI = handler.initUI(
                context, queryMakers,
                rootThesaurusBefore, rootThesaurusAfter,
                thesaurusBefore, thesaurusAfter);
        thesaurusChangesUI.setModifiedQueries(Boolean.FALSE);
        thesaurusChangesUI.setVisible(true);

        if (!thesaurusChangesUI.isModifiedQueries()) {
            return;
        }

        try {
            // retrieve updated groups
            ThesaurusChangesTreeTableModel model = thesaurusChangesUI
                    .getContextValue(ThesaurusChangesTreeTableModel.class);
            java.util.List<Group> updatedQueryMakers = model.getUpdatedQueryMakers();

            // save updated groups
            VradiService.getWikittyProxy().store(updatedQueryMakers);

            if (log.isDebugEnabled()) {
                log.debug("[ModifyQueries] Store modifications");
            }

        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    /**
     * Change color
     *
     * @param context jaxx context
     * @param helper  helpers of adminThesaurus
     */
    public void changeThesaurusColor(JAXXContext context,
                              ThesaurusTreeTableHelper helper){

        VradiTreeTableNode selectedNode = getSelectedNode(helper);
        Thesaurus selected = getThesaurus(selectedNode);

        int depth = ThesaurusDataHelper.getDepth(selected);

        Color initialColor = ColorsMap.getInstance().get(depth);
        Color colorChoosed = JColorChooser.showDialog((AdminThesaurusUI) context, _("vradi.thesaurus.colorTitle"), initialColor);

        // Save in properties
        if (log.isDebugEnabled()){
            log.debug("Store color " + colorChoosed + " for depth " + depth);
        }
        VradiHelper.storeThesaurusColor(depth, colorChoosed);
       
        reinitColors();
    }

    public void reinitColors(){
        ColorsMap.getInstance().updateColors();
    }

    /**
     * Export all clients as CSV.
     *
     * Used in {@link com.jurismarches.vradi.ui.admin.nav.AdminClientNavUI}.
     *
     * @param adminThesaurusUI parent ui
     */
    public void exportThesaurusCSV(AdminThesaurusUI adminThesaurusUI) {

        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminThesaurus.exportThesaurusTitle"));
            chooser.setApproveButtonText(_("vradi.adminThesaurus.exportButtonText"));
            int returnVal = chooser.showOpenDialog(adminThesaurusUI);

            if (returnVal == JFileChooser.APPROVE_OPTION) {
                File selectedFile = chooser.getSelectedFile();

                if (log.isDebugEnabled()) {
                    log.debug("Exporting thesaurus to : " + selectedFile.getAbsolutePath());
                }

                // creation du criteria wikitty
                // (export de tous les wikitty avec extension "thesaurus"

                // TODO EC-20100416 check to not export "all" tree nodes
                // just "thesaurus" tree nodes
                
                //FIXME EC20100913 don't use business code here, move to services
                Search search = Search.query().eq(org.nuiton.wikitty.search.Element.ELT_EXTENSION,
                        TreeNode.EXT_TREENODE);
                Criteria criteria = search.criteria();

                // export
                VradiStorageService vradiStorageService = getVradiStorageService();
                String vscContent = vradiStorageService.exportAsCSV(criteria);

                // write content into file
                FileUtils.writeStringToFile(selectedFile, vscContent);
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export thesaurus", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export thesaurus", eee);
            }
        }
    }

    private VradiStorageService vradiStorageService = null;

    protected VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = VradiService.getVradiStorageService();
        }
        return vradiStorageService;
    }

    /**
     * Import CSV file.
     *
     * Can be used to import clients, groups or thesaurus.
     *
     * @param adminThesaurusUI parent ui
     */
    public void importCSV(AdminThesaurusUI adminThesaurusUI) {
        
        // warn user about thesaurus deletion
        int response = JOptionPane.showConfirmDialog(adminThesaurusUI,
                _("Warning all your thesaurus will be deleted before new import. Do you want to continue ?"),
                _("Confirm thesaurus deletion"), JOptionPane.OK_CANCEL_OPTION);
        if (response != JOptionPane.OK_OPTION) {
            return;
        }
        
        try {
            JFileChooser chooser = new JFileChooser();
            chooser.setDialogTitle(_("vradi.adminThesaurus.importTitle"));
            chooser.setApproveButtonText(_("vradi.adminThesaurus.importButtonText"));
            int returnVal = chooser.showOpenDialog(adminThesaurusUI);
            if (returnVal == JFileChooser.APPROVE_OPTION) {

                // delete all thesaurus
                VradiService.getVradiDataService().deleteAllThesaurus();
                
                File selectedFile = chooser.getSelectedFile();

                if (log.isDebugEnabled()) {
                    log.debug("Importing file : " + selectedFile.getAbsolutePath());
                }

                // export
                VradiStorageService vradiStorageService = getVradiStorageService();
                String uri = VradiService.getFileService().uploadTempFile(selectedFile);
                vradiStorageService.importAsCSV(uri);
            }
        } catch (VradiException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
            ErrorDialogUI.showError(eee);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import file", eee);
            }
        }
    }

    /**
     * Copy to clipboard thesaurus action.
     * 
     * Get selection thesaurus, and set following string in clipboard :
     * rootThesaurusName:"selectedThesaurusName"
     */
    public static class CopyToClipboard extends AbstractAction {

        /** serialVersionUID. */
        private static final long serialVersionUID = 1L;

        protected AdminThesaurusUI thesaurusUI;

        protected ThesaurusCartographyUI thesaurusCartography;

        /**
         * Register the keystroke Control+C (copy) on the treetable of the
         * specified AdminThesaurusUI.
         *
         * @param thesaurusUI ui of thesaurus selection for AO
         */
        public CopyToClipboard(AdminThesaurusUI thesaurusUI) {
            super(_("vradi.common.copy"));
            this.thesaurusUI = thesaurusUI;

            JXTreeTable treeTable = thesaurusUI.getThesaurus();
            InputMap inputMap = treeTable.getInputMap();
            inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_MASK),
                    "ThesaurusHandler.CopyToClipboard");

            ActionMap actionMap = treeTable.getActionMap();
            actionMap.put("ThesaurusHandler.CopyToClipboard", this);
        }

        /**
         * Register the keystroke Control+C (copy) on the treetable of the
         * specified thesaurus cartography.
         *
         * @param thesaurusCartography ui of thesaurus selection for AO
         */
        public CopyToClipboard(ThesaurusCartographyUI thesaurusCartography) {
            super(_("vradi.common.copy"));
            this.thesaurusCartography = thesaurusCartography;

            JXTreeTable treeTable = thesaurusCartography.getCartographyTable();
            InputMap inputMap = treeTable.getInputMap();
            inputMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_MASK),
                    "ThesaurusHandler.CopyToClipboard");

            ActionMap actionMap = treeTable.getActionMap();
            actionMap.put("ThesaurusHandler.CopyToClipboard", this);
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            Thesaurus selectedThesaurus = null;
            if (thesaurusUI != null) {
                ThesaurusTreeTableHelper helper = thesaurusUI.helper;

                VradiTreeTableNode selectedNode = helper.getSelectedNode();
                if (selectedNode == null || selectedNode.isRoot()) {
                    return;
                }
                String id = selectedNode.getId();
                selectedThesaurus = VradiService.getWikittyProxy().restore(Thesaurus.class, id);
            } else {
                int selectedRow = thesaurusCartography.getCartographyTable().getSelectedRow();
                Object selected = thesaurusCartography.getCartographyTable().getValueAt(selectedRow, 0);
                if (!(selected instanceof Thesaurus)) {
                    return;
                }
                selectedThesaurus = (Thesaurus)selected;
            }

            if (selectedThesaurus == null) {
                return;
            }
            RootThesaurus rootThesaurus = VradiService.getWikittyProxy().restore(RootThesaurus.class, selectedThesaurus.getRootThesaurus());

            String name = "\"" + selectedThesaurus.getName() + "\"";
            StringSelection selection = new StringSelection(rootThesaurus.getName() + ":" + name);

            Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
            clipboard.setContents(selection, null);
        }
    }
}
