package com.jurismarches.vradi.ui.query;

import com.jurismarches.vradi.VradiService;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiQueryBean;
import com.jurismarches.vradi.ui.VradiComparators;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;
import org.nuiton.i18n.I18n;

import javax.swing.tree.TreePath;
import java.io.IOException;
import java.util.*;

/**
 * ThesaurusChangesTreeTableModel.
 *
 * @author schorlet
 * @version $Revision: 887 $ $Date: 2010-05-10 15:23:58 +0200 (lun., 10 mai 2010) $
 * @since 8 avr. 2010 15:19:16
 */
public class ThesaurusChangesTreeTableModel extends AbstractTreeTableModel {
    private static final Log log = LogFactory.getLog(ThesaurusChangesTreeTableModel.class);
    
    /** clients and groups */
    final ArrayList<QueryMaker> topLevels = new ArrayList<QueryMaker>();

    /** users indexed by clientId */
    final HashMap<String, List<User>> userMap = new HashMap<String, List<User>>();
    
    /** queryMakers indexed by wikittyId */
    final HashMap<String, QueryMaker> queryMakersMap = new HashMap<String, QueryMaker>();
    
    /** queries indexed by wikittyId */
    final HashMap<String, List<VradiQueryBean>> queryBeanMap = new HashMap<String, List<VradiQueryBean>>();
    
    /** key:before, value:after */
    final HashMap<VradiQueryBean, VradiQueryBean> modifiedBeanMap = new HashMap<VradiQueryBean, VradiQueryBean>();
    
    /** queries validation state */
    final HashMap<VradiQueryBean, Boolean> queryBeanValidate = new HashMap<VradiQueryBean, Boolean>();
    
    public ThesaurusChangesTreeTableModel(Map<QueryMaker, List<VradiQueryBean>> queryMap,
            String before, String after) {
        super(1L);
        
        HashMap<String, Client> clientMap = new HashMap<String, Client>();
        HashMap<String, User> userMap2 = new HashMap<String, User>();
        HashMap<String, Group> groupMap = new HashMap<String, Group>();
        
        for (Map.Entry<QueryMaker, List<VradiQueryBean>> entry : queryMap.entrySet()) {
            // queryMaker
            QueryMaker queryMaker = entry.getKey();
            // original queries for queryMaker
            List<VradiQueryBean> queries = entry.getValue();
            
            String wikittyId = queryMaker.getWikittyId();

            // index queries  by wikittyId
            queryBeanMap.put(wikittyId, queries);

            // index queryMaker by wikittyId
            if (queryMaker instanceof Client) {
                // client
                clientMap.put(wikittyId, (Client) queryMaker);
                
            } else if (queryMaker instanceof Group) {
                // group
                groupMap.put(wikittyId, (Group) queryMaker);
                
            } else if (queryMaker instanceof User) {
                // user
                userMap2.put(wikittyId, (User) queryMaker);
                
                User user = (User) queryMaker;
                String clientId = user.getClient();
                
                if (userMap.containsKey(clientId)) {
                    userMap.get(clientId).add(user);
                } else {
                    List<User> users = new ArrayList<User>();
                    users.add(user);
                    userMap.put(clientId, users);
                }
            }
            
            // index queries by original:modified
            for (VradiQueryBean original : queries) {
                String query = original.getQuery();
                if (before.indexOf(' ') > 0) {
                    query = StringUtils.replace(query, before, after);
                } else if (after.indexOf(' ') == -1) {
                    query = StringUtils.replace(query, before, after);
                } else {
                    query = StringUtils.replace(query, before, "\"" + after + "\"");
                }
                
                VradiQueryBean modified = original.setQuery(query);
                
                modifiedBeanMap.put(original, modified);
                queryBeanValidate.put(original, Boolean.TRUE);
            }
        }
        
        // find client for users
        for (String clientId : userMap.keySet()) {
            if (!clientMap.containsKey(clientId)) {
                try {
                    Client client = getVradiStorageService().getClient(clientId);
                    if (client != null) {
                        clientMap.put(client.getWikittyId(), client);
                        
                        queryBeanMap.put(client.getWikittyId(),
                            Collections.<VradiQueryBean>emptyList());
                    }
                } catch (VradiException e) {
                    log.error(e.getMessage(), e);
                }
            }
        }
        
        List<Client> clientList = new ArrayList<Client>(clientMap.values());
        List<Group> groupList = new ArrayList<Group>(groupMap.values());
        
        Collections.sort(clientList, VradiComparators.CLIENT_COMPARATOR);
        Collections.sort(groupList, VradiComparators.GROUP_COMPARATOR);
        
        topLevels.addAll(clientList);
        topLevels.addAll(groupList);
        
        queryMakersMap.putAll(clientMap);
        queryMakersMap.putAll(userMap2);
        queryMakersMap.putAll(groupMap);
    }
    
    public List<QueryMaker> getUpdatedQueryMakers() throws IOException {
        List<QueryMaker> list = new ArrayList<QueryMaker>();
        
        for (QueryMaker queryMaker : queryMakersMap.values()) {
            list.add(queryMaker);
            Set<String> queries = new HashSet<String>();
            Set<String> initialQueries = queryMaker.getQueries();
            if (initialQueries != null) {
                queries.addAll(initialQueries);
            }
            queryMaker.clearQueries();
            
            for (String query : queries) {
                VradiQueryBean original = new VradiQueryBean(query, queryMaker.getWikittyId());
                
                if (queryBeanValidate.containsKey(original) && queryBeanValidate.get(original)) {
                    VradiQueryBean modified = modifiedBeanMap.get(original);
                    queryMaker.addQueries(modified.toString());
                    
                } else {
                    queryMaker.addQueries(query);
                }
            }
            
        }
        return list;
    }
    
    @Override
    public int getColumnCount() {
        return 5;
    }

    @Override
    public Object getValueAt(Object node, int column) {
        if (node instanceof QueryMaker) {
            if (column == 0) {
                QueryMaker queryMaker = (QueryMaker) node;
                if (queryMaker instanceof Client) {
                    return ((Client)queryMaker).getName();
                    
                } else if (queryMaker instanceof User) {
                    return ((User)queryMaker).getName();
                    
                } else if (queryMaker instanceof Group) {
                    return ((Group)queryMaker).getName();
                }
            }
            
        } else if (node instanceof VradiQueryBean) {
            VradiQueryBean bean = (VradiQueryBean) node;
            if (column == 0) {
                return bean.getName();
                
            } else if (column == 1) {
                return bean.getDescription();
                
            } else if (column == 2) {
                return bean.getQuery();
                
            } else if (column == 3) {
                VradiQueryBean modifiedBean = modifiedBeanMap.get(bean);
                return modifiedBean.getQuery();
                
            } else if (column == 4) {
                boolean validate = queryBeanValidate.get(bean);
                return validate;
            }
        }
        
        return null;
    }

    @Override
    public boolean isCellEditable(Object node, int column) {
        if (column == 3 || column == 4) {
            boolean b = node instanceof VradiQueryBean;
            return b;
        }
        return false;
    }
    
    @Override
    public void setValueAt(Object value, Object node, int column) {
        if (column == 3) {
            VradiQueryBean bean = (VradiQueryBean) node;
            String beanId = bean.getId();
            if (beanId == null) {
                return;
            }
            
            VradiQueryBean initialBean = modifiedBeanMap.get(bean);
            VradiQueryBean modifiedBean = initialBean.setQuery(String.valueOf(value));

            if (!initialBean.equals(modifiedBean)) {
                modifiedBeanMap.put(bean, modifiedBean);
                queryBeanValidate.put(bean, Boolean.TRUE);
                fireQueryChanged(bean);
            }
            
        } else if (column == 4) {
            VradiQueryBean bean = (VradiQueryBean) node;
            String beanId = bean.getId();
            if (beanId == null) {
                return;
            }
            
            queryBeanValidate.put(bean, (Boolean) value);
            fireQueryChanged(bean);
        }
    }
    
    private void fireQueryChanged(VradiQueryBean bean) {
        String beanId = bean.getId();
        
        TreePath parentPath = null;
        int index = 0;
        
        QueryMaker queryMaker = queryMakersMap.get(beanId);
        
        if (queryMaker instanceof User) {
            User user = (User) queryMaker;
            QueryMaker client = queryMakersMap.get(user.getClient());

            Object[] paths = new Object[3];
            paths[0] = 1L;
            paths[1] = client;
            paths[2] = queryMaker;
            
            parentPath = new TreePath(paths);
            index = getIndexOfChild(paths[2], bean);
        
        } else {
            Object[] paths = new Object[2];
            paths[0] = 1L;
            paths[1] = queryMaker;
            
            parentPath = new TreePath(paths);
            index = getIndexOfChild(paths[1], bean);
        }
        
        modelSupport.fireChildChanged(parentPath, index, bean);
    }
    
    @Override
    public Object getChild(Object parent, int index) {
        if (parent instanceof Long) {
            return topLevels.get(index);
            
        } else if (parent instanceof Client) {
            Client client = (Client) parent;
            String wikittyId = client.getWikittyId();
            
            if (userMap.containsKey(wikittyId)) {
                List<User> users = userMap.get(wikittyId);
                if (index < users.size()) {
                    return users.get(index);
                } else {
                    index = index - users.size();
                    List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                    VradiQueryBean bean = list.get(index);
                    return bean;
                }
                
            } else {
                List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                VradiQueryBean bean = list.get(index);
                return bean;
            }
        
        } else if (parent instanceof User) {
            User user = (User) parent;
            List<VradiQueryBean> list = queryBeanMap.get(user.getWikittyId());
            VradiQueryBean bean = list.get(index);
            return bean;
        
        } else if (parent instanceof Group) {
            Group group = (Group) parent;
            List<VradiQueryBean> list = queryBeanMap.get(group.getWikittyId());
            VradiQueryBean bean = list.get(index);
            return bean;
        }
        
        return null;
    }

    @Override
    public int getChildCount(Object parent) {
        if (parent instanceof Long) {
            return topLevels.size();
            
        } else if (parent instanceof Client) {
            Client client = (Client) parent;
            String wikittyId = client.getWikittyId();
            
            if (userMap.containsKey(wikittyId)) {
                List<User> users = userMap.get(wikittyId);
                List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                return users.size() + list.size();
            } else {
                List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                return list.size();
            }
        
        } else if (parent instanceof User) {
            User user = (User) parent;
            List<VradiQueryBean> list = queryBeanMap.get(user.getWikittyId());
            return list.size();
        
        } else if (parent instanceof Group) {
            Group group = (Group) parent;
            List<VradiQueryBean> list = queryBeanMap.get(group.getWikittyId());
            return list.size();
        }
        
        return 0;
    }
    
    @Override
    public int getIndexOfChild(Object parent, Object child) {
        if (parent instanceof Long) {
            return topLevels.indexOf(child);
            
        } else if (parent instanceof Client) {
            Client client = (Client) parent;
            String wikittyId = client.getWikittyId();
            
            if (userMap.containsKey(wikittyId)) {
                List<User> users = userMap.get(wikittyId);
                if (child instanceof User) {
                    return users.indexOf(child);
                } else {
                    List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                    return users.size() + list.indexOf(child);
                }
                
            } else {
                List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                return list.indexOf(child);
            }
        
        } else if (parent instanceof User) {
            User user = (User) parent;
            List<VradiQueryBean> list = queryBeanMap.get(user.getWikittyId());
            return list.indexOf(child);
        
        } else if (parent instanceof Group) {
            Group group = (Group) parent;
            List<VradiQueryBean> list = queryBeanMap.get(group.getWikittyId());
            return list.indexOf(child);
        }
        
        return -1;
    }
    
    @Override
    public String getColumnName(int column) {
        String columnName = null;
        if (column == 0) {
            columnName = I18n._("vradi.requestFormView.resultTable.name");
        } else if (column == 1) {
            columnName = I18n._("vradi.requestFormView.resultTable.description");
        } else if (column == 2) {
            columnName = I18n._("vradi.requestFormView.resultTable.oldQuery");
        } else if (column == 3) {
            columnName = I18n._("vradi.requestFormView.resultTable.newQuery");
        }
        return columnName;
    }
    
    private VradiStorageService vradiStorageService = null;
    
    protected VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = VradiService.getVradiStorageService();
        }
        return vradiStorageService;
    }
    
}
