/*
 * *##%
 * Vradi :: Swing
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableModel;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.context.JAXXInitialContext;

import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.jdesktop.swingx.treetable.AbstractTreeTableModel;
import org.jdesktop.swingx.treetable.TreeTableModel;
import org.nuiton.i18n.I18n;

import com.jurismarches.vradi.VradiService;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiStorageService;
import com.jurismarches.vradi.services.dto.VradiQueryBean;

/**
 * RequestFormViewHandler.
 *
 * @author schorlet
 * @version $Revision$ $Date$
 * @since 19 mars 2010 14:42:03
 */
public class RequestFormViewHandler {
    private VradiStorageService vradiStorageService;

    protected VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = VradiService.getVradiStorageService();
        }
        return vradiStorageService;
    }
    
    public RequestFormViewUI initUI(JAXXContext rootContext, Map<QueryMaker, List<VradiQueryBean>> queries) {
        JAXXInitialContext context = new JAXXInitialContext().add(rootContext).add(this);
        context.add("queries", queries);
        
        RequestFormViewUI requestFormViewUI = new RequestFormViewUI(context);
        JXTreeTable resultTreeTable = requestFormViewUI.getResultTreeTable();
        JXTable resultTable = requestFormViewUI.getResultTable();
        resultTreeTable.expandAll();
        
        Highlighter highlighter = HighlighterFactory.createAlternateStriping(
        HighlighterFactory.NOTEPAD, HighlighterFactory.GENERIC_GRAY);
        resultTable.addHighlighter(highlighter);
        resultTreeTable.addHighlighter(highlighter);
    
        return requestFormViewUI;
    }

    public TableModel getResultTableModel(RequestFormViewUI context) {
        QueryTableModel tableModel = context.getContextValue(QueryTableModel.class);
        if (tableModel == null) {
            Map<QueryMaker, List<VradiQueryBean>> queries = context.getContextValue(Map.class, "queries");
            tableModel = new QueryTableModel(queries);
            context.setContextValue(tableModel);
            
            final JXTable resultTable = context.getResultTable();
            tableModel.addTableModelListener(new TableModelListener() {
                @Override
                public void tableChanged(TableModelEvent e) {
                    resultTable.packAll();
                }
            });
        }
        return tableModel;
    }
    
    public TreeTableModel getResultTreeTableModel(RequestFormViewUI context) {
        QueryTreeTableModel treeTableModel = context.getContextValue(QueryTreeTableModel.class);
        if (treeTableModel == null) {
            Map<QueryMaker, List<VradiQueryBean>> queries = context.getContextValue(Map.class, "queries");
            treeTableModel = new QueryTreeTableModel(queries);
            context.setContextValue(treeTableModel);
            
            JXTreeTable resultTreeTable = context.getResultTreeTable();
            QueryTreeTableModelListener treeModelListener = new QueryTreeTableModelListener(resultTreeTable);
            treeTableModel.addTreeModelListener(treeModelListener);
        }
        return treeTableModel;
    }
    
    void close(RequestFormViewUI context) {
        context.setVisible(false);
        context.dispose();
    }
    
    class QueryTreeTableModelListener implements TreeModelListener {
        final JXTreeTable treeTable;
        
        public QueryTreeTableModelListener(JXTreeTable treeTable) {
            this.treeTable = treeTable;
        }
        
        @Override
        public void treeNodesChanged(TreeModelEvent e) {
        }
        @Override
        public void treeNodesInserted(TreeModelEvent e) {
        }
        @Override
        public void treeNodesRemoved(TreeModelEvent e) {
        }
        @Override
        public void treeStructureChanged(TreeModelEvent e) {
            treeTable.packAll();
        }
    }
    
    class QueryTreeTableModel extends AbstractTreeTableModel {
        /** clients and groups */
        final ArrayList<QueryMaker> topLevels = new ArrayList<QueryMaker>();

        /** users indexed by clientId */
        final HashMap<String, List<User>> userMap = new HashMap<String, List<User>>();
        
        /** queries indexed by wikittyId */
        final HashMap<String, List<VradiQueryBean>> queryBeanMap = new HashMap<String, List<VradiQueryBean>>();
        
        public QueryTreeTableModel(Map<QueryMaker, List<VradiQueryBean>> queryMap) {
            super(1L);
            
            HashMap<String, Client> clientMap = new HashMap<String, Client>();
            HashMap<String, Group> groupMap = new HashMap<String, Group>();

            for (Map.Entry<QueryMaker, List<VradiQueryBean>> entry : queryMap.entrySet()) {
                QueryMaker queryMaker = entry.getKey();
                List<VradiQueryBean> queries = entry.getValue();
                
                String wikittyId = queryMaker.getWikittyId();

                // client
                if (queryMaker instanceof Client) {
                    clientMap.put(wikittyId, (Client) queryMaker);
                    
                // group
                } else if (queryMaker instanceof Group) {
                    groupMap.put(wikittyId, (Group) queryMaker);
                    
                // user
                } else if (queryMaker instanceof User) {
                    User user = (User) queryMaker;
                    String clientId = user.getClient();
                    
                    if (userMap.containsKey(clientId)) {
                        userMap.get(clientId).add(user);
                    } else {
                        List<User> users = new ArrayList<User>();
                        users.add(user);
                        userMap.put(clientId, users);
                    }
                }
                
                // queries
                queryBeanMap.put(wikittyId, queries);
            }
            
            // find client for users
            for (String clientId : userMap.keySet()) {
                if (!clientMap.containsKey(clientId)) {
                    try {
                        Client client = getVradiStorageService().getClient(clientId);
                        if (client != null) {
                            clientMap.put(client.getWikittyId(), client);
                            queryBeanMap.put(client.getWikittyId(), Collections.<VradiQueryBean>emptyList());
                        }
                    } catch (VradiException e) {
                    }
                }
            }
            
            List<Client> clientList = new ArrayList<Client>(clientMap.values());
            List<Group> groupList = new ArrayList<Group>(groupMap.values());
            
            Collections.sort(clientList, VradiComparators.CLIENT_COMPARATOR);
            Collections.sort(groupList, VradiComparators.GROUP_COMPARATOR);
            
            topLevels.addAll(clientList);
            topLevels.addAll(groupList);
        }
        
        @Override
        public int getColumnCount() {
            return 3;
        }

        @Override
        public Object getValueAt(Object node, int column) {
            if (node instanceof QueryMaker) {
                if (column == 0) {
                    QueryMaker queryMaker = (QueryMaker) node;
                    if (queryMaker instanceof Client) {
                        return ((Client)queryMaker).getName();
                        
                    } else if (queryMaker instanceof User) {
                        return ((User)queryMaker).getName();
                        
                    } else if (queryMaker instanceof Group) {
                        return ((Group)queryMaker).getName();
                    }
                }
                
            } else if (node instanceof VradiQueryBean) {
                VradiQueryBean bean = (VradiQueryBean) node;
                if (column == 0) {
                    return bean.getName();
                    
                } else if (column == 1) {
                    return bean.getDescription();
                    
                } else if (column == 2) {
                    return bean.getQuery();
                }
            }
            
            return null;
        }

        @Override
        public Object getChild(Object parent, int index) {
            if (parent instanceof Long) {
                return topLevels.get(index);
                
            } else if (parent instanceof Client) {
                Client client = (Client) parent;
                String wikittyId = client.getWikittyId();
                
                if (userMap.containsKey(wikittyId)) {
                    List<User> users = userMap.get(wikittyId);
                    if (index < users.size()) {
                        return users.get(index);
                    } else {
                        index = index - users.size();
                        List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                        VradiQueryBean bean = list.get(index);
                        return bean;
                    }
                    
                } else {
                    List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                    VradiQueryBean bean = list.get(index);
                    return bean;
                }
            
            } else if (parent instanceof User) {
                User user = (User) parent;
                List<VradiQueryBean> list = queryBeanMap.get(user.getWikittyId());
                VradiQueryBean bean = list.get(index);
                return bean;
            
            } else if (parent instanceof Group) {
                Group group = (Group) parent;
                List<VradiQueryBean> list = queryBeanMap.get(group.getWikittyId());
                VradiQueryBean bean = list.get(index);
                return bean;
            }
            
            return null;
        }

        @Override
        public int getChildCount(Object parent) {
            if (parent instanceof Long) {
                return topLevels.size();
                
            } else if (parent instanceof Client) {
                Client client = (Client) parent;
                String wikittyId = client.getWikittyId();
                
                if (userMap.containsKey(wikittyId)) {
                    List<User> users = userMap.get(wikittyId);
                    List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                    return users.size() + list.size();
                } else {
                    List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                    return list.size();
                }
            
            } else if (parent instanceof User) {
                User user = (User) parent;
                List<VradiQueryBean> list = queryBeanMap.get(user.getWikittyId());
                return list.size();
            
            } else if (parent instanceof Group) {
                Group group = (Group) parent;
                List<VradiQueryBean> list = queryBeanMap.get(group.getWikittyId());
                return list.size();
            }
            
            return 0;
        }
        
        @Override
        public int getIndexOfChild(Object parent, Object child) {
            if (parent instanceof Long) {
                return topLevels.indexOf(child);
                
            } else if (parent instanceof Client) {
                Client client = (Client) parent;
                String wikittyId = client.getWikittyId();
                List<User> users = userMap.get(wikittyId);
                
                if (child instanceof User) {
                    return users.indexOf(child);
                    
                } else {
                    List<VradiQueryBean> list = queryBeanMap.get(wikittyId);
                    return users.size() + list.indexOf(child);
                }
            
            } else if (parent instanceof User) {
                User user = (User) parent;
                List<VradiQueryBean> list = queryBeanMap.get(user.getWikittyId());
                return list.indexOf(child);
            
            } else if (parent instanceof Group) {
                Group group = (Group) parent;
                List<VradiQueryBean> list = queryBeanMap.get(group.getWikittyId());
                return list.indexOf(child);
            }
            
            return 0;
        }
        
        @Override
        public String getColumnName(int column) {
            String columnName = null;
            if (column == 0) {
                columnName = I18n._("vradi.requestFormView.resultTable.name");
            } else if (column == 1) {
                columnName = I18n._("vradi.requestFormView.resultTable.description");
            } else if (column == 2) {
                columnName = I18n._("vradi.requestFormView.resultTable.query");
            }
            return columnName;
        }
    }
    
    /**
     * QueryTableModel.
     */
    class QueryTableModel extends AbstractTableModel {
        private static final long serialVersionUID = 1L;
        final VradiQueryBean[] queries;
        
        public QueryTableModel(Map<QueryMaker, List<VradiQueryBean>> queryMap) {
            HashSet<VradiQueryBean> querySet = new HashSet<VradiQueryBean>();
            Collection<List<VradiQueryBean>> lists = queryMap.values();
            
            for (List<VradiQueryBean> list : lists) {
                querySet.addAll(list);
            }
            
            queries = new VradiQueryBean[querySet.size()];
            querySet.toArray(queries);
        }
        
        @Override
        public String getColumnName(int column) {
            String columnName = null;
            if (column == 0) {
                columnName = I18n._("vradi.requestFormView.resultTable.name");
            } else if (column == 1) {
                columnName = I18n._("vradi.requestFormView.resultTable.description");
            } else if (column == 2) {
                columnName = I18n._("vradi.requestFormView.resultTable.query");
            }
            return columnName;
        }
        
        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            /*
             * columnIndex  | Model         |  VradiQueryBean
             * -------------+---------------+------------------
             * 0            | name          | query
             * 1            | description   | name
             * 2            | query         | description
             */
            VradiQueryBean queryBean = queries[rowIndex];
            String[] queryArray = queryBean.toArray();
            
            if (columnIndex < 2) {
                return queryArray[columnIndex + 1];
            } else {
                return queryArray[0];
            }
        }
        
        @Override
        public int getRowCount() {
            return queries.length;
        }
        
        @Override
        public int getColumnCount() {
            return 3;
        }
    }
}
