/* *##%
 * Copyright (C) 2010 Code Lutin, Chatellier Eric
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *##%*/

package com.jurismarches.vradi.services;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.TreeNode;
import org.nuiton.wikitty.TreeNodeImpl;
import org.nuiton.wikitty.Wikitty;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyExtensionMigration;
import org.nuiton.wikitty.WikittyExtensionMigrationRename;
import org.nuiton.wikitty.WikittyImpl;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyTransaction;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants.FormStatus;
import com.jurismarches.vradi.entities.RootThesaurusImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.VradiUser;

/**
 * Configure wikitty migration.
 * 
 * @author chatellier
 * @version $Revision: 1334 $
 * 
 * Last update : $Date: 2010-09-10 15:20:44 +0200 (ven., 10 sept. 2010) $
 * By : $Author: chatellier $
 */
public class ServiceMigration {

    /** Log. */
    private static final Log log = LogFactory.getLog(ServiceMigration.class);

    /**
     * Install all migration.
     */
    public static void configureMigration() {
        configureMigration010();
    }

    /**
     * Configure migration from 0.0.5 to 0.1.0.
     */
    protected static void configureMigration010() {
        
        // extension VradiUser renamed to WikittyUser
        WikittyExtensionMigration.migrationRegistry.put(VradiUser.EXT_VRADIUSER, new MigrationVradiUserWikittyUser());
    }

    static class MigrationVradiUserWikittyUser extends WikittyExtensionMigrationRename {

        /**
         * Some field names has also changed.
         * 
         * Migration 1.0 > 2.0:
         *   "VradiUser.name" to VradiUser.FQ_FIELD_LOGIN
         *   "VradiUser.password" to VradiUser.FQ_FIELD_PASSWORD
         */
        @Override
        public Wikitty migrate(WikittyService service, WikittyTransaction transaction, Wikitty wikitty, WikittyExtension oldExt, WikittyExtension newExt) {

            String wikittyId = wikitty.getId();
            String wikittyVersion = wikitty.getVersion();

            Wikitty result = new WikittyImpl(wikittyId);
            result.setVersion(wikittyVersion);

            // Add all extension and replace old by the new
            for (WikittyExtension extension : wikitty.getExtensions()) {
                // add one by one to manage require

                // during loop add new when old one is found
                if (extension.equals(oldExt)) {
                    addExtension(service, transaction, result, newExt);
                }
                else {
                    addExtension(service, transaction, result, extension);
                }
            }

            // Pas sur de la method de migration
            // les champs VradiUser.name, VradiUser.password
            // existait en 1.0, pas après
            if (oldExt.getVersion().equals("1.0")) {
                // Copy other values
                Set<String> fqFieldNames = wikitty.fieldNames();
                for (String fieldName : fqFieldNames) {
                    if(fieldName.equals("VradiUser.name")) {
                        Object value = wikitty.getFqField(fieldName);
                        result.setFqField(VradiUser.FQ_FIELD_WIKITTYUSER_LOGIN, value);
                    }
                    else if(fieldName.equals("VradiUser.password")) {
                        Object value = wikitty.getFqField(fieldName);
                        result.setFqField(VradiUser.FQ_FIELD_WIKITTYUSER_PASSWORD, value);
                    }
                    else {
                        Object value = wikitty.getFqField(fieldName);
                        result.setFqField(fieldName, value);
                    }
                }
            }
            else {
                // Copy other values
                Set<String> fqFieldNames = wikitty.fieldNames();
                for (String fieldName : fqFieldNames) {
                    Object value = wikitty.getFqField(fieldName);
                    result.setFqField(fieldName, value);
                }
            }
            return result;
        }
    }

    /**
     * Do migration that wikitty extension migration can't handle.
     * 
     * They are done only at version change time, because, can take
     * some times to perform...
     * 
     * @param lastVersion last launch version
     * @param currentVersion current application version
     * 
     * @param localWikittyProxy 
     */
    public static void versionChangeMigration(String lastVersion,
            String currentVersion, WikittyProxy localWikittyProxy) {

        // 0.0.5 to 0.1.0 => add extension RootThesaurus on "Thesaurus" node
        if ("0.0.5".equals(lastVersion) && "0.1.0".equals(currentVersion)) {

            // modification du noeud root
            if (log.isInfoEnabled()) {
                log.info("Migration root Thesaurus node : add RootThesaurus extension");
            }

            Search search = Search.query().eq(Element.ELT_EXTENSION, TreeNode.EXT_TREENODE)
                .eq(TreeNode.FQ_FIELD_TREENODE_NAME, "Thesaurus");
            Criteria criteria = search.criteria();
            TreeNodeImpl rootThesaurus = localWikittyProxy.findByCriteria(TreeNodeImpl.class, criteria);
            if (rootThesaurus != null) {
                if (!rootThesaurus.getWikitty().hasExtension(RootThesaurusImpl.EXT_ROOTTHESAURUS)) {
                    rootThesaurus.getWikitty().addExtension(RootThesaurusImpl.extensionRootThesaurus);
                    // WARNING, fake modification to force wikitty version increments
                    rootThesaurus.setName("Thesaurus");
                    localWikittyProxy.store(rootThesaurus);
                }

                // referencement du noeud root pour tous les noeud existants
                if (log.isInfoEnabled()) {
                    log.info("Set rootThesaurus id to all existing thesaurus");
                }
                search = Search.query().eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS);
                criteria = search.criteria();
                List<Thesaurus> thesauruses = localWikittyProxy.findAllByCriteria(Thesaurus.class, criteria).getAll();
                for (Thesaurus thesaurus : thesauruses) {
                    thesaurus.setRootThesaurus(rootThesaurus.getWikittyId());
                }
                localWikittyProxy.store(thesauruses);
            }

            // migration des status
            Criteria criteriaStatus = Search.query().eq(Element.ELT_EXTENSION, Status.EXT_STATUS).criteria();
            List<Status> statuses = localWikittyProxy.findAllByCriteria(Status.class, criteriaStatus).getAll();
            List<String> statusesIds = new ArrayList<String>();
            for (Status status : statuses) {
                statusesIds.add(status.getWikittyId());
                if (log.isInfoEnabled()) {
                    log.info("Deleting status " + status.getName());
                }
            }
            localWikittyProxy.delete(statusesIds);
            // les status sont normalement créer tout seul
        }
    }
}
