package com.franciaflex.faxtomail.web.job;

/*
 * #%L
 * FaxToMail :: Web
 * $Id: MailFilterJob.java 185 2014-06-11 05:32:51Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.2/faxtomail-ui-web/src/main/java/com/franciaflex/faxtomail/web/job/MailFilterJob.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.File;
import java.io.FileOutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.URLName;
import javax.mail.internet.ContentType;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;

import com.franciaflex.faxtomail.persistence.entities.Client;
import com.franciaflex.faxtomail.services.service.ClientService;

import org.apache.commons.io.Charsets;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.decorator.Decorator;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.AttachmentImpl;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.EmailAccount;
import com.franciaflex.faxtomail.persistence.entities.EmailImpl;
import com.franciaflex.faxtomail.persistence.entities.EmailProtocol;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.services.DecoratorService;
import com.franciaflex.faxtomail.services.FaxToMailServiceContext;
import com.franciaflex.faxtomail.services.service.ConfigurationService;
import com.franciaflex.faxtomail.services.service.EmailService;
import com.franciaflex.faxtomail.services.service.MailFolderService;

/**
 * @author Kevin Morin (Code Lutin)
 * @since x.x
 */
public class MailFilterJob extends AbstractFaxToMailJob {

    private static final Log log = LogFactory.getLog(MailFilterJob.class);

    protected FaxToMailServiceContext serviceContext;

    //TODO kmorin 20140414 handle imap protocol
    @Override
    public void execute(JobExecutionContext jobExecutionContext) throws JobExecutionException {
        serviceContext = getServiceContext(jobExecutionContext);

        ConfigurationService configurationService = serviceContext.newService(ConfigurationService.class);
        Collection<EmailAccount> emailAccounts = configurationService.getEmailAccounts();

        for (EmailAccount account : emailAccounts) {
            if (account.getProtocol() == EmailProtocol.POP3) {
                checkEmailsOnPop3(account);
            }
        }
    }

    /**
     * Checks the emails of the account
     * @param account
     */
    public void checkEmailsOnPop3(EmailAccount account) {
        Properties properties = new Properties();
        properties.setProperty("mail.store.protocol", "pop3");
        properties.setProperty("mail.pop3.host", account.getHost());
        properties.setProperty("mail.pop3.user", account.getLogin());
        properties.setProperty("mail.pop3.port", String.valueOf(account.getPort()));

        Session session = Session.getInstance(properties);
        Store store = null;
        Folder defaultFolder = null;
        Folder inbox = null;

        try {
            store = session.getStore(new URLName("pop3://" + account.getHost()));
            store.connect(account.getLogin(), account.getPassword());
            defaultFolder = store.getDefaultFolder();
            inbox = defaultFolder.getFolder("INBOX");
            checkEmailsOfFolder(inbox);

        } catch (Exception e) {
            log.error("Error while getting emails from the mailbox", e);

        } finally {
            close(inbox);
            close(defaultFolder);
            try {
                if (store != null && store.isConnected()) {
                    store.close();
                }
            } catch (MessagingException e) {
                log.error("Error while closing the store", e);
            }
        }

    }

    protected void close(Folder folder) {
        if (folder != null && folder.isOpen()) {
            try {
                // TODO 20140414 kmorin close(true) in production
                folder.close(false); // false -> On n'efface pas les messages marqués DELETED

            } catch (Exception e) {
                log.error("Error while closing the folder", e);
            }
        }
    }

    /**
     * Check the emails of teh folder, create the emails in the database and delete the email in the folder
     * @param folder the folder to check
     */
    protected void checkEmailsOfFolder(Folder folder) {
        try {
            folder.open(Folder.READ_WRITE);

            int count = folder.getMessageCount();
            int unread = folder.getUnreadMessageCount();

            log.debug(count + " messages, " + unread + " unread");

            EmailService emailService = serviceContext.newService(EmailService.class);

            for (int i = 0 ; i < count ; i++) {
                Email email = new EmailImpl();

                int messageNumber = count - i;
                Message message = folder.getMessage(messageNumber);

                Charset charset = getCharset(message);

                List<String> modifiedProperties = new ArrayList<>();

//                log.debug("Object: " + message.getSubject());
//                email.s(message.getSubject());
//                modifiedProperties.add(Email.PROPERTY_OBJECT);

                List<Address> recipients = new ArrayList<>();
                Address[] toRecipients = message.getRecipients(Message.RecipientType.TO);
                if (toRecipients != null) {
                    recipients.addAll(Arrays.asList(toRecipients));
                }
                Address[] ccRecipients = message.getRecipients(Message.RecipientType.CC);
                if (ccRecipients != null) {
                    recipients.addAll(Arrays.asList(ccRecipients));
                }

                for (Address address : recipients) {
                    String recipient = address.toString();
                    MailFolder mailFolder = serviceContext.newService(MailFolderService.class).getFolderForRecipient(recipient);

                    if (mailFolder != null) {
                        log.debug("\tTo : " + address + " -> " + (mailFolder != null ? mailFolder.getName() : "none"));

                        email.setRecipient(recipient);
                        email.setMailFolder(mailFolder);
                        modifiedProperties.add(Email.PROPERTY_RECIPIENT);
                        modifiedProperties.add(Email.PROPERTY_MAIL_FOLDER);
                        break;
                    }
                }
                if (email.getMailFolder() == null) {
                    continue;
                }
                emailService.saveEmail(email, null);

                Address[] addresses = message.getFrom();
                if (addresses != null && addresses.length > 0) {
                    String sender = addresses[0].toString();
                    email.setSender(sender);
                    modifiedProperties.add(Email.PROPERTY_SENDER);

                    Client client = serviceContext.newService(ClientService.class)
                            .getClientForEmailAddress(sender, email);

                    if (client != null) {
                        String object = client.getCaracteristic1() + " / " +
                                client.getCode() + " / " +
                                client.getName() + " / " +
                                client.getFaxNumber() + " / " +
                                DateFormat.getDateInstance(DateFormat.MEDIUM).format(new Date());
                        email.setObject(object);
                        modifiedProperties.add(Email.PROPERTY_OBJECT);
                        modifiedProperties.add(Email.PROPERTY_CLIENT);

                        //TODO kmorin 20140602 find the responsible's folder
//                        client.getCustomerResponsible()
                    }
                    email.setClient(client);
                }

                Date receivedDate = message.getReceivedDate();
                if (receivedDate == null) {
                    receivedDate = message.getSentDate();
                }
                email.setReceptionDate(receivedDate);
                modifiedProperties.add(Email.PROPERTY_RECEPTION_DATE);

                Date now = new Date();

                DecoratorService decoratorService = serviceContext.newService(DecoratorService.class);
                Decorator<Date> dateDecorator = decoratorService.getDecoratorByType(Date.class, DecoratorService.DATE);
                String projectRef = t("faxtomail.email.projectReference.default", dateDecorator.toString(now));
                email.setProjectReference(projectRef);
                modifiedProperties.add(Email.PROPERTY_PROJECT_REFERENCE);

                email.setDemandStatus(DemandStatus.UNTREATED);
                modifiedProperties.add(Email.PROPERTY_DEMAND_STATUS);

                StringBuilder emailSource = new StringBuilder();
                Enumeration<String> headerLines = ((MimeMessage)message).getAllHeaderLines();
                while (headerLines.hasMoreElements()) {
                    String headerLine = headerLines.nextElement();
                    emailSource.append(headerLine).append("\n");
                }
                String originalContent = IOUtils.toString(message.getInputStream(), charset);
                emailSource.append("\n").append(originalContent);
                email.setOriginalEmail(emailSource.toString());

                List<Attachment> attachements = new ArrayList<>();
                if (message.isMimeType("multipart/*")) {
                    decomposeMultipartEmail(attachements, message, email, emailService);
//                } else {
//                    String content = IOUtils.toString(message.getInputStream(), charset);
//                    email.setPlainContent(content);
                }

                emailService.saveEmail(email,
                        attachements,
                        email.getClient() != null ? email.getClient().getCode() : null,
                        null,
                        modifiedProperties.toArray(new String[modifiedProperties.size()]));

                message.setFlag(Flags.Flag.DELETED, true);
            }
            log.debug("End of emails");

        } catch (Exception e) {
            log.error("Error while reading the emails", e);
        }
    }

    /**
     * Decompose a multipart part.
     * - sets the email content if the part contains a text bodypart
     * - adds attachments to the email
     *
     * @param part the part to decompose
     * @param email the email object to fill with the attachments or content
     * @param emailService email service
     * @throws Exception
     */
    protected void decomposeMultipartEmail(List<Attachment> attachments, Part part, Email email, EmailService emailService) throws Exception {
        DataSource dataSource = part.getDataHandler().getDataSource();
        MimeMultipart mimeMultipart = new MimeMultipart(dataSource);
        int multiPartCount = mimeMultipart.getCount();

        
        for (int j = 0; j < multiPartCount; j++) {
            BodyPart bp = mimeMultipart.getBodyPart(j);
            Charset charset = getCharset(bp);

            // if it is a text part, the,n this is the email content
            String disposition = bp.getDisposition();
            if (bp.isMimeType("text/*") && !Part.ATTACHMENT.equals(disposition)) {
//                String content = IOUtils.toString(bp.getInputStream(), charset);
//                if (bp.isMimeType("text/plain")) {
//                    email.setPlainContent(content);
//                } else {
//                    email.setHtmlContent(content);
//                }

            // if it is multipart part, decompose it
            } else if (bp.isMimeType("multipart/*")) {
                decomposeMultipartEmail(attachments, bp, email, emailService);

            // else, this is an attachment
            } else {
                String fileName = bp.getFileName();
                if (fileName == null) {
                    String[] headers = bp.getHeader("Content-ID");
                    if (headers == null) {
                        break;
                    }
                    fileName = headers[0];
                    // remove the guillemets between the id
                    fileName = fileName.replaceFirst("^<(.*)>$", "$1");
                }

                try {
                    fileName = MimeUtility.decodeText(fileName);
                    log.debug("FileName : " + fileName);
                } catch (UnsupportedEncodingException ex) {
                    // don't care, use filename raw value
                    if (log.isWarnEnabled()) {
                        log.warn("Can't debug email file name", ex);
                    }
                }

                File dir = new File(FileUtils.getTempDirectory(), email.getTopiaId());
                if (!dir.exists()) {
                    dir.mkdir();
                }
                File file = new File(dir, fileName);
                FileOutputStream fos = new FileOutputStream(file);

                DataHandler dh = bp.getDataHandler();
                dh.writeTo(fos);

                Attachment attachment = new AttachmentImpl();
                attachment.setAddedByUser(false);
                AttachmentFile attachmentFile = emailService.getAttachmentFileFromStream(dh.getInputStream());
                attachmentFile.setFilename(fileName);
                attachment.setOriginalFile(attachmentFile);

                //email.addAttachment(attachment);
                attachments.add(attachment);
            }
        }
    }

    protected Charset getCharset(Part part) throws MessagingException {
        ContentType contentType = new ContentType(part.getContentType());
        String charsetName = contentType.getParameter("charset");
        Charset charset = Charsets.toCharset(charsetName);
        return charset;
    }
}
