package com.franciaflex.faxtomail.services.service;

/*
 * #%L
 * FaxToMail :: Service
 * $Id: EmailServiceImpl.java 666 2014-10-03 10:33:08Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-1.0/faxtomail-service/src/main/java/com/franciaflex/faxtomail/services/service/EmailServiceImpl.java $
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;

import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.MessagingException;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.mail.DefaultAuthenticator;
import org.apache.commons.mail.EmailConstants;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.MultiPartEmail;
import org.hibernate.Hibernate;
import org.nuiton.csv.Import;
import org.nuiton.csv.ImportRuntimeException;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.ApplicationTechnicalException;
import org.nuiton.topia.persistence.TopiaEntities;
import org.nuiton.topia.persistence.support.TopiaSqlSupport;
import org.nuiton.topia.persistence.support.TopiaSqlWork;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;
import org.nuiton.util.pagination.PaginationParameter;
import org.nuiton.util.pagination.PaginationResult;
import org.xhtmlrenderer.pdf.ITextRenderer;

import com.franciaflex.faxtomail.persistence.entities.*;
import com.franciaflex.faxtomail.services.FaxToMailServiceSupport;
import com.franciaflex.faxtomail.services.FaxToMailServiceUtils;
import com.franciaflex.faxtomail.services.service.exceptions.AlreadyLockedMailException;
import com.franciaflex.faxtomail.services.service.exceptions.FolderNotReadableException;
import com.franciaflex.faxtomail.services.service.exceptions.InvalidClientException;
import com.franciaflex.faxtomail.services.service.imports.ArchiveImportBean;
import com.franciaflex.faxtomail.services.service.imports.ArchiveImportModel;
import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.github.mustachejava.MustacheFactory;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * @author kmorin - kmorin@codelutin.com
 */
public class EmailServiceImpl extends FaxToMailServiceSupport implements EmailService {

    private static final Log log = LogFactory.getLog(EmailServiceImpl.class);

    protected Binder<Attachment, Attachment> attachmentBinder =
            BinderFactory.newBinder(Attachment.class, Attachment.class);

    @Override
    public Email getEmailById(String id) {
        Email email = getPersistenceContext().getEmailDao().forTopiaIdEquals(id).findUnique();
        return email;
    }

    protected Email getEmailById(String id, String fetch, String... otherFetches) {
        Email email = getPersistenceContext().getEmailDao().forTopiaIdEquals(id).addAllFetches(fetch, otherFetches).findUnique();
        return email;
    }

    @Override
    public Email getFullEmailById(String id) {

        Email email = getPersistenceContext().getEmailDao()
                .forTopiaIdEquals(id)
                .addAllFetches(Email.PROPERTY_DEMAND_TYPE,
                               //Email.PROPERTY_RANGE_ROW + "." + RangeRow.PROPERTY_RANGE,
                               //Email.PROPERTY_HISTORY + "." + History.PROPERTY_FAX_TO_MAIL_USER,
                               //Email.PROPERTY_REPLIES,
                               //Email.PROPERTY_ATTACHMENT,
                               Email.PROPERTY_TAKEN_BY,
                               Email.PROPERTY_CLIENT,
                               //Email.PROPERTY_EMAIL_GROUP + "." + EmailGroup.PROPERTY_EMAIL,
                               //Email.PROPERTY_EMAIL_GROUP + "." + EmailGroup.PROPERTY_EMAIL + "." + Email.PROPERTY_MAIL_FOLDER,
                               Email.PROPERTY_WAITING_STATE,
                               Email.PROPERTY_PRIORITY).findUnique();

        // manual fetch
        //Hibernate.initialize(email.getPriority());
        //Hibernate.initialize(email.getDemandType());
        //Hibernate.initialize(email.getClient());
        List<RangeRow> rangeRows = email.getRangeRow();
        if (rangeRows != null) {
            for (RangeRow rangeRow : rangeRows) {
                Hibernate.initialize(rangeRow.getRange());
            }
        }
        //Hibernate.initialize(email.getWaitingState()));
        //Hibernate.initialize(email.getTakenBy());
        Hibernate.initialize(email.getReplies());
        Hibernate.initialize(email.getAttachment());
        List<History> histories = email.getHistory();
        if (histories != null) {
            for (History history : histories) {
                Hibernate.initialize(history.getFaxToMailUser());
            }
        }
        EmailGroup emailGroup = email.getEmailGroup();
        if (emailGroup != null) {
            Collection<Email> emails = emailGroup.getEmail();
            for (Email email2 : emails) {
                Hibernate.initialize(email2.getMailFolder());
            }
        }

        return email;
    }

    @Override
    public Email getFullEmailById(String id, FaxToMailUser user) {
        Email email = getFullEmailById(id);

        MailFolderService mailFolderService = serviceContext.getMailFolderService();
        MailFolder folder = email.getMailFolder();
        browseFolderParent(folder, user, mailFolderService);

        return email;
    }

    protected void browseFolderParent(MailFolder folder, FaxToMailUser user, MailFolderService mailFolderService) {
        Hibernate.initialize(folder);
        Hibernate.initialize(folder.getChildren());
        mailFolderService.fetchFolderAttributes(folder);

        MailFolder parent = folder.getParent();
        if (parent != null) {
            browseFolderParent(parent, user, mailFolderService);
        }

        boolean writable = parent != null && parent.isFolderWritable();
        if (folder.getWriteRightGroups() != null && user.getUserGroups() != null) {
            writable |= CollectionUtils.containsAny(folder.getWriteRightGroups(), user.getUserGroups());
        }
        writable |= folder.containsWriteRightUsers(user);

        boolean readable = parent != null && parent.isFolderReadable();
        if (folder.getReadRightGroups() != null && user.getUserGroups() != null) {
            readable |= CollectionUtils.containsAny(folder.getReadRightGroups(), user.getUserGroups());
        }
        readable |= folder.containsReadRightUsers(user);

        folder.setFolderWritable(writable);
        folder.setFolderReadable(readable);
    }

    /**
     * Save email with default email client code.
     * 
     * @param email email
     * @param user user
     * @param modifiedFields modified fields
     * @return updated email instance
     * @throws InvalidClientException if client code is not valid
     */
    @Override
    public Email saveEmail(Email email, FaxToMailUser user, String... modifiedFields) throws InvalidClientException {
        Client client = email.getClient();
        return saveEmail(email, null, null, client != null ? client.getCode() : null, user, modifiedFields);
    }

    @Override
    public Email saveEmail(Email email, Collection<Attachment> attachments, Collection<Reply> replies, String clientCode,
            FaxToMailUser user, String... modifiedFields) throws InvalidClientException {
        Date now = getNow();
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();

        MailFolder folder = email.getMailFolder();
        while (!folder.isUseCurrentLevelCompany() && folder.getParent() != null) {
            folder = folder.getParent();
        }
        String company = folder.getCompany();

        // if client code is null, do not manage client at all
        Client client = null;
        if (StringUtils.isNotBlank(clientCode)) {
            Client emailClient = email.getClient();
            if (emailClient != null && clientCode.equals(email.getClient().getCode())) {
                client = emailClient;

            } else {
                client = getClientService().getClientForCode(clientCode, company);
                if (client == null) {
                    String message = t("faxtomail.service.email.save.clientCode.error", clientCode);
                    throw new InvalidClientException(message);
                }
            }
        }
        email.setClient(client);

        if (attachments != null) {
            updateAttachments(email, attachments);
        }
        if (replies != null) {
            updateReplies(email);
        }

        if (email.getRangeRow() != null) {
            RangeRowTopiaDao rangeRowDao = getPersistenceContext().getRangeRowDao();
            for (RangeRow rangeRow : email.getRangeRow()) {
                if (!rangeRow.isPersisted()) {
                    rangeRowDao.create(rangeRow);
                } else {
                    rangeRowDao.update(rangeRow);
                }
            }
        }

        if (!email.isPersisted()) {
            if (email.getHistory() != null) {
                historyDao.createAll(email.getHistory());
            }

            email = dao.create(email);
        }

        Set<String> fieldSet = Sets.newHashSet(modifiedFields);

        History transmissionToEdi = CollectionUtils.find(email.getHistory(), new Predicate<History>() {
            @Override
            public boolean evaluate(History object) {
                return object.getType() == HistoryType.TRANSMISSION_TO_EDI;
            }
        });

        // we transmit to EDI if:
        // - it has never been transmitted before
        // - all the required fields are filled
        if (transmissionToEdi == null
                && email.getClient() != null
                && email.getDemandType() != null
                && StringUtils.isNotBlank(email.getProjectReference())) {

            handleEdiTransmission(email, historyDao, fieldSet);
        }

        History history = null;
        if (fieldSet.contains(Email.PROPERTY_ARCHIVE_DATE)) {
            history = historyDao.create(History.PROPERTY_TYPE, HistoryType.ARCHIVED,
                                        History.PROPERTY_FAX_TO_MAIL_USER, user,
                                        History.PROPERTY_MODIFICATION_DATE, now);

        } else if (fieldSet.contains(Email.PROPERTY_MAIL_FOLDER)) {
            history = historyDao.create(History.PROPERTY_TYPE, HistoryType.TRANSMISSION,
                                        History.PROPERTY_FAX_TO_MAIL_USER, user,
                                        History.PROPERTY_MODIFICATION_DATE, now);

        } else {
            if (email.getTakenBy() == null &&
                    !fieldSet.isEmpty() && !fieldSet.contains(Email.PROPERTY_TAKEN_BY)) {
                email.setTakenBy(user);
                fieldSet.add(Email.PROPERTY_TAKEN_BY);
            }

            boolean historyEmpty = email.isHistoryEmpty();
            if (historyEmpty || !fieldSet.isEmpty()) {
                history = historyDao.create(History.PROPERTY_TYPE, historyEmpty ? HistoryType.CREATION : HistoryType.MODIFICATION,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_MODIFICATION_DATE, now);
                history.setFields(fieldSet);
            }
        }
        if (history != null) {
            email.addHistory(history);
        }

        Email result = dao.update(email);
        getPersistenceContext().commit();

        return result;
    }

    /**
     * Les réponses sont modifiées par l'ui pour ne pas charger le contenu binaire, pour la sauvegarde,
     * il faut bien rétablir les replyContent avec ceux en base.
     * 
     * @param email
     */
    protected void updateReplies(Email email) {
        ReplyTopiaDao replyTopiaDao = getPersistenceContext().getReplyDao();

        List<Reply> currentReplies = new ArrayList<Reply>(email.getReplies());
        email.clearReplies();
        for (Reply reply : currentReplies) {
            // normalement elle sont toutes persistée et non modifiées par l'ui
            // donc reprendre la version en base avec un replyContent non null est suffisant dans ce cas
            Reply newReply = replyTopiaDao.forTopiaIdEquals(reply.getTopiaId()).findUnique();
            email.addReplies(newReply);
        }
    }

    /**
     * Les attachment sont copiés dans l'UI sans la valorisation des champs "originalFile" et "editedFile".
     * Ici ont revalorise ces champs pour la sauvegarde.
     * 
     * @param email
     * @param attachments
     */
    protected void updateAttachments(Email email, Collection<Attachment> attachments) {
        AttachmentTopiaDao attachmentTopiaDao = getPersistenceContext().getAttachmentDao();

        List<Attachment> currentAttachments = attachmentTopiaDao.forTopiaIdIn(email.getAttachmentTopiaIds()).findAll();
        if (currentAttachments == null) {
            currentAttachments = new ArrayList<>();
        }
        Map<String, Attachment> currentAttachmentIndex = new HashMap<>(Maps.uniqueIndex(currentAttachments, TopiaEntities.getTopiaIdFunction()));

        AttachmentFileTopiaDao attachementFileTopiaDao = getPersistenceContext().getAttachmentFileDao();
        for (Attachment attachment : attachments) {

            // get session attachment from id
            Attachment currentAttachment;
            if (StringUtils.isNotBlank(attachment.getTopiaId())) {
                currentAttachment = currentAttachmentIndex.remove(attachment.getTopiaId());
            } else {
                currentAttachment = new AttachmentImpl();
            }

            AttachmentFile originalFile = null;
            AttachmentFile editedFile = null;
            // dans tout les cas, les pieces jointes courante de l'attchent prévale
            // sur celles en base
            if (attachment.getOriginalFile() != null) {
                originalFile = attachment.getOriginalFile();
            } else {
                originalFile = currentAttachment.getOriginalFile();
            }
            if (attachment.getEditedFile() != null) {
                editedFile = attachment.getEditedFile();
            } else {
                editedFile = currentAttachment.getEditedFile();
            }

            // copy new data
            attachmentBinder.copyExcluding(attachment, currentAttachment,
                                           Attachment.PROPERTY_EDITED_FILE,
                                           Attachment.PROPERTY_EDITED_FILE_NAME,
                                           Attachment.PROPERTY_ORIGINAL_FILE,
                                           Attachment.PROPERTY_ORIGINAL_FILE_NAME,
                                           Attachment.PROPERTY_TOPIA_CREATE_DATE,
                                           Attachment.PROPERTY_TOPIA_ID,
                                           Attachment.PROPERTY_TOPIA_VERSION);

            currentAttachment.setEditedFile(editedFile);
            currentAttachment.setOriginalFile(originalFile);

            // ici les pieces jointes peuvent être sauvegardées sans que les pièces jointes
            // soit présentes dans les entités car l'ui ne les a pas copiées
            // pour ne pas les charger inutilement
            // donc on les remet manuellement

            if (!originalFile.isPersisted()) {
                attachementFileTopiaDao.create(originalFile);
            } else {
                attachementFileTopiaDao.update(originalFile);
            }

            if (editedFile != null) {
                if (!editedFile.isPersisted()) {
                    attachementFileTopiaDao.create(editedFile);
                } else {
                    attachementFileTopiaDao.update(editedFile);
                }
            }

            if (!currentAttachment.isPersisted()) {
                // persist using cascade
                currentAttachments.add(currentAttachment);
            }
        }

        // delete not found attachments
        for (Attachment attachment : currentAttachmentIndex.values()) {
            currentAttachments.remove(attachment);
        }

        email.setAttachment(currentAttachments);
    }

    protected void handleEdiTransmission(Email email, HistoryTopiaDao historyDao, Set<String> fieldSet) {
        History transmissionToEdi;

        // le transfert EDI est configuré par type de demande ET par société
        if (FaxToMailServiceUtils.contains(email.getDemandType().getRequiredFields(), MailField.RANGE_ROW)
                && email.getDemandType().isEdiTransfer()) {

            // si les gammes sont vides, la demande n'est pas valide, on ne fait rien dans ce cas
            // sinon on transfer à edi
            if (email.isRangeRowNotEmpty()) {

                // recherche parmis les dossiers parent, si la configuration ediTranfer à été demandée
                Boolean ediTranfer = null;
                MailFolder loopFolder = email.getMailFolder();
                do {
                    ediTranfer = loopFolder.getEdiTransfer();
                    loopFolder = loopFolder.getParent();
                } while (ediTranfer == null && loopFolder != null);

                if (BooleanUtils.isTrue(ediTranfer)) {
                    // ajout d'un historique
                    transmissionToEdi = historyDao.create(History.PROPERTY_TYPE, HistoryType.TRANSMISSION_TO_EDI,
                                                          History.PROPERTY_MODIFICATION_DATE, new Date());
                    email.addHistory(transmissionToEdi);

                    // changement du status
                    email.setDemandStatus(DemandStatus.TRANSMISSION_TO_EDI);
                    fieldSet.add(Email.PROPERTY_DEMAND_STATUS);

                } else {
                    // TODO echatellier : pas sur qu'il faille l'ajouter tout le temps, mais sinon, le transfer se reproduira
                    transmissionToEdi = historyDao.create(History.PROPERTY_TYPE, HistoryType.TRANSMISSION_TO_EDI,
                            History.PROPERTY_MODIFICATION_DATE, new Date());
                    email.addHistory(transmissionToEdi);

                    // passage en status
                    email.setDemandStatus(DemandStatus.IN_PROGRESS);
                    fieldSet.add(Email.PROPERTY_DEMAND_STATUS);
                }
            }

        } else {

            // TODO echatellier : pas sur qu'il faille l'ajouter tout le temps, mais sinon, le transfer se reproduira
            transmissionToEdi = historyDao.create(History.PROPERTY_TYPE, HistoryType.TRANSMISSION_TO_EDI,
                    History.PROPERTY_MODIFICATION_DATE, new Date());
            email.addHistory(transmissionToEdi);

            // passage en status
            email.setDemandStatus(DemandStatus.IN_PROGRESS);
            fieldSet.add(Email.PROPERTY_DEMAND_STATUS);
        }
    }

    @Override
    public void transmitPendingDemandsToEdi() {
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();

        List<Email> toTransmitToEdi = dao.forDemandStatusEquals(DemandStatus.TRANSMISSION_TO_EDI).findAll();
        if (log.isDebugEnabled()) {
            log.debug(toTransmitToEdi.size() + " demands to transmit to edi");
        }
        for (Email email : toTransmitToEdi) {
            transmitDemandToEdi(email);
        }
    }

    /**
     * Generate txt file to send demand to EDI system.
     * 
     * @param email email to send
     */
    protected void transmitDemandToEdi(Email email) {
        Preconditions.checkArgument(email.getDemandStatus() == DemandStatus.TRANSMISSION_TO_EDI);
        Preconditions.checkArgument(!email.getRangeRow().isEmpty());
       
        // recherche du dossier ou deposer les demandes EDI
        MailFolder folder = email.getMailFolder();
        while (!folder.isUseCurrentLevelEdiFolder() && folder.getParent() != null) {
            folder = folder.getParent();
        }
        String ediFolder = folder.getEdiFolder();
        if (StringUtils.isBlank(ediFolder)) {
            if (log.isFatalEnabled()) {
                log.fatal("Aucun dossier de depot des demandes EDI défini pour le dossier " + folder.getName());
            }
            return;
        }
        File ediFolderDirectory = new File(ediFolder);
        if (!ediFolderDirectory.canWrite()) {
            if (log.isFatalEnabled()) {
                log.fatal("Le dossier " + ediFolderDirectory.getAbsolutePath() + " ne dispose pas des droits d'écriture !");
            }
            return;
        }

        //TODO kmorin 20140521 maybe check if the file is not being read

        // get output file with pattern name depending on date
        Date now = serviceContext.getNow();
        String fileDate = DateFormatUtils.format(now, "yyMMddHHmmss");

        // create a file for each commande
        for (RangeRow rangeRow : email.getRangeRow()) {

            String fileName = "ORD_FAX_" + fileDate + "_" + rangeRow.getTopiaId() + ".txt";
            
            File ediFile = new File(ediFolderDirectory, fileName);
    
            // generate output content
            String separator = ";";
            try (Writer ediWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(ediFile), StandardCharsets.UTF_8))) {
                
                ediWriter.write("%BEGIN_ENTETE_QUOTE\n");
                
                //N° Champ/Champ/Type/Longueur/Observation
                //01/Id Enregistrement/Alpha/1/Cst : E (En-tête)
                ediWriter.write("E" + separator);
                //02/N° d’ordre achat/Numérique/6/Cst : OARFAX
                ediWriter.write("OARFAX" + separator);
                //03/Date traitement/Date/6/JJMMAA dte système
                ediWriter.write(DateFormatUtils.format(now, "ddMMyy") + separator);
                //04/Date livraison prévue/Date/6/JJMMAA dte de réception
                ediWriter.write(DateFormatUtils.format(email.getReceptionDate(), "ddMMyy") + separator);
                //05/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //06/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //07/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //08/Commentaire/Alpha/70/Vide
                ediWriter.write(separator);
                //09//Référence client FX/Alpha/36/Référence Chantier
                ediWriter.write(Strings.nullToEmpty(email.getProjectReference()) + separator);
                //10/Nom du contact/Alpha/35/Vide
                ediWriter.write(separator);
                //11/Téléphone contact/Alpha/15/Vide
                ediWriter.write(separator);
                //12/Fax contact/Alpha/15/Vide
                ediWriter.write(separator);
                //13/Devise/Alpha/3/Cst : EUR
                ediWriter.write("EUR" + separator);
                //14/Condition de livraison/Alpha/3/Vide
                ediWriter.write(separator);
                //15/Type de commande/Alpha/Type de document
                ediWriter.write(email.getDemandType().getLabel() + separator);
                //16/Adresse livr : Nom/Alpha/35/Vide
                ediWriter.write(separator);
                //18/Adresse livr : adr1/Alpha/30/Vide
                ediWriter.write(separator);
                //19/Adresse livr : adr2/Alpha/30/Vide
                ediWriter.write(separator);
                //20/Adresse livr : cp ville/Alpha/30/Vide
                ediWriter.write(separator);
                //21/Adresse livr : compl./Alpha/30/Vide
                ediWriter.write(separator);
                //22/Adresse livr : Code postal/Alpha/10/Vide
                ediWriter.write(separator);
                //23/Adresse livr : Code Pays/Alpha/3/Vide
                ediWriter.write(separator);
                //24/Code Ean acheteur/Alpha/13/Code client
                ediWriter.write(email.getClient().getCode() + separator);
                //25/Code Ean facturé/Alpha/13/Code client
                ediWriter.write(email.getClient().getCode() + separator);
                //26/Code Ean fournisseur/Alpha/13/Gamme
                ediWriter.write(rangeRow.getRange().getLabel() + separator);
                //27/Identifiant Unique
                ediWriter.write(rangeRow.getTopiaId() + "\n");

                ediWriter.write("%END_ENTETE_QUOTE\n");

            } catch (IOException ex) {
                throw new RuntimeException("Can't generate EDI file");
            }

        }

        // changement du status
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        email.setDemandStatus(DemandStatus.TRANSMITTED_TO_EDI);
        dao.update(email);
    }

    @Override
    public Set<Object> getDistinctValues(MailFolder folder, String[] properties, boolean sum) {
        Set<Object> result = null;
        if (folder.isFolderReadable()) {
            EmailTopiaDao dao = getPersistenceContext().getEmailDao();
            result = dao.getDistinctValues(folder, properties, sum);
        }
        return result;
    }

    /**
     * Recupère les demandes d'un dossier visible par un utilisateur.
     * 
     * La methode et les résultats sont paginés.
     * 
     * @param folder folder to get demande
     * @param currentUser user to check rigth
     * @param page pagination
     * @return paginated results
     */
    @Override
    public PaginationResult<Email> getEmailForFolder(MailFolder folder,
                                                     FaxToMailUser currentUser,
                                                     EmailFilter filter,
                                                     PaginationParameter page) {

        // perform request or not depending on rigths
        PaginationResult<Email> result;
        if (folder.isFolderReadable()) {
            EmailTopiaDao dao = getPersistenceContext().getEmailDao();

            result = dao.getEmailForFolder(filter, folder, page);
//            TopiaQueryBuilderAddCriteriaOrRunQueryStep<Email> builderAddCriteriaOrRunQueryStep =
//                    dao.forMailFolderEquals(folder)
//                        .addNull(Email.PROPERTY_ARCHIVE_DATE)
//                        .addNotEquals(Email.PROPERTY_DEMAND_STATUS, DemandStatus.ARCHIVED);
//
//            if (CollectionUtils.isNotEmpty(filter.getDemandStatus())) {
//                builderAddCriteriaOrRunQueryStep.addIn(Email.PROPERTY_DEMAND_STATUS, filter.getDemandStatus());
//            }
//            if (CollectionUtils.isNotEmpty(filter.getSenders())) {
//                builderAddCriteriaOrRunQueryStep.addIn(Email.PROPERTY_SENDER, filter.getSenders());
//            }
//
//            if (Boolean.getBoolean("faxtomail.fullfetch")) {
//                result = builderAddCriteriaOrRunQueryStep
//                        .addAllFetches(Email.PROPERTY_PRIORITY,
//                                       Email.PROPERTY_DEMAND_TYPE,
//                                       Email.PROPERTY_CLIENT,
//                                       Email.PROPERTY_RANGE_ROW + "." + RangeRow.PROPERTY_RANGE,
//                                       Email.PROPERTY_WAITING_STATE,
//                                       Email.PROPERTY_TAKEN_BY,
//                                       Email.PROPERTY_REPLIES,
//                                       Email.PROPERTY_ATTACHMENT
//                                       //Email.PROPERTY_HISTORY + "." + History.PROPERTY_FAX_TO_MAIL_USER
//                        ).findPage(page);
//            } else {
//                result = builderAddCriteriaOrRunQueryStep
//                                     .addAllFetches(Email.PROPERTY_PRIORITY,
//                                                    Email.PROPERTY_DEMAND_TYPE,
//                                                    Email.PROPERTY_CLIENT,
//                                                    //Email.PROPERTY_RANGE_ROW + "." + RangeRow.PROPERTY_RANGE,
//                                                    Email.PROPERTY_WAITING_STATE,
//                                                    Email.PROPERTY_TAKEN_BY
//                                                    //Email.PROPERTY_REPLIES,
//                                                    //Email.PROPERTY_ATTACHMENT,
//                                                    //Email.PROPERTY_HISTORY + "." + History.PROPERTY_FAX_TO_MAIL_USER
//                                     ).findPage(page);
//
//
//                // manual fetch
//                for (Email email : result.getElements()) {
//                    //Hibernate.initialize(email.getPriority());
//                    //Hibernate.initialize(email.getDemandType());
//                    //Hibernate.initialize(email.getClient());
//                    List<RangeRow> rangeRows = email.getRangeRow();
//                    if (rangeRows != null) {
//                        for (RangeRow rangeRow : rangeRows) {
//                            Hibernate.initialize(rangeRow.getRange());
//                        }
//                    }
//                    //Hibernate.initialize(email.getWaitingState());
//                    //Hibernate.initialize(email.getTakenBy());
//                    Hibernate.initialize(email.getReplies());
//                    Hibernate.initialize(email.getAttachment());
//                    /*List<History> histories = email.getHistory();
//                    if (histories != null) {
//                        for (History history : histories) {
//                            Hibernate.initialize(history.getFaxToMailUser());
//                        }
//                    }*/
//                }
//            }

        } else {
            List<Email> elements = Collections.emptyList();
            result = PaginationResult.of(elements, 0, page);
        }
        return result;
    }

    @Override
    public List<MailFolder> getChildrenRecursively(MailFolder folder) {
        List<MailFolder> folders = new ArrayList<>();
        folders.add(folder);
        Collection<MailFolder> children = folder.getChildren();
        if (children != null) {
            for (MailFolder child : children) {
                folders.addAll(getChildrenRecursively(child));
            }
        }
        return folders;
    }

    @Override
    public Map<Range, Long[]> computeQuantitiesByRange(MailFolder rootFolder) {
        // get results by topia id
        List<MailFolder> folders = getChildrenRecursively(rootFolder);
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        Map<String, Long[]> daoResult = emailDao.computeQuantitiesByRange(folders);
        
        // convert topiaId to entities
        RangeTopiaDao rangeDao = getPersistenceContext().getRangeDao();
        Map<Range, Long[]> result = new TreeMap<Range, Long[]>(new Comparator<Range>() {
            @Override
            public int compare(Range o1, Range o2) {
                return o1.getLabel().compareTo(o2.getLabel());
            }
        });
        for (Entry<String, Long[]> entry : daoResult.entrySet()) {
            Range range = rangeDao.forTopiaIdEquals(entry.getKey()).findUnique();
            result.put(range, entry.getValue());
        }
        
        return result;
    }

    @Override
    public Email addToHistory(String emailId, HistoryType type, FaxToMailUser user, Date date, String... fields) {
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        Email email = getEmailById(emailId,
                                   Email.PROPERTY_HISTORY + "." + History.PROPERTY_FAX_TO_MAIL_USER);

        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        History history = historyDao.create(History.PROPERTY_TYPE, type,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_MODIFICATION_DATE, date);
        history.setFields(Sets.newHashSet(fields));

        email.addHistory(history);

        // pour eviter de charger tout l'historique dans la liste, on duplique l'information de
        // dernier ouvreur de PJ directement dans l'email
        if (type == HistoryType.ATTACHMENT_OPENING) {
            email.setLastAttachmentOpener(user);
        }

        email = emailDao.update(email);
        getPersistenceContext().commit();

        return email;
    }

    @Override
    public Email openEmail(String emailId, FaxToMailUser user, boolean takeEmail) throws FolderNotReadableException {
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        Email email = getFullEmailById(emailId, user);

        MailFolder mailFolder = email.getMailFolder();
        if (!mailFolder.isFolderReadable()) {
            throw new FolderNotReadableException(String.format("Mail folder %s not readable by %s",
                                                               mailFolder.getName(),
                                                               user.getLogin()),
                                                 mailFolder);
        }

        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        History history = historyDao.create(History.PROPERTY_TYPE, HistoryType.OPENING,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_MODIFICATION_DATE, new Date());
        email.addHistory(history);
        if (takeEmail) {
            history = historyDao.create(History.PROPERTY_TYPE, HistoryType.MODIFICATION,
                                        History.PROPERTY_FAX_TO_MAIL_USER, user,
                                        History.PROPERTY_MODIFICATION_DATE, new Date());
            history.setFields(Sets.newHashSet(Email.PROPERTY_TAKEN_BY));
            email.addHistory(history);
            email.setTakenBy(user);
        }

        email = emailDao.update(email);
        getPersistenceContext().commit();
        return email;
    }

    /**
     * Vérrouille une demande par un utilisateur empechant les autres de l'ouvrir.
     * 
     * @param emailId topiaId de la demande à vérouiller
     * @param currentUser user
     * @return email
     * @throws AlreadyLockedMailException if email is already locked by another user
     */
    @Override
    public Email lockEmail(String emailId, FaxToMailUser currentUser) throws AlreadyLockedMailException,
                                                                                FolderNotReadableException {

        // get current lock on mail if any
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        MailLockTopiaDao mailLockDao = getPersistenceContext().getMailLockDao();
        Email email = emailDao.forTopiaIdEquals(emailId).findUnique();

        MailFolderService mailFolderService = serviceContext.getMailFolderService();
        MailFolder folder = email.getMailFolder();
        browseFolderParent(folder, currentUser, mailFolderService);

        if (!folder.isFolderReadable()) {
            throw new FolderNotReadableException(String.format("Mail folder %s not readable by %s",
                                                               folder.getName(),
                                                               currentUser.getLogin()),
                                                 folder);
        }

        Hibernate.initialize(email.getTakenBy());

        MailLock mailLock = mailLockDao.forLockOnEquals(email).findUniqueOrNull();

        // if no lock found, create new one
        if (mailLock == null) {

            // dévérouillage automatique des mails qui ne font pas partit du même group que celui du mail
            // qui vient d'être locké
            List<MailLock> mailLocksToRemove;
            if (email.getEmailGroup() != null) {
                // si le mail courant à un group, ont déverrouilles tous les mails
                // qui ne porte pas sur le même groupe
                mailLocksToRemove = mailLockDao.forLockByEquals(currentUser)
                        .addNotEquals(MailLock.PROPERTY_LOCK_ON + "." + Email.PROPERTY_EMAIL_GROUP, email.getEmailGroup()).findAll();
            } else {
                mailLocksToRemove = mailLockDao.forLockByEquals(currentUser).findAll();
            }
            if (log.isDebugEnabled()) {
                for (MailLock mailLockToRemove : mailLocksToRemove) {
                    log.debug("[UNLOCK] " + mailLockToRemove.getLockOn().getTopiaId() + " unlocked (automatic)");
                }
            }
            mailLockDao.deleteAll(mailLocksToRemove);

            // ajout d'un nouveau lock
            mailLock = new MailLockImpl();
            mailLock.setLockBy(currentUser);
            mailLock.setLockOn(email);
            mailLock = mailLockDao.create(mailLock);

            if (log.isDebugEnabled()) {
                log.debug("[LOCK] " + emailId + " locked by " + currentUser.getLogin());
            }

            getPersistenceContext().commit();

        } else if (!mailLock.getLockBy().equals(currentUser)) {
            // throw exception if already locked by another user
            throw new AlreadyLockedMailException(String.format("Mail %s already locked by %s", emailId, mailLock.getLockBy().getTopiaId()), mailLock.getLockBy());
        }

        return email;
    }

    /**
     * Dévérrouille une demande.
     * 
     * @param emailId topiaId de la demande à devérouiller
     */
    @Override
    public void unlockEmail(String emailId) {
        MailLockTopiaDao mailLockDao = getPersistenceContext().getMailLockDao();
        MailLock mailLock = mailLockDao.forAll().addEquals(MailLock.PROPERTY_LOCK_ON + "." + Email.PROPERTY_TOPIA_ID, emailId).findUniqueOrNull();
        // ca peut être null si c'est un emailId qui est valorisé suite à une creation et donc
        // qui n'a pas été vérrouillé avant
        if (mailLock != null) {
            mailLockDao.delete(mailLock);
            if (log.isDebugEnabled()) {
                log.debug("[UNLOCK] " + emailId + " unlocked");
            }
            getPersistenceContext().commit();
        }
    }

    /**
     * Calcule recursivement l'ensemble des répertoires lisible par un utilisateur récursivement.
     * 
     * @param readMailFolders result mail folders
     * @param mailFolders mail folder to allow
     */
    protected void computeUserReadableFolder(Set<MailFolder> readMailFolders, Iterable<MailFolder> mailFolders) {
        if (mailFolders != null) {
            for (MailFolder mailFolder : mailFolders) {
                readMailFolders.add(mailFolder);
                computeUserReadableFolder(readMailFolders, mailFolder.getChildren());
            }
        }
    }

    @Override
    public PaginationResult<Email> search(SearchFilter emailFilter, FaxToMailUser user, PaginationParameter pagination) {

        // compute rigths
        MailFolderTopiaDao mailFolderDao = getPersistenceContext().getMailFolderDao();
        Set<MailFolder> readMailFolders = new HashSet<MailFolder>();
        // read rights for user
        Iterable<MailFolder> mailFolders = mailFolderDao.forReadRightUsersContains(user).findAll();
        computeUserReadableFolder(readMailFolders, mailFolders);
        // read rigths for groups
        if (user.getUserGroups() != null) {
            for (FaxToMailUserGroup group : user.getUserGroups()) {
                mailFolders = mailFolderDao.forReadRightGroupsContains(group).findAll();
                computeUserReadableFolder(readMailFolders, mailFolders);
            }
        }

        // compute search
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        PaginationResult<Email> result = emailDao.search(emailFilter, readMailFolders, pagination);
        return result;
    }

    @Override
    public Email groupEmails(String email1Id, String email2Id, FaxToMailUser user) {
        EmailGroupTopiaDao groupDao = getPersistenceContext().getEmailGroupDao();

        Email email1 = getFullEmailById(email1Id, user);
        Email email2 = getEmailById(email2Id);

        EmailGroup group1 = email1.getEmailGroup();
        EmailGroup group2 = email2.getEmailGroup();

        // if both groups are null
        if (group1 == null && group2 == null) {
            EmailGroup group = groupDao.create(EmailGroup.PROPERTY_EMAIL, Sets.newHashSet(email1, email2));
            email1.setEmailGroup(group);
            email2.setEmailGroup(group);

        // if only group 1 is null
        } else if (group1 == null) {
            email1.setEmailGroup(group2);
            group2.addEmail(email1);
            groupDao.update(group2);

        // if only group 2 is null
        } else if (group2 == null) {
            email2.setEmailGroup(group1);
            group1.addEmail(email2);
            groupDao.update(group1);

        // if the groups are equals, do nothing
        } else if (group1.equals(group2)) {
           return email1;

        // if both groups exist, merge them
        } else {
            group1.addAllEmail(group2.getEmail());
            email2.setEmailGroup(group1);
            group2.clearEmail();
            groupDao.delete(group2);
            groupDao.update(group1);
        }

        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        Date now = new Date();

        History history = historyDao.create(History.PROPERTY_TYPE, HistoryType.GROUP,
                History.PROPERTY_FAX_TO_MAIL_USER, user,
                History.PROPERTY_MODIFICATION_DATE, now);
        history.setFields(Sets.newHashSet(email2.getObject()));
        email1.addHistory(history);

        Email result = emailDao.update(email1);

        History history2 = historyDao.create(History.PROPERTY_TYPE, HistoryType.GROUP,
                History.PROPERTY_FAX_TO_MAIL_USER, user,
                History.PROPERTY_MODIFICATION_DATE, now);
        history2.setFields(Sets.newHashSet(email1.getObject()));
        email2.addHistory(history2);
        emailDao.update(email2);

        getPersistenceContext().commit();

        return result;
    }

    /**
     * Envoi une réponse et retourne l'email.
     * 
     * @param from from
     * @param to to
     * @param cc cc
     * @param bcc bcc
     * @param subject subject
     * @param content content
     * @param attachments attachement
     * @param originalEmailId mail topia id
     * @param user user to add new history entry for user
     * @return modified email
     * @throws EmailException if message can't be sent
     * @throws MessagingException if message can't be sent
     * @throws IOException if message can't be sent
     */
    @Override
    public Email reply(String from, String to, String cc, String bcc, String subject,
                       String content, Collection<AttachmentFile> attachments,
                       String originalEmailId, FaxToMailUser user) throws EmailException, MessagingException, IOException {

        Email email = getEmailById(originalEmailId,
                                   Email.PROPERTY_HISTORY + "." + History.PROPERTY_FAX_TO_MAIL_USER,
                                   Email.PROPERTY_REPLIES);

        final String smtpUser = getApplicationConfig().getSmtpUser();
        final String password = getApplicationConfig().getSmtpPassword();
        final boolean useSsl = getApplicationConfig().isSmtpUseSsl();

        MultiPartEmail message = new MultiPartEmail();
        message.setHostName(getApplicationConfig().getSmtpHost());
        message.setSmtpPort(getApplicationConfig().getSmtpPort());
        if (StringUtils.isNotBlank(smtpUser) && password != null) {
            message.setAuthenticator(new DefaultAuthenticator(smtpUser, password));
        }
        message.setSSLOnConnect(useSsl);

        message.setCharset(EmailConstants.UTF_8);
        message.setFrom(from);
        message.addTo(to);
        if (StringUtils.isNotBlank(cc)) {
            message.addCc(cc);
        }
        if (StringUtils.isNotBlank(bcc)) {
            message.addBcc(bcc);
        }
        message.setSubject(subject);
        message.setMsg(content);

        for (AttachmentFile attachmentFile : attachments) {
            // Create the attachment
            File file = attachmentFile.getFile();
            DataSource source = new FileDataSource(file);
            message.attach(source, attachmentFile.getFilename(), null);
        }

        message.send();

        ReplyTopiaDao replyTopiaDao = getPersistenceContext().getReplyDao();
        ReplyContentTopiaDao replyContentTopiaDao = getPersistenceContext().getReplyContentDao();
        Date now = new Date();

        // build mail header part
        StringBuilder emailHeaders = new StringBuilder();
        Enumeration<String> headerLines = message.getMimeMessage().getAllHeaderLines();
        while (headerLines.hasMoreElements()) {
            String headerLine = headerLines.nextElement();
            emailHeaders.append(headerLine).append("\n");
        }
        emailHeaders.append("\n"); // blank line (IMPORTANT)
        
        // build mail serialization part
        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        IOUtils.write(emailHeaders, byteOut);
        IOUtils.copy(message.getMimeMessage().getInputStream(), byteOut);

        ReplyContent replyContent = replyContentTopiaDao.createByNotNull(byteOut.toByteArray());

        Reply reply = replyTopiaDao.create(Reply.PROPERTY_REPLY_CONTENT, replyContent,
                                           Reply.PROPERTY_SENT_DATE, now,
                                           Reply.PROPERTY_SUBJECT, subject,
                                           Reply.PROPERTY_SENDER, from,
                                           Reply.PROPERTY_RECIPIENT, to,
                                           Reply.PROPERTY_SENT_BY, user);

        email.addReplies(reply);

        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();
        History history = historyDao.create(History.PROPERTY_TYPE, HistoryType.REPLY,
                                            History.PROPERTY_FAX_TO_MAIL_USER, user,
                                            History.PROPERTY_MODIFICATION_DATE, now);
        email.addHistory(history);

        email = saveEmail(email, user);

        return email;
    }

    @Override
    public void transmit(Collection<String> emailIds, MailFolder newFolder, FaxToMailUser currentUser) {
        EmailTopiaDao dao = getPersistenceContext().getEmailDao();
        HistoryTopiaDao historyDao = getPersistenceContext().getHistoryDao();

        List<Email> emails = dao.forTopiaIdIn(emailIds).findAll();
        for (Email email : emails) {
            email.setDemandStatus(DemandStatus.QUALIFIED);
            email.setMailFolder(newFolder);
            email.setTakenBy(null);

            History history = historyDao.create(History.PROPERTY_TYPE, HistoryType.TRANSMISSION,
                                                History.PROPERTY_FAX_TO_MAIL_USER, currentUser,
                                                History.PROPERTY_MODIFICATION_DATE, new Date());

            email.addHistory(history);
        }
        dao.updateAll(emails);
        getPersistenceContext().commit();
    }

    /**
     * Save content stream into attachment file content.
     * 
     * @param contentStream content
     * @return attachmentFile filled by content
     */
    @Override
    public AttachmentFile getAttachmentFileFromStream(InputStream contentStream) {
        AttachmentFile attachmentFile = new AttachmentFileImpl();
        try {
            //Session hibernateSession = getPersistenceContext().getHibernateSupport().getHibernateSession();
            //Blob contentBlob = Hibernate.getLobCreator(hibernateSession).createBlob(contentStream, contentStream.available());
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            IOUtils.copy(contentStream, output);
            attachmentFile.setContent(output.toByteArray());
        } catch (Exception ex) {
            throw new RuntimeException("Can't save content", ex);
        }
        return attachmentFile;
    }

    @Override
    public AttachmentFile getAttachmentFile(String attachmentId, boolean original) {
        AttachmentTopiaDao dao = getPersistenceContext().getAttachmentDao();
        Attachment attachment = dao.forTopiaIdEquals(attachmentId).findUnique();
        AttachmentFile result = original ? attachment.getOriginalFile() : attachment.getEditedFile();
        if (result != null) {
            // force lazy initialize
            Hibernate.initialize(result);
        }
        return result;
    }

    @Override
    public ReplyContent getReplyContent(String replyId) {
        ReplyTopiaDao dao = getPersistenceContext().getReplyDao();
        Reply reply = dao.forTopiaIdEquals(replyId).findUnique();
        ReplyContent result = reply.getReplyContent();
        // force lazy initialize
        Hibernate.initialize(result);
        return result;
    }

    /**
     * Retourne la liste des vérrouillages actifs.
     * 
     * @return email list
     */
    @Override
    public List<MailLock> getAllMailLocks() {
        MailLockTopiaDao mailLockDao = getPersistenceContext().getMailLockDao();
        List<MailLock> result = mailLockDao.findAll();
        return result;
    }

    /**
     * Dévérrouille les mails specifiés.
     * 
     * @param mailLockIds mail lock ids to unlock
     */
    @Override
    public void unlockMails(List<String> mailLockIds) {
        MailLockTopiaDao mailLockDao = getPersistenceContext().getMailLockDao();
        Collection<MailLock> mailLocks = mailLockDao.forTopiaIdIn(mailLockIds).findAll();
        mailLockDao.deleteAll(mailLocks);

        getPersistenceContext().commit();
    }

    @Override
    public void rejectEmail(String from, String to, String subject, String content)
            throws EmailException, MessagingException, IOException {

        final String smtpUser = getApplicationConfig().getSmtpUser();
        final String password = getApplicationConfig().getSmtpPassword();
        final boolean useSsl = getApplicationConfig().isSmtpUseSsl();

        MultiPartEmail message = new MultiPartEmail();
        message.setHostName(getApplicationConfig().getSmtpHost());
        message.setSmtpPort(getApplicationConfig().getSmtpPort());
        if (StringUtils.isNotBlank(smtpUser) && password != null) {
            message.setAuthenticator(new DefaultAuthenticator(smtpUser, password));
        }
        message.setSSLOnConnect(useSsl);

        message.setCharset(EmailConstants.UTF_8);
        message.setFrom(from);
        message.addTo(to);
        message.setSubject(subject);
        message.setMsg(content);

        message.send();
    }

    @Override
    public void updateRangeRowsWithEdiReturns() {
        final RangeRowTopiaDao rangeRowTopiaDao = getPersistenceContext().getRangeRowDao();
        final EmailTopiaDao emailTopiaDao = getPersistenceContext().getEmailDao();

        final Set<Email> emailsToUpdate = new HashSet<>();

        TopiaSqlSupport sqlSupport = getPersistenceContext().getSqlSupport();
        sqlSupport.doSqlWork(new TopiaSqlWork() {

            @Override
            public void execute(Connection connection) throws SQLException {

                int importedCount = 0;

                // attention, le sql à pour cible postgresql, mssql, h2, il doit rester simple
                String query = String.format("SELECT %s, %s, %s FROM %s",
                        EdiReturn.PROPERTY_RANGE_ROW_TOPIA_ID,
                        EdiReturn.PROPERTY_COMMAND_NUMBER,
                        EdiReturn.PROPERTY_ERROR,
                        EdiReturn.class.getSimpleName());
                Statement stat = connection.createStatement();

                // execute query
                ResultSet resultSet = stat.executeQuery(query);
                while (resultSet.next()) {

                    String rangeRowTopiaId = resultSet.getString(EdiReturn.PROPERTY_RANGE_ROW_TOPIA_ID);
                    String commandNumber = resultSet.getString(EdiReturn.PROPERTY_COMMAND_NUMBER);
                    String error = resultSet.getString(EdiReturn.PROPERTY_ERROR);

                    // manage email and range row
                    RangeRow rangeRow = rangeRowTopiaDao.forTopiaIdEquals(rangeRowTopiaId).findUniqueOrNull();
                    if (rangeRow == null) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't find rangeRow " + rangeRowTopiaId + " to update");
                        }
                        continue;
                    }
                    Email email = emailTopiaDao.forRangeRowContains(rangeRow).findUniqueOrNull();
                    if (email == null) {
                        if (log.isErrorEnabled()) {
                            log.error("Can't find email for " + rangeRowTopiaId + " to update");
                        }
                        continue;
                    }

                    if (StringUtils.isNotBlank(error)) {
                        //error
                        String oldError = email.getEdiError();
                        if (StringUtils.isNotBlank(oldError)) {
                            error = oldError + ", " + error;
                        }
                        email.setEdiError(error);

                    } else {
                        rangeRow.setCommandNumber(commandNumber);
                        rangeRowTopiaDao.update(rangeRow);
                    }

                    email.setDemandStatus(DemandStatus.IN_PROGRESS);
                    emailsToUpdate.add(email);
                    
                    importedCount++;
                }
                
                // delete all rows
                stat = connection.createStatement();
                stat.execute("DELETE FROM " + EdiReturn.class.getSimpleName());
                
                // usefull log info (do not remove)
                if (importedCount > 0 && log.isInfoEnabled()) {
                    log.info(String.format("Imported %d ediReturn rows", importedCount));
                }
            }
        });

        emailTopiaDao.updateAll(emailsToUpdate);
        getPersistenceContext().commit();
    }

    /**
     * Generate email details as PDF and return it as an printable attachment.
     * 
     * @param email email
     * @return attachment filled with pdf content
     */
    @Override
    public AttachmentFile getEmailDetailAsAttachment(Email email) {

        AttachmentFile result = null;

        try {

            // get html content
            String content = getEmailDetailAsHtml(email);

            // generate pdf
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ITextRenderer renderer = new ITextRenderer();
            renderer.setDocumentFromString(content);
            renderer.layout();
            renderer.createPDF(out);

            // create attachent
            result = new AttachmentFileImpl();
            result.setContent(out.toByteArray());
            result.setFilename(t("faxtomail.attachment.demand.filename") + ".pdf");
            result.getFile();

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("", e);
            }
        }
        return result;
    }
    
    protected String getEmailDetailAsHtml(Email email) {
        Decorator<FaxToMailUser> userDecorator = getDecoratorService().getDecoratorByType(FaxToMailUser.class);

        Map<String, Object> scopes = new HashMap<String, Object>();

        String result = email.getObject();
        String ref = email.getReference();
        if (!ref.isEmpty()) {
            result = ref + " - " + result;
        }

        scopes.put("title", result);
        scopes.put("receivedDate", DateFormatUtils.format(email.getReceptionDate(), "dd/MM/yyyy HH:mm"));
        scopes.put("sender", email.getSender());
        scopes.put("object", email.getObject());
        scopes.put("client", email.getClient());
        scopes.put("demandType", email.getDemandType());
        scopes.put("priority", email.getPriority());
        scopes.put("projectReference", email.getProjectReference());
        scopes.put("companyReference", email.getCompanyReference());
        scopes.put("waitingState", email.getWaitingState());
        scopes.put("status", email.getDemandStatus());
        scopes.put("takenBy", email.getTakenBy() == null ? "" : userDecorator.toString(email.getTakenBy()));
        scopes.put("message", email.getComment());
        scopes.put("date", DateFormatUtils.format(serviceContext.getNow(), "dd/MM/yyyy HH:mm"));

        scopes.put("firstOpeningUser", email.getFirstOpeningUser() == null ?
                "" : userDecorator.toString(email.getFirstOpeningUser()));
        scopes.put("firstOpeningDate", email.getFirstOpeningDate() == null ?
                "" : DateFormatUtils.format(email.getFirstOpeningDate(), "dd/MM/yyyy HH:mm"));

        scopes.put("lastModificationUser", email.getLastModificationUser() == null ?
                "" : userDecorator.toString(email.getLastModificationUser()));
        scopes.put("lastModificationDate", email.getLastModificationDate() == null ?
                "" : DateFormatUtils.format(email.getLastModificationDate(), "dd/MM/yyyy HH:mm"));

        scopes.put("lastAttachmentOpeningInFolderUser", email.getLastAttachmentOpeningInFolderUser() == null ?
                "" : userDecorator.toString(email.getLastAttachmentOpeningInFolderUser()));
        scopes.put("lastAttachmentOpeningInFolderDate", email.getLastAttachmentOpeningInFolderDate() == null ?
                "" : DateFormatUtils.format(email.getLastAttachmentOpeningInFolderDate(), "dd/MM/yyyy HH:mm"));

        scopes.put("hasRangeRows", email.sizeRangeRow() > 0);
        scopes.put("rangeRows", email.getRangeRow());

        // find template
        InputStream is = EmailServiceImpl.class.getResourceAsStream("/pdf/demande.mustache");
        MustacheFactory mf = new DefaultMustacheFactory();
        Mustache mustache = mf.compile(new InputStreamReader(is, StandardCharsets.UTF_8), "demande");
        StringWriter writer = new StringWriter();
        mustache.execute(writer, scopes);
        writer.flush();
        
        return writer.toString();
    }

    /**
     * Compute mail folder path (separated by /) from root to current.
     * 
     * @param folder folder to get path
     * @return full mail folder path
     */
    protected String getFullMailFolderPath(MailFolder folder) {
        StringBuilder sb = new StringBuilder(folder.getName());
        MailFolder loopFolder = folder.getParent();
        while (loopFolder != null) {
            sb.insert(0, "/");
            sb.insert(0, loopFolder.getName());
            loopFolder = loopFolder.getParent();
        }
        return sb.toString();
    }

    @Override
    public void importArchive(InputStream inputStream, File attachmentBase) {

        // all doas involved
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        DemandTypeTopiaDao demandTypedao = getPersistenceContext().getDemandTypeDao();
        PriorityTopiaDao priorityDao = getPersistenceContext().getPriorityDao();
        WaitingStateTopiaDao waitingStateDao = getPersistenceContext().getWaitingStateDao();
        MailFolderTopiaDao mailFolderDao = getPersistenceContext().getMailFolderDao();
        AttachmentTopiaDao attachmentDao = getPersistenceContext().getAttachmentDao();
        AttachmentFileTopiaDao attachmentFileDao = getPersistenceContext().getAttachmentFileDao();

        // get referentiel map
        Map<String, DemandType> allDemandTypes = Maps.uniqueIndex(demandTypedao, new Function<DemandType, String>() {
            @Override
            public String apply(DemandType input) {
                return input.getLabel();
            }
        });
        Map<String, Priority> allPriority = Maps.uniqueIndex(priorityDao, new Function<Priority, String>() {
            @Override
            public String apply(Priority input) {
                return input.getLabel();
            }
        });
        Map<String, WaitingState> allWaitingStates = Maps.uniqueIndex(waitingStateDao, new Function<WaitingState, String>() {
            @Override
            public String apply(WaitingState input) {
                return input.getLabel();
            }
        });
        
        // build folder map
        Map<String, MailFolder> mailFolderMap = Maps.uniqueIndex(mailFolderDao, new Function<MailFolder, String>() {
            @Override
            public String apply(MailFolder input) {
                return getFullMailFolderPath(input);
            }
        });

        // run import
        ArchiveImportModel archiveImportModel = new ArchiveImportModel(';', allWaitingStates, allDemandTypes, allPriority, mailFolderMap);
        Binder<ArchiveImportBean, Email> emailBinder = BinderFactory.newBinder(ArchiveImportBean.class, Email.class);
        Import<ArchiveImportBean> importer = null;
        try {
            importer = Import.newImport(archiveImportModel, new InputStreamReader(inputStream, getApplicationConfig().getImportFileEncoding()));
            for (ArchiveImportBean archiveBean : importer) {

                // create new email to persist
                Email email = emailDao.newInstance();
                emailBinder.copy(archiveBean, email);
                email.setDemandStatus(DemandStatus.ARCHIVED);

                // persist it
                email = emailDao.create(email);

                // manage attachments
                if (archiveBean.getAttachmentPaths() != null) {
                    Iterable<String> itAttachmentPaths = Splitter.on(',').omitEmptyStrings().trimResults().split(archiveBean.getAttachmentPaths());
                    for (String attachmentPath : itAttachmentPaths) {
                        File attFile;
                        if (attachmentBase != null) {
                            attFile = new File(attachmentBase, attachmentPath);
                        } else {
                            attFile = new File(attachmentPath);
                        }
                        if (!attFile.isFile()) {
                            throw new RuntimeException("Missing file " + attFile.getAbsolutePath());
                        }

                        Attachment attachment = new AttachmentImpl();
                        AttachmentFile attachmentFile = new AttachmentFileImpl();
                        attachmentFile.setFilename(attFile.getName());
                        attachmentFile.setContent(FileUtils.readFileToByteArray(attFile));
                        attachmentFile = attachmentFileDao.create(attachmentFile);
                        attachment.setOriginalFile(attachmentFile);

                        attachment = attachmentDao.create(attachment);
                        email.addAttachment(attachment);
                    }
                    
                    emailDao.update(email);
                }
            }

            getPersistenceContext().commit();

        } catch (ImportRuntimeException|IOException e) {
            String message;
            if (e.getCause() != null) {
                message = e.getCause().getMessage();
            } else {
                message = e.getMessage();
            }
            throw new ApplicationTechnicalException(message, e);
        } finally {
            IOUtils.closeQuietly(importer);
            IOUtils.closeQuietly(inputStream);
        }
    }

    @Override
    public long getArchivedMailCount() {
        EmailTopiaDao emailDao = getPersistenceContext().getEmailDao();
        long result = emailDao.forDemandStatusEquals(DemandStatus.ARCHIVED).count();
        return result;
    }
}
