package com.franciaflex.faxtomail.services.service;

/*
 * #%L
 * FaxToMail :: Service
 * $Id: LdapService.java 159 2014-06-09 16:20:09Z echatellier $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.2/faxtomail-service/src/main/java/com/franciaflex/faxtomail/services/service/LdapService.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUserGroup;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUserGroupTopiaDao;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUserImpl;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUserTopiaDao;
import com.franciaflex.faxtomail.services.FaxToMailServiceSupport;
import com.franciaflex.faxtomail.services.service.ldap.AuthenticationException;
import com.franciaflex.faxtomail.services.service.ldap.LdapUser;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.SearchResult;
import com.unboundid.ldap.sdk.SearchResultEntry;
import com.unboundid.ldap.sdk.SearchScope;

public class LdapService extends FaxToMailServiceSupport {

    private static final Log log = LogFactory.getLog(LdapService.class);

    /**
     * Get ldap connection.
     * 
     * @return
     * @throws LDAPException
     */
    protected LDAPConnection getLDAPConnection() throws LDAPException {
        // host, port, username and password
        return new LDAPConnection(getApplicationConfig().getLdapHost(), 
                getApplicationConfig().getLdapPort(),
                getApplicationConfig().getLdapUser(),
                getApplicationConfig().getLdapPassword());
    }

    /**
     * Get all user from ldap.
     * 
     * @return ldap users with group infos
     */
    public Collection<LdapUser> getAllLdapUsers() {
        Collection<LdapUser> results = new ArrayList<>();

        // ldapsearch -h ldap.codelutin.home -b "ou=People,DC=codelutin,DC=home" "objectClass=posixGroup"
        // ldapsearch -h ldap.codelutin.home -b "ou=People,DC=codelutin,DC=home" "objectClass=account"
        String[] baseDNs = {
                "OU=Utilisateurs,OU=Faber,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=OU Informatique,OU=France-Fermetures,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",

                // OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net
                "OU=Utilisateurs,OU=Carros,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=Checy,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=Guipry,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=LeRheu,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=Luzech,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=MaisonAlfort,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=Mauguio,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=Migennes,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=Nomades,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net",
                "OU=Utilisateurs,OU=RocheToirin,OU=Franciaflex,OU=Utilisateurs,OU=Mac-Groupe,DC=mac-groupe,DC=net"
        };
        String filter = "(objectClass=user)";

        LDAPConnection connection = null;
        try {
            connection = getLDAPConnection();
            if (connection.isConnected()) {
                for (String baseDN : baseDNs) {
                    SearchResult searchResult = connection.search(baseDN, SearchScope.ONE, filter);

                    List<SearchResultEntry> searchEntries = searchResult.getSearchEntries();
                    for (SearchResultEntry searchEntry : searchEntries) {
                        LdapUser user = new LdapUser();
                        user.setLogin(searchEntry.getAttributeValue("userPrincipalName"));

                        // Parse name
                        String fullName = searchEntry.getAttributeValue("name");
                        if (fullName.indexOf(' ') != -1) {
                            String lastName = fullName.substring(0, fullName.indexOf(' '));
                            String firstName = fullName.substring(fullName.indexOf(' ') + 1);
                            user.setFirstName(firstName);
                            user.setLastName(lastName);
                        } else {
                            user.setFirstName("");
                            user.setLastName(fullName);
                        }

                        // parse groups
                        String[] groups = searchEntry.getAttributeValues("memberOf");
                        if (ArrayUtils.isNotEmpty(groups)) {
                            for (String group : groups) {
                                user.addGroup(group);
                            }
                        }

                        results.add(user);
                    }
                }
            }
        } catch (LDAPException ex) {
            throw new RuntimeException("Can't connect to ldap", ex);
        } finally {
            if (connection != null) {
                connection.close();
            }
        }

        return results;
    }

    /**
     * Recupere les utilisateurs/groupes du ldap et met à jour la base locale.
     */
    public void updateLdapData() {
        if (StringUtils.isBlank(getApplicationConfig().getLdapHost())) {
            if (log.isDebugEnabled()) {
                log.debug("Ldap service not configured !");
            }
            return;
        } else {
            if (log.isDebugEnabled()) {
                log.debug("Updating user data from ldap");
            }
        }

        Collection<LdapUser> ldapUsers = getAllLdapUsers();

        FaxToMailUserTopiaDao faxtomailUserDao = getPersistenceContext().getFaxToMailUserDao();
        FaxToMailUserGroupTopiaDao faxToMailUserGroupDao = getPersistenceContext().getFaxToMailUserGroupDao();
        Binder<FaxToMailUser, FaxToMailUser> userBinder = BinderFactory.newBinder(FaxToMailUser.class);
        for (LdapUser ldapUser : ldapUsers) {

            // manage user from login
            FaxToMailUser user = faxtomailUserDao.forLoginEquals(ldapUser.getLogin()).findUniqueOrNull();
            if (user == null) {
                user = new FaxToMailUserImpl();
            }

            // FIXME echatellier 20140601 : it's not necessary to save user in database if information didn't change
            userBinder.copyExcluding(ldapUser, user,
                    FaxToMailUser.PROPERTY_TOPIA_ID,
                    FaxToMailUser.PROPERTY_TOPIA_CREATE_DATE,
                    FaxToMailUser.PROPERTY_TOPIA_VERSION,
                    FaxToMailUser.PROPERTY_USER_GROUPS);

            // manage user group
            Collection<String> groups = ldapUser.getGroups();
            user.clearUserGroups();
            for (String group : groups) {
                String groupPath = getGroupFullPath(group);
                String groupName = StringUtils.substringAfterLast(groupPath, "/");
                FaxToMailUserGroup userGroup = faxToMailUserGroupDao.forNameEquals(groupName).findUniqueOrNull();
                if (userGroup == null) {
                    userGroup = faxToMailUserGroupDao.create(
                        FaxToMailUserGroup.PROPERTY_NAME, groupName,
                        FaxToMailUserGroup.PROPERTY_FULL_PATH, groupPath);
                }
                user.addUserGroups(userGroup);
            }

            // persist user
            if (user.isPersisted()) {
                faxtomailUserDao.update(user);
            } else {
                faxtomailUserDao.create(user);
            }
        }

        getPersistenceContext().commit();
    }

    /**
     * Transform group CN to group path.
     * 
     * Example:
     * CN=Tout Franciaflex,OU=Listes de distribution,OU=Comptes Spéciaux,OU=Mac-Groupe,DC=mac-groupe,DC=net
     * net/mac-groupe/Mac-Groupe/Comptes Spéciaux/Listes de distribution/Tout Franciaflex
     * 
     * @param groupCN
     * @return
     */
    protected String getGroupFullPath(String groupCN) {
        String[] part = groupCN.split(",");
        ArrayUtils.reverse(part);
        String result = StringUtils.join(part, '/');
        return result;
    }

    /**
     * Return user for given user id.
     * 
     * @param userTopiaId userTopiaId
     * @return user bean (without password)
     */
    public FaxToMailUser getUserBean(String userTopiaId) {
        FaxToMailUser result = null;
        
        FaxToMailUserTopiaDao faxtomailUserDao = getPersistenceContext().getFaxToMailUserDao();
        FaxToMailUser user = faxtomailUserDao.forTopiaIdEquals(userTopiaId).findUniqueOrNull();
        if (user != null) {
            Binder<FaxToMailUser, FaxToMailUser> faxToMailUserBinder = BinderFactory.newBinder(FaxToMailUser.class);
            result = new FaxToMailUserImpl();
            faxToMailUserBinder.copyExcluding(user, result);
        }
        return result;
    }

    /**
     * Authenticate user.
     * 
     * @param login login
     * @param password password
     * @return authenticated user
     * @throws AuthenticationException if authentication fails
     */
    public FaxToMailUser authenticateUser(String login, String password) throws AuthenticationException {
        FaxToMailUserTopiaDao faxtomailUserDao = getPersistenceContext().getFaxToMailUserDao();
        FaxToMailUser user = faxtomailUserDao.forAll().findAnyOrNull();
        FaxToMailUser result = null;
        if (user != null) {
            Binder<FaxToMailUser, FaxToMailUser> faxToMailUserBinder = BinderFactory.newBinder(FaxToMailUser.class);
            result = new FaxToMailUserImpl();
            faxToMailUserBinder.copyExcluding(user, result);
        }
        return result;
    }
}
