/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.runner.impl;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Properties;
import org.sonar.runner.cache.Logger;
import org.sonar.runner.cache.PersistentCache;
import org.sonar.runner.commonsio.FileUtils;
import org.sonar.runner.commonsio.IOUtils;
import org.sonar.runner.impl.OkHttpClientFactory;
import org.sonar.runner.okhttp.OkHttpClient;
import org.sonar.runner.okhttp.Request;
import org.sonar.runner.okhttp.Response;
import org.sonar.runner.okhttp.ResponseBody;

class ServerConnection {
    private final String baseUrlWithoutTrailingSlash;
    private final String userAgent;
    private final OkHttpClient httpClient;
    private final PersistentCache wsCache;
    private final boolean preferCache;
    private final Logger logger;
    private final boolean isCacheEnabled;

    ServerConnection(String baseUrl, String userAgent, boolean preferCache, boolean cacheEnabled, PersistentCache cache, Logger logger) {
        this.isCacheEnabled = cacheEnabled;
        this.logger = logger;
        this.baseUrlWithoutTrailingSlash = ServerConnection.removeTrailingSlash(baseUrl);
        this.userAgent = userAgent;
        this.wsCache = cache;
        this.preferCache = preferCache;
        this.httpClient = OkHttpClientFactory.create();
    }

    private static String removeTrailingSlash(String url) {
        return url.replaceAll("(/)+$", "");
    }

    public static ServerConnection create(Properties props, PersistentCache cache, Logger logger, boolean preferCache) {
        String serverUrl = props.getProperty("sonar.host.url");
        String userAgent = String.format("%s/%s", props.getProperty("sonarRunner.app"), props.getProperty("sonarRunner.appVersion"));
        boolean enableCache = "issues".equalsIgnoreCase(props.getProperty("sonar.analysis.mode"));
        return new ServerConnection(serverUrl, userAgent, preferCache, enableCache, cache, logger);
    }

    boolean isCacheEnabled() {
        return this.isCacheEnabled;
    }

    public void downloadFile(String urlPath, File toFile) throws IOException {
        if (!urlPath.startsWith("/")) {
            throw new IllegalArgumentException(String.format("URL path must start with slash: %s", urlPath));
        }
        String url = this.baseUrlWithoutTrailingSlash + urlPath;
        this.logger.debug(String.format("Download %s to %s", url, toFile.getAbsolutePath()));
        ResponseBody responseBody = this.callUrl(url);
        try (FileOutputStream fileOutput = new FileOutputStream(toFile);){
            IOUtils.copyLarge(responseBody.byteStream(), fileOutput);
        }
        catch (IOException | RuntimeException e) {
            FileUtils.deleteQuietly(toFile);
            throw e;
        }
    }

    public String download(String urlPath) throws IOException {
        if (!urlPath.startsWith("/")) {
            throw new IllegalArgumentException(String.format("URL path must start with slash: %s", urlPath));
        }
        String url = this.baseUrlWithoutTrailingSlash + urlPath;
        if (this.isCacheEnabled && this.preferCache) {
            return this.tryCacheFirst(url);
        }
        return this.tryServerFirst(url);
    }

    private String downloadString(String url, boolean saveCache) throws IOException {
        this.logger.debug(String.format("Download: %s", url));
        ResponseBody responseBody = this.callUrl(url);
        String content = responseBody.string();
        if (saveCache) {
            try {
                this.wsCache.put(url, content.getBytes(StandardCharsets.UTF_8));
            }
            catch (IOException e) {
                this.logger.warn("Failed to cache WS call: " + e.getMessage());
            }
        }
        return content;
    }

    private String tryCacheFirst(String url) throws IOException {
        String cached = this.getFromCache(url);
        if (cached != null) {
            return cached;
        }
        try {
            return this.downloadString(url, this.preferCache);
        }
        catch (IOException | RuntimeException e) {
            this.logger.error(String.format("Data is not cached and SonarQube server [%s] can not be reached", this.baseUrlWithoutTrailingSlash));
            throw e;
        }
    }

    private String tryServerFirst(String url) throws IOException {
        try {
            return this.downloadString(url, this.isCacheEnabled);
        }
        catch (IOException e) {
            if (this.isCacheEnabled) {
                this.logger.info(String.format("SonarQube server [%s] can not be reached, trying cache", this.baseUrlWithoutTrailingSlash));
                String cached = this.getFromCache(url);
                if (cached != null) {
                    return cached;
                }
                this.logger.error(String.format("SonarQube server [%s] can not be reached and data is not cached", this.baseUrlWithoutTrailingSlash));
                throw e;
            }
            this.logger.error(String.format("SonarQube server [%s] can not be reached", this.baseUrlWithoutTrailingSlash));
            throw e;
        }
    }

    private String getFromCache(String fullUrl) {
        try {
            return this.wsCache.getString(fullUrl);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to access cache", e);
        }
    }

    private ResponseBody callUrl(String url) throws IOException, IllegalStateException {
        Request request = new Request.Builder().url(url).addHeader("User-Agent", this.userAgent).get().build();
        Response response = this.httpClient.newCall(request).execute();
        if (!response.isSuccessful()) {
            throw new IllegalStateException(String.format("Status returned by url [%s] is not valid: [%s]", response.request().url(), response.code()));
        }
        return response.body();
    }
}

