package org.planx.xpath.expr.operator;

import java.util.Iterator;
import org.planx.xpath.Navigator;
import org.planx.xpath.XPathException;
import org.planx.xpath.expr.Expression;
import org.planx.xpath.object.*;


/**
 * Relational operators include equals, not equals, less than,
 * greater then, etc.
 **/
public abstract class RelationalOperator extends Operator {
    public RelationalOperator(Expression e1, Expression e2) {
        super(e1, e2);
    }

    protected XObject evaluate(XObject o1, XObject o2, Navigator navigator)
                                                     throws XPathException {
        boolean result;

        if ((o1 instanceof XNodeSet) && (o2 instanceof XNodeSet)) {
            result = compare((XNodeSet) o1, (XNodeSet) o2, navigator);
        } else if ((o1 instanceof XNodeSet) && (o2 instanceof XSimple)) {
            XSimple v2 = (XSimple) o2;
            result = compare((XNodeSet) o1, v2.doubleValue(), navigator,false);
        } else if ((o1 instanceof XSimple) && (o2 instanceof XNodeSet)) {
            XSimple v1 = (XSimple) o1;
            result = compare((XNodeSet) o2, v1.doubleValue(), navigator, true);
        } else if ((o1 instanceof XSimple) && (o2 instanceof XSimple)) {
            XSimple v1 = (XSimple) o1;
            XSimple v2 = (XSimple) o2;
            result = compare(v1.doubleValue(), v2.doubleValue());
        } else {
            throw new XPathException("Unsupported subclass of XObject");
        }
        return result ? XBoolean.TRUE : XBoolean.FALSE;
    }

    /**
     * Compare XNodeSet and XNodeSet.
     * True if there is a node in the first set
     * and a node in the second set such that the
     * comparison is true.
     **/
    private boolean compare(XNodeSet o1, XNodeSet o2, Navigator navigator)
                                                    throws XPathException {
        double v1_min = Double.MAX_VALUE, v1_max = Double.MIN_VALUE;
        double v2_min = Double.MAX_VALUE, v2_max = Double.MIN_VALUE;

        for (int i = 0, size = o1.size(); i < size; i++) {
            double v=XString.doubleValue(navigator.getStringValue(o1.get(i)));
            if (v < v1_min) v1_min = v;
            if (v > v1_max) v1_max = v;
        }

        for (int i = 0, size = o2.size(); i < size; i++) {
            double v=XString.doubleValue(navigator.getStringValue(o2.get(i)));
            if (v < v2_min) v2_min = v;
            if (v > v2_max) v2_max = v;
        }

        return compareSets(v1_min, v1_max, v2_min, v2_max);
    }

    protected abstract boolean compareSets(double v1_min, double v1_max,
                                           double v2_min, double v2_max);

    private boolean compare(XNodeSet o1, double v2, Navigator navigator,
                                    boolean flip) throws XPathException {
        for (int i = 0, size = o1.size(); i < size; i++) {
            Object node = o1.get(i);
            double v1 = XString.doubleValue(navigator.getStringValue(node));
            if (flip ? compare(v2, v1) : compare(v1, v2)) return true;
        }
        return false;
    }

    protected abstract boolean compare(double v1, double v2);
}
