package org.planx.xpath;

/**
 * A Navigator is a bridge between XPath and a document model.
 **/
public interface Navigator {
    public static final int ROOT = 1;
    public static final int ELEMENT = 2;
    public static final int TEXT = 3;
    public static final int ATTRIBUTE = 4;
    public static final int NAMESPACE = 5;
    public static final int PROCESSING_INSTRUCTION = 6;
    public static final int COMMENT = 7;

    // Navigation

    /**
     * Return the root node of the document to which the
     * specified node belongs.
     **/
    public Object getRoot(Object node) throws XPathException;

    /**
     * Return the parent of the specified node or <code>null</code>
     * if it does not have a parent.
     **/
    public Object getParent(Object node) throws XPathException;

    /**
     * Return the next sibling of the specified node or <code>null</code>
     * if it does not have a one.
     **/
    public Object nextSibling(Object node) throws XPathException;

    /**
     * Return the previous sibling of the specified node or <code>null</code>
     * if it does not have a one.
     **/
    public Object previousSibling(Object node) throws XPathException;

    /**
     * Return the child of the specified node with the specified child index.
     **/
    public Object getChild(Object node, int index) throws XPathException;

    /**
     * Return the number of children of the specified node.
     **/
    public int childCount(Object node) throws XPathException;

    /**
     * Return the attribute of the specified node with the specified
     * attribute index.
     * Note that the indexing is merely for convenience and does
     * not imply a specific ordering of the attributes.
     **/
    public Object getAttribute(Object node, int index) throws XPathException;

    /**
     * Return the number of attributes of the specified node.
     **/
    public int attributeCount(Object node) throws XPathException;

    // Data

    /**
     * Return the name of the specified node. For element nodes this
     * is the name of the node. For text node this is <code>null</code>.
     * For attribute nodes this is the name of the attribute.
     **/
    public String getName(Object node) throws XPathException;

    /**
     * Return the value of the specified node as defined in the XPath
     * specification. For element nodes this is the concatenation
     * of the values of the descendant text nodes. For text nodes this
     * is the character contents of the node. For attribute nodes this
     * is the value of the attribute.
     **/
    public String getStringValue(Object node) throws XPathException;

    /**
     * Return the type of the node as specified in this interface.
     **/
    public int getType(Object node) throws XPathException;
}
