package org.planx.xmlstore.io;

import java.io.*;
import java.util.*;
import org.planx.msd.Discriminator;
import org.planx.msd.Memory;

/**
 * An implementation of <code>FileSystem</code> that uses an on-disk file.
 * It is not synchronized and should thus be synchronized externally.
 *
 * @author Thomas Ambus
 */
public class LocalFileSystem implements FileSystem {
    private static final int BUFFER_SIZE = 128*1024;
    private byte[] buffer = new byte[BUFFER_SIZE];
    private String name;
    private RandomAccessFile diskw;               // used for writing
    private RandomAccessFile diskr;               // used for reading
    private FileSystemIdentifier fsi;
    private SortedSet<LocalLocator> usedList;     // blocks of used space
    private LocalLocator pointer = null;          // next-fit allocation pointer

    /**
     * Creates a new file system with the specified file name or recreates
     * a previously persisted one.
     */
    public LocalFileSystem(String name) throws IOException {
        this.name = name;
        fsi = new FileSystemIdentifier();
        diskw = new RandomAccessFile(name, "rw"); // read-write disk
        diskr = new RandomAccessFile(name, "r");  // readonly disk
        usedList = new TreeSet<LocalLocator>();   // red-black tree, log(n) performance
        readUsedList();
    }

    public FileSystemIdentifier currentIdentifier() {
        // TODO: Could perhaps just return "fsi", but this is safe
        return new FileSystemIdentifier();
    }

    public long size() throws IOException {
        return diskw.length();
    }

    public void close() throws IOException {
        writeUsedList();
        diskw.close();
        diskr.close();
    }

    // READ/WRITE METHODS

    public void copy(FileSystem fs, LocalLocator fromLoc, LocalLocator locTo)
                       throws IOException, UnknownLocatorException {
        checkLocator(locTo);
        if (locTo.getLen() < fromLoc.getLen())
            throw new IOException("Not enough space at "+locTo+
                " to contain "+fromLoc);
        diskw.seek(locTo.getOff());
        int off = (int) diskw.getFilePointer();

        // Make raw copy using buffer
        DataInput in = fs.getInput(fromLoc);
        int remain = fromLoc.getLen();
        while (remain > 0) {
            int s = (remain > BUFFER_SIZE) ? BUFFER_SIZE : remain;
            in.readFully(buffer, 0, s);
            diskw.write(buffer, 0, s);
            remain -= s;
        }

        // Check size is correct
        int len = (int) diskw.getFilePointer() - off;
        if (len != fromLoc.getLen())
            throw new IOException("Size "+fromLoc.getLen()+" in argument "+
                                  "does not match copied bytes: "+len);
    }

    public DataInput getInput(LocalLocator l) throws IOException,
                                         UnknownLocatorException {
        checkLocator(l);
        diskr.seek(l.getOff());
        return diskr;
    }

    public DataOutput getOutput(LocalLocator l) throws IOException,
                                           UnknownLocatorException {
        checkLocator(l);
        diskw.seek(l.getOff());
        return diskw;
    }

    public LocalLocator all() throws IOException {
        return new LocalLocator(0, (int) diskw.length(), fsi);
    }

    // MAINTENANCE OF FREE-LIST

    public LocalLocator allocate() throws IOException {
        throw new UnsupportedOperationException();
    }

    public LocalLocator allocate(int size) throws IOException {
        return allocate(size, fsi);
    }

    public LocalLocator allocate(int size, FileSystemIdentifier id)
                                                throws IOException {
        LocalLocator loc = null;
        // Find free space using next-fit
        // The following simply ensures that the search is wrapped and
        // stops when the original starting point is reach again
        if (!usedList.isEmpty()) {
            if (pointer == null) pointer = usedList.first();
            LocalLocator start = pointer;
            loc = findFreeSpace(size, pointer, id);
            if (loc == null) {
                // No space found, wrap search
                pointer = usedList.first();
                if (start.compareTo(pointer) > 0) {
                    loc = findFreeSpace(size, null, id);
                }
            }
        }
        if (loc == null) {
            // No free space of required size could be found, so append
            // instead Note, that it cannot be the same locator inserted
            // in the used-list and returned, since the bound of the locator
            // in the used-list can be changed
            if (!usedList.isEmpty()) {
                LocalLocator last = usedList.last();
                loc = new LocalLocator(last.getOff()+last.getLen(), size, id);
                if (last.getFileSystemId().equals(id)) {
                    last.setLen(last.getLen()+size);
                } else {
                    usedList.add(new LocalLocator(last.getOff()+
                        last.getLen(), size, id));
                }
            } else {
                loc = new LocalLocator((int) diskw.length(), size, id);
                usedList.add(new LocalLocator((int) diskw.length(), size, id));
            }
        }
        return loc;
    }

    /**
     * Searches for free space of <code>size</code> from the current
     * <code>pointer</code>, which is updated during the search.
     * Returns <code>null</code> if no sufficiently large space was found.
     */
    private LocalLocator findFreeSpace(int size, LocalLocator prev,
                                            FileSystemIdentifier id) {
        SortedSet<LocalLocator> tailSet = usedList.tailSet(pointer); // greater or equal
        for (LocalLocator loc : tailSet) {
            if (loc == prev) continue;
            pointer = loc;
            int gap_off = (prev==null) ? 0 : prev.getOff()+prev.getLen();
            int gap_len = loc.getOff()-gap_off;
            if (gap_len < 0) throw new IllegalStateException(
                "LocalLocator "+prev+" extends into "+loc);
            if (gap_len >= size) {
                // Free space found, try to merge with surrounding blocks in
                // used-list, which is only allowed if they have matching fsi's
                if (prev != null && prev.getFileSystemId().equals(id)) {
                    if (gap_len == size && loc.getFileSystemId().equals(id)) {
                        // Perfect match: Can merge with both previous and next
                        prev.setLen(prev.getLen()+size+loc.getLen());
                    } else {
                        // Can merge only with previous
                        prev.setLen(prev.getLen()+size);
                    }
                } else {
                    if (gap_len == size && loc.getFileSystemId().equals(id)) {
                        // Can merge only with next
                        loc.setOff(gap_off);
                        loc.setLen(size+loc.getLen());
                    } else {
                        // Cannot merge with any
                        usedList.add(new LocalLocator(gap_off, size, id));
                    }
                }
                return new LocalLocator(gap_off, size, id);
            }
            prev = loc;
        }
        return null;
    }

    public void free(LocalLocator l) throws IOException, UnknownLocatorException {
        // Get encapsulating locator and check it
        LocalLocator inLoc = checkLocator(l);
        int off = l.getOff();
        int len = l.getLen();

        // Chop, divide, or remove encapsulating locator
        int ioff = inLoc.getOff();
        int ilen = inLoc.getLen();
        if (ioff == off && len == ilen) {
            // Perfect match
            usedList.remove(inLoc);
        } else if (ioff == off && len < ilen) {
            // Head-aligned, chop beginning of inLoc off
            inLoc.setOff(off+len);
            inLoc.setLen(ilen-len);
        } else if (ioff < off && off+len == ioff+ilen) {
            // Tail-aligned, chop end of inLoc off
            inLoc.setLen(off-ioff);
        } else if (ioff < off && off+len < ioff+ilen) {
            // Internal, divide inLoc in two
            LocalLocator nloc = new LocalLocator(off+len, (ioff+ilen)-(off+len),
                                                       inLoc.getFileSystemId());
            inLoc.setLen(off-ioff);
            usedList.add(nloc);
        } else {
            throw new IllegalArgumentException("LocalLocator "+l+
                                    " is not completely contained "+
                                               "in block "+inLoc);
        }
        truncate();

        // Generate new fsi to invalidate old locators
        fsi = new FileSystemIdentifier();
    }

    /**
     * If the last block in free list is at end of file, truncate file
     */
    private void truncate() throws IOException {
        if (!usedList.isEmpty()) {
            LocalLocator loc = usedList.last();
            int end = loc.getOff()+loc.getLen();
            if (diskw.length() > end) {
                diskw.setLength(end);       // truncate file
            }
        } else {
            diskw.setLength(0);             // truncate file
        }
    }

    public boolean isContained(LocalLocator loc) {
        try {
            checkLocator(loc);
            return true;
        } catch (UnknownLocatorException e) {
            return false;
        } catch (IOException e) {
            return false;
        }
    }

    /**
     * Returns the encapsulating locator in the used-list.
     */
    private LocalLocator checkLocator(LocalLocator l) throws IOException,
                                                 UnknownLocatorException {
        int off = l.getOff();
        int len = l.getLen();
        LocalLocator inLoc = null;
        SortedSet<LocalLocator> tailSet = usedList.tailSet(l);
        if (!tailSet.isEmpty()) {
            inLoc = tailSet.first();                 // greater or equal
            if (inLoc.getOff() != off) inLoc = null;
        }
        if (inLoc == null) {
            SortedSet<LocalLocator> headSet = usedList.headSet(l);
            if (!headSet.isEmpty()) inLoc = headSet.last(); // strictly less
        }
        if (inLoc == null)
            throw new IllegalArgumentException("No encapsulating locator for "+l);
        if (!inLoc.getFileSystemId().equals(l.getFileSystemId()))
            throw new IllegalArgumentException("Mismatching IDs in argument "+l+" and "+
                                               "encapsulating locator "+inLoc);
        int ioff = inLoc.getOff();
        int ilen = inLoc.getLen();
        if (ioff <= off && off+len <= ioff+ilen) {
            return inLoc;
        } else {
            throw new IllegalArgumentException("Misalignment of argument "+l+
                              " in relation to encapsulating locator "+inLoc);
        }
    }

    // PERSISTENCE OF FREE-LIST

    // TODO: Should write usedList more often than just when closing
    private void writeUsedList() throws IOException {
        Streamer<LocalLocator> locStreamer = LocalLocator.getStreamer(false);
        DataOutputStream out = new DataOutputStream(new FileOutputStream(name+".free"));
        out.writeInt(usedList.size());
        for (LocalLocator loc : usedList) {
            locStreamer.toStream(out, loc);
        }
        out.flush();
        out.close();
    }

    private void readUsedList() throws IOException {
        try {
            Streamer<LocalLocator> locStreamer = LocalLocator.getStreamer(false);
            DataInputStream in = new DataInputStream(new FileInputStream(name+".free"));
            int size = in.readInt();
            for (int i=0; i<size; i++) {
                LocalLocator loc = locStreamer.fromStream(in);
                usedList.add(loc);
            }
            in.close();
        } catch (FileNotFoundException e) {}
    }

    public String toString() {
        return "fsi="+fsi.toString()+", usedList="+usedList;
    }
}
