package org.planx.xmlstore;

import java.io.IOException;

/**
 * An <code>XMLStore</code> provides facilities for persisting semi-structured
 * data in the form of <code>Node</code>s. Once persisted, data can be retrieved
 * again using a <code>Reference</code>.
 * <p>
 * An <code>XMLStore</code> <i>may</i> assume that {@link #load(Reference)}s
 * are only applied to <code>Reference</code>s produced by the store
 * itself. However, it likely will not check that this is case. Indeed,
 * many <code>XMLStore</code> decorators will work without any problems even if
 * this is not case, though this is not guaranteed.
 *
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public interface XMLStore {
    /**
     * Saves the XML data represented by the specified <code>Node</code>.
     * The method returns a unique location independent
     * reference, which can be used to load the node, when needed.
     *
     * @param  node representing XML data in the form of semi-structured tree data
     * @return a reference to the saved node/tree
     * @throws IOException if an error occurs during disk, network, etc access
     */
    public Reference save(Node node) throws IOException;

    /**
     * Loads XML data identified by a reference.
     * The returned value is semi-structured tree data, represented by a "root
     * node" having 0 or more subtrees each represented by <code>Node</code>s.
     *
     * @param  ref the <code>Reference</code> to the data
     * @return node representing the loaded XML data
     * @throws IOException if an error occurs during disk, network, etc access
     * @throws UnknownReferenceException if the data referenced is not stored
     *         in the <code>XMLStore</code> or the <code>Reference</code> is of
     *         a type not supported by the <code>XMLStore</code>
     */
    public Node load(Reference ref) throws IOException,
                             UnknownReferenceException;

    /**
     * The method is used to let the <code>XMLStore</code> properly commit all
     * data to the underlying ressource, release resources, and so
     * forth. Once close has been issued on an <code>XMLStore</code>, the store
     * is no longer available for handling requests. Any subsequent
     * request to the store will result in an exception being
     * thrown. An <code>XMLStore</code> implementation <i>has</i> to ensure that
     * this happens.
     *
     * @throws IOException if it is not possible to properly close the store
     */
    public void close() throws IOException;
}
