package org.planx.util;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Utility class that provides methods for easy manipulation of <code>List</code>s
 * of <code>Pair</code>s.
 *
 * @author Thomas Ambus
 */
public final class Pairs {
    private Pairs() {}

    /**
     * Returns a <code>List</code> of <code>Pair</code>s where the two components
     * the <i>i</i>th element return the value of the <i>i</i>th element of
     * <code>values</code>.
     */
    public static <E> List<Pair<E,E>> identityPairs(E[] values) {
        return new ArrayIdentityPairs<E>(values);
    }

    private static class ArrayIdentityPairs<E> extends AbstractList<Pair<E,E>> {
        private final E[] values;

        ArrayIdentityPairs(E[] values) {
            this.values = values;
        }

        public int size() {
            return values.length;
        }

        public Pair<E,E> get(int index) {
            E elm = values[index];
            return new Association<E,E>(elm,elm);
        }
    }

    /**
     * Returns a <code>List</code> of <code>Pair</code>s where the two components
     * the <i>i</i>th element return the value of the <i>i</i>th element of
     * <code>values</code>.
     */
    public static <E> List<Pair<E,E>> identityPairs(List<? extends E> values) {
        return new IdentityPairsList<E>(values);
    }

    private static class IdentityPairsList<E> extends AbstractList<Pair<E,E>> {
        private final List<? extends E> values;

        IdentityPairsList(List<? extends E> values) {
            this.values = values;
        }

        public int size() {
            return values.size();
        }

        public Pair<E,E> get(int index) {
            E elm = values.get(index);
            return new Association<E,E>(elm,elm);
        }
    }

    /**
     * Returns a <code>List</code> of <code>Pair</code>s where the first component
     * of the <i>i</i>th element returns the value of the <i>i</i>th element of
     * <code>labels</code> and the second component of the <i>i</i>th element
     * returns the value of the <i>i</i>th element of <code>values</code>.
     * The two arguments must be of equal size.
     */
    public static <E,F> List<Pair<E,F>> mergePairs(E[] labels, F[] values) {
        return new ArrayMergePairs<E,F>(labels, values);
    }

    private static class ArrayMergePairs<E,F> extends AbstractList<Pair<E,F>> {
        private final E[] labels;
        private final F[] values;

        ArrayMergePairs(E[] labels, F[] values) {
            this.labels = labels;
            this.values = values;
        }

        public int size() {
            return labels.length;
        }

        public Pair<E,F> get(int index) {
            return new Association<E,F>(labels[index],values[index]);
        }
    }

    /**
     * Returns a <code>List</code> of <code>Pair</code>s where the first component
     * of the <i>i</i>th element returns the value of the <i>i</i>th element of
     * <code>labels</code> and the second component of the <i>i</i>th element
     * returns the value of the <i>i</i>th element of <code>values</code>.
     * The two arguments must be of equal size.
     */
    public static <E,F> List<Pair<E,F>> mergePairs(List<? extends E> labels, List<? extends F> values) {
        return new MergePairsList<E,F>(labels, values);
    }

    private static class MergePairsList<E,F> extends AbstractList<Pair<E,F>> {
        private final List<? extends E> labels;
        private final List<? extends F> values;

        MergePairsList(List<? extends E> labels, List<? extends F> values) {
            this.labels = labels;
            this.values = values;
        }

        public int size() {
            return values.size();
        }

        public Pair<E,F> get(int index) {
            return new Association<E,F>(labels.get(index), values.get(index));
        }
    }

    /**
     * Returns a <code>List</code> containing only the first components of the specified
     * <code>List</code> of <code>Pair</code>s.
     */
    public static <E,F> List<E> retainFirst(List<? extends Pair<? extends E,? extends F>> pairs) {
        return new FilterFirstList<E,F>(pairs);
    }

    private static class FilterFirstList<E,F> extends AbstractList<E> {
        private final List<? extends Pair<? extends E,? extends F>> values;

        FilterFirstList(List<? extends Pair<? extends E,? extends F>> values) {
            this.values = values;
        }

        public int size() {
            return values.size();
        }

        public E get(int index) {
            Pair<? extends E,? extends F> pair = values.get(index);
            return (pair == null) ? null : pair.getFirst();
        }
    }

    /**
     * Returns a <code>List</code> containing only the second components of the specified
     * <code>List</code> of <code>Pair</code>s.
     */
    public static <E,F> List<F> retainSecond(List<? extends Pair<? extends E,? extends F>> pairs) {
        return new FilterSecondList<E,F>(pairs);
    }

    private static class FilterSecondList<E,F> extends AbstractList<F> {
        private final List<? extends Pair<? extends E,? extends F>> values;

        FilterSecondList(List<? extends Pair<? extends E,? extends F>> values) {
            this.values = values;
        }

        public int size() {
            return values.size();
        }

        public F get(int index) {
            Pair<? extends E,? extends F> pair = values.get(index);
            return (pair == null) ? null : pair.getSecond();
        }
    }
}
