package org.planx.msd.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.planx.msd.*;
import org.planx.util.Pair;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset
 * of <code>Pair</code>s. <code>Discriminator</code>s for the two types
 * contained within the <code>Pair</code>s must be provided in the
 * constructor.
 *
 * @author Thomas Ambus
 */
public class PairDiscriminator<A,B> extends AbstractDiscriminator<Pair<A,B>> {
    private final Discriminator<A> d1;
    private final Discriminator<B> d2;

    public PairDiscriminator(Discriminator<A> d1, Discriminator<B> d2) {
        this.d1 = d1;
        this.d2 = d2;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                final Extractor<U,? extends Pair<A,B>,S> e) {
        if (values.isEmpty()) return Collections.emptyList();

        // Discriminate according to first component

        Extractor<U,A,U> firstExtractor =
            new Extractor<U,A,U>() {
                public A getLabel(U elm) {
                    return e.getLabel(elm).getFirst();
                }
                public U getValue(U elm) {
                    return elm;
                }
            };

        Collection<List<U>> result1 = d1.discriminate(values, firstExtractor);

        // Discriminate according to second component

        Extractor<U,B,S> secondExtractor =
            new Extractor<U,B,S>() {
                public B getLabel(U elm) {
                    return e.getLabel(elm).getSecond();
                }
                public S getValue(U elm) {
                    return e.getValue(elm);
                }
            };

        Collection<List<S>> result2 = new ArrayList<List<S>>();
        for (List<U> block : result1) {
            result2.addAll(d2.discriminate(block, secondExtractor));
        }
        return result2;
    }
}
