package org.planx.msd;

import java.util.AbstractList;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.planx.util.*;

/**
 * Utility class that provides methods for common discriminator tasks.
 *
 * @author Thomas Ambus
 */
public final class Discriminators {
    /**
     * Returns <code>true</code> if and only if <code>o1</code> and
     * <code>o2</code> are equivalent under the equivalence defined by
     * the specified <code>Discriminator</code>.
     */
    public static <E> boolean equals(E o1, E o2, Discriminator<E> d) {
        List<E> twoElms = Arrays.asList((E[]) new Object[] {o1, o2});
        Collection<List<E>> eqs = d.discriminate(twoElms);
        // Elms are equivalent if and only if there is one equivalence class
        return (eqs.size() != 1);
    }

    /**
     * Efficient equivalence testing of two <code>CharSequence</code>s.
     * Equivalence on the <code>CharSequence</code>s is defined as the
     * standard equality, where two sequences are equal if and only if
     * they have the same length and their elements are pair-wise equal.
     */
    public static boolean equals(CharSequence c1, CharSequence c2) {
        if (c1 == c2) return true;
        int n = c1.length();
        if (n != c2.length()) return false;
        for (int i=0; i<n; i++) {
            if (c1.charAt(i) != c2.charAt(i)) return false;
        }
        return true;
    }

    // EXTRACTOR UTILITIES

    /**
     * Returns an <code>Extractor</code> where the label and value are
     * identical to the original value.
     */
    public static <T> Extractor<T,T,T> identityExtractor() {
        // Cast is more efficient than type-checked construction of new obj
        return (Extractor<T,T,T>) IDENTITY_EXTRACTOR;
    }
    private static final Extractor<Object,Object,Object> IDENTITY_EXTRACTOR =
        new Extractor<Object,Object,Object>() {
            public Object getLabel(Object elm) {
                return elm;
            }
            public Object getValue(Object elm) {
                return elm;
            }
        };

    /**
     * Returns an <code>Extractor</code> where the label is extracted by
     * the specified <code>Extractor</code> and the value is just the
     * original value.
     */
    public static <U,T> Extractor<U,T,U> identityExtractor(
                                  final Extractor<U,T,?> e) {
        return new Extractor<U,T,U>() {
            public T getLabel(U elm) {
                return e.getLabel(elm);
            }
            public U getValue(U elm) {
                return elm;
            }
        };
    }

    /**
     * Returns an <code>Extractor</code> that extracts the first component
     * of a <code>Pair</code> as the label and the second component as the
     * value.
     */
    public static <T,S,P extends Pair<T,S>> Extractor<P,T,S> pairExtractor() {
        // Cast is more efficient than type-checked construction of new obj
        return (Extractor) PAIR_EXTRACTOR;
    }
    private static final Extractor<Pair<Object,Object>,Object,Object>
                                                      PAIR_EXTRACTOR =
        new Extractor<Pair<Object,Object>,Object,Object>() {
            public Object getLabel(Pair<Object,Object> elm) {
                return elm.getFirst();
            }
            public Object getValue(Pair<Object,Object> elm) {
                return elm.getSecond();
            }
        };

    /**
     * Returns an <code>Extractor</code> that extracts the first component
     * of a <code>Pair</code> as the value and the second component as the
     * label.
     */
    public static <T,S,P extends Pair<T,S>> Extractor<P,S,T> pairFlipExtractor() {
        // Cast is more efficient than type-checked construction of new obj
        return (Extractor) PAIR_FLIP_EXTRACTOR;
    }
    private static final Extractor<Pair<Object,Object>,Object,Object>
                                                 PAIR_FLIP_EXTRACTOR =
        new Extractor<Pair<Object,Object>,Object,Object>() {
            public Object getLabel(Pair<Object,Object> elm) {
                return elm.getSecond();
            }
            public Object getValue(Pair<Object,Object> elm) {
                return elm.getFirst();
            }
        };

    /**
     * Returns an <code>Extractor</code> that extracts the first component
     * of a <code>Pair</code> as the label and the whole pair as the value.
     */
    public static <T,P extends Pair<T,?>> Extractor<P,T,P> pairFirstExtractor() {
        // Cast is more efficient than type-checked construction of new obj
        return (Extractor) PAIR_FIRST_EXTRACTOR;
    }
    private static final Extractor<Pair<Object,Object>,Object,
                         Pair<Object,Object>> PAIR_FIRST_EXTRACTOR =
        new Extractor<Pair<Object,Object>,Object,Pair<Object,Object>>() {
            public Object getLabel(Pair<Object,Object> elm) {
                return elm.getFirst();
            }
            public Pair<Object,Object> getValue(Pair<Object,Object> elm) {
                return elm;
            }
        };

    /**
     * Returns an <code>Extractor</code> that extracts the second component
     * of a <code>Pair</code> as the label and the whole pair as the value.
     */
    public static <S,P extends Pair<?,S>> Extractor<P,S,P> pairSecondExtractor() {
        // Cast is more efficient than type-checked construction of new obj
        return (Extractor) PAIR_SECOND_EXTRACTOR;
    }
    private static final Extractor<Pair<Object,Object>,Object,
                   Pair<Object,Object>> PAIR_SECOND_EXTRACTOR =
        new Extractor<Pair<Object,Object>,Object,Pair<Object,Object>>() {
            public Object getLabel(Pair<Object,Object> elm) {
                return elm.getSecond();
            }
            public Pair<Object,Object> getValue(Pair<Object,Object> elm) {
                return elm;
            }
        };

    // LIST UTILITIES

    /**
     * Returns a <code>List</code> containing the values (as determined by
     * the <code>Extractor</code>
     * <code>e</code>) of the specified <code>List</code>.
     */
    public static <U,S> List<S> valueList(List<? extends U> list,
                                          Extractor<U,?,S> e) {
        return new ValueList<U,S>(list, e);
    }

    private static class ValueList<U,S> extends AbstractList<S> {
        private final List<? extends U> list;
        private final Extractor<U,?,S> e;

        ValueList(List<? extends U> list, Extractor<U,?,S> e) {
            this.list = list;
            this.e = e;
        }

        public int size() {
            return list.size();
        }

        public S get(int index) {
            return e.getValue(list.get(index));
        }
    }

    // COMPOSED DISCRIMINATION UTILITIES

    public static <U,S,A,B> Collection<List<S>> discriminate(
                List<? extends U> values,
                Discriminator<A> d1, Extractor<U,? extends A,U> e1,
                Discriminator<B> d2, Extractor<U,? extends B,S> e2) {
        Discriminator[] ds = new Discriminator[] {d1, d2};
        Extractor[] es = new Extractor[] {e1, e2};
        return discriminate(values, ds, es);
    }

    public static <U,S,A,B,C> Collection<List<S>> discriminate(
                List<? extends U> values,
                Discriminator<A> d1, Extractor<U,? extends A,U> e1,
                Discriminator<B> d2, Extractor<U,? extends B,U> e2,
                Discriminator<C> d3, Extractor<U,? extends C,S> e3) {
        Discriminator[] ds = new Discriminator[] {d1, d2, d3};
        Extractor[] es = new Extractor[] {e1, e2, e3};
        return discriminate(values, ds, es);
    }

    public static <U,S,A,B,C,D> Collection<List<S>> discriminate(
                List<? extends U> values,
                Discriminator<A> d1, Extractor<U,? extends A,U> e1,
                Discriminator<B> d2, Extractor<U,? extends B,U> e2,
                Discriminator<C> d3, Extractor<U,? extends C,U> e3,
                Discriminator<D> d4, Extractor<U,? extends D,S> e4) {
        Discriminator[] ds = new Discriminator[] {d1, d2, d3, d4};
        Extractor[] es = new Extractor[] {e1, e2, e3, e4};
        return discriminate(values, ds, es);
    }

    private static final int INIT_CAPACITY = 10;

    static <U,S> Collection<List<S>> discriminate(
                List<? extends U> values,
                Discriminator[] ds, Extractor[] es) {
        if (values.isEmpty()) return Collections.emptyList();

        Collection<List<S>> result = new ArrayList<List<S>>();
        Collection<List<S>> lastPartition;
        Collection<List<U>> partition;

        int[] indexes = new int[INIT_CAPACITY]; // Note: Initialized to 0
        List[] work = new List[INIT_CAPACITY];
        int work_capacity = INIT_CAPACITY;
        int work_head = 1;
        work[0] = values;

        // Step through all discriminator/extractor pairs using result of first
        // discriminator as input for the next, etc.

        while (work_head > 0) {
            if (work_head < work_capacity) work[work_head] = null;
            work_head--;

            // Subpartition current unfinished block

            List<U> block = work[work_head];
            int blockSize = block.size();
            int index = indexes[work_head];

            if (ds.length > index+1) {

                // Last discriminator not reached, partition further

                partition = ds[index].discriminate(block, es[index]);

                // Add blocks in partition to result or unfinished

                index++;
                for (List<U> subBlock : partition) {
                    if (subBlock.size() > 1) {
                        if (work_head+1 >= work_capacity) {
                            work = Array.ensureCapacity(
                                work, work_head, work_capacity+1);
                            indexes = Array.ensureCapacity(indexes,
                                work_head, work_capacity+1);
                            work_capacity = work.length;
                        }
                        work[work_head] = subBlock;
                        indexes[work_head] = index;
                        work_head++;
                    } else {
                        result.add(Discriminators.valueList(subBlock,
                            es[es.length-1]));
                    }
                }
            } else {
                // Last discriminator reached, this will return the correct
                // type of result
                // (i.e. type S instead of type U)
                lastPartition = ds[index].discriminate(block, es[index]);
                result.addAll(lastPartition);
            }
        }
        return result;
    }
}
