package org.planx.xpath.object;

import org.planx.xpath.XPathException;

/**
 * A number returned by an XPath expression.
 **/
public class XNumber extends XSimple {
    public static final XNumber NaN = new XNumber(Double.NaN);
    private final Double value;

    /**
     * Construct an XNumber which represents the
     * specified <code>double</code> value.
     **/
    public XNumber(double value) {
        this(new Double(value));
    }

    /**
     * Construct an XNumber which represents the
     * specified Double value.
     **/
    public XNumber(Double value) {
        this.value = value;
    }

    // Conversion

    /**
     * Returns the value as a <code>double</code>.
     **/
    public double doubleValue() {
        return value.doubleValue();
    }

    /**
     * Returns the value as a <code>long</code>.
     **/
    public long longValue() {
        return value.longValue();
    }

    /**
     * Returns the value as an <code>int</code>.
     **/
    public int intValue() {
        return value.intValue();
    }

    /**
     * Returns <code>false</code> if the value of this XNumber
     * is <code>NaN</code> or 0. Otherwise returns <code>true</code>.
     **/
    public boolean booleanValue() {
        double value = doubleValue();
        if (Double.isNaN(value) || value == 0D) return false;
        else return true;
    }

    /**
     * Returns this XNumber as a string.
     **/
    public String stringValue() {
        long lngValue = longValue();
        if (doubleValue() == ((double) lngValue)) {
            return Long.toString(lngValue);
        } else {
            return value.toString();
        }
    }

    /**
     * Returns this XNumber as a string.
     **/
    public String toString() {
        return stringValue();
    }

    // Override methods in Object

    /**
     * Returns true if and only if <code>o</code> is
     * an instance of XNumber and it has the same
     * value as this.
     **/
    public boolean equals(Object o) {
        if (o instanceof XNumber) {
            XNumber other = (XNumber) o;
            return doubleValue() == other.doubleValue();
        }
        return false;
    }

    /**
     * Returns a hash code value for this object.
     **/
    public int hashCode() {
        return value.hashCode();
    }
}

