package org.planx.xmlstore.nodes;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import org.planx.msd.Discriminator;
import org.planx.msd.DiscriminatorFactory;
import org.planx.msd.Memory;
import org.planx.msd.character.CharSequenceDiscriminator;
import org.planx.msd.util.DiscriminatorAdapter;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.io.Streamer;
import org.planx.xmlstore.io.Streamers;

/**
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class DVMAttribute implements Attribute, Comparable<Attribute>, CharSequence {
    private static final char MAGIC_CHAR = 0;
    private final String name;
    private final String value;

    public DVMAttribute(String name, String value) {
        if (name == null) throw new NullPointerException
            ("Name may not be null");
        if (value == null) throw new NullPointerException
            ("Value may not be null");
        this.name = name;
        this.value = value;
    }

    public String getName() {
        return name;
    }

    public String getValue() {
        return value;
    }

    public int compareTo(Attribute o) {
        if (o == null) throw new NullPointerException();
        int v = compare(name, o.getName());
        return (v != 0) ? v : compare(value, o.getValue());
    }

    private static int compare(String s1, String s2) {
        return (s1==s2) ? 0 : (s1==null) ? -1 :
            (s2==null) ? 1 : s1.compareTo(s2);
    }

    public boolean equals(Object o) {
        if (o == this) return true;
        if (!(o instanceof Attribute)) return false;

        Attribute other = (Attribute) o;
        Object oName = other.getName();
        Object oValue = other.getValue();
        return ((name == null && oName == null) ||
                (name != null && name.equals(oName))) &&
               ((value == null && oValue == null) ||
                (value != null && value.equals(oValue)));
    }

    public int hashCode() {
        int h1 = (name == null) ? 0 : name.hashCode();
        int h2 = (value == null) ? 0 : value.hashCode();
        return 31*h1+h2;
    }

    public String toString() {
        return name +"="+ value;
    }

    public char charAt(int index) {
        int len1 = name.length();
        if (index < len1) return name.charAt(index);
        if (index == len1) return MAGIC_CHAR;
        return value.charAt(index - len1 - 1);
    }

    public int length() {
        return name.length() + value.length() + 1;
    }

    public CharSequence subSequence(int start, int end) {
        throw new UnsupportedOperationException();
    }

    public static Streamer<Attribute> getStreamer() {
        return new Streamer<Attribute>() {
            public void toStream(DataOutput out, Attribute attr)
                        throws IOException {
                Streamers.writeUTF(out, attr.getName());
                Streamers.writeUTF(out, attr.getValue());
            }

            public Attribute fromStream(DataInput in) throws IOException {
                String name = Streamers.readUTF(in);
                String value = Streamers.readUTF(in);
                return new DVMAttribute(name, value);
            }
        };
    }

    public static Discriminator<Attribute> getDiscriminator() {
        return getDiscriminator(DiscriminatorFactory.instance().getMemory());
    }

    public static Discriminator<Attribute> getDiscriminator(Memory memory) {
        return new CharSequenceDiscriminator<Attribute>(memory);
    }
}
