package org.planx.xmlstore.convert;

import java.util.Arrays;
import java.util.List;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.NodeFactory;

/**
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 */
public class DOMBuilder {
    private DOMBuilder() {}

    /**
     * Converts a DOM representation of an XML tree to a DVM representation.
     * The DVM representation only contains immutable nodes.
     *
     * @param  domNode the DOM node representing an XML tree.
     * @return root node of created DVM representation.
     * @throws XMLException
     */
    public static Node build(org.w3c.dom.Node domNode) throws XMLException {
        NodeFactory factory = NodeFactory.instance();
        switch (domNode.getNodeType()) {
        case org.w3c.dom.Node.TEXT_NODE:
            return factory.createCharDataNode(domNode.getNodeValue());

        case org.w3c.dom.Node.ELEMENT_NODE:
            org.w3c.dom.NamedNodeMap domAttrs = domNode.getAttributes();
            Attribute[] attr = new Attribute[domAttrs.getLength()];

            for (int i=0; i<attr.length; i++) {
                org.w3c.dom.Node att = domAttrs.item(i);
                attr[i] = factory.createAttribute(att.getNodeName(),
                                                  att.getNodeValue());
            }
            List<Attribute> attributes = Arrays.asList(attr);

            org.w3c.dom.NodeList domChildren = domNode.getChildNodes();
            Node[] dvmChildren = new Node[domChildren.getLength()];

            for (int i=0; i<dvmChildren.length; i++) {
                dvmChildren[i] = build(domChildren.item(i));
            }
            List<Node> childNodes = Arrays.asList(dvmChildren);

            return factory.createElementNode(domNode.getNodeName(),
                                             childNodes, attributes);
        default:
            throw new XMLException
                ("node type "+domNode.getNodeType()+" not supported");
        }
    }
}
