package org.planx.util;

import java.util.AbstractList;
import java.util.List;
import java.util.RandomAccess;

/**
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public final class Array {
    private Array() {}

    /**
     * Find the first occurrence of a value in an array starting from
     * the offset and looking no further than offset plus the length.
     *
     * @param array   the array to search in
     * @param value   the value to look for
     * @param off     the offset to start at
     * @param len     the maximal length to search through
     *
     * @return the index of the value if found in array[off..off+len-1],
     *     -1 otherwise.
     **/
    public static int indexOf(byte[] array, byte value, int off, int len) {
        int pos = 0;
        for (int i = off; i<off+len; i++) {
            if (array[i] == value) {
                pos = i; break;
            }
        }
        if (pos<off || pos>off+len) {
            return -1;
        }
        return pos;
    }

    /** Compare to arrays (according to the Comparable.compareTo
     *  comparison convention) lexicographically.
     *
     * @param b1 first array to compare
     * @param b2 second array to compare
     *
     * @return 0 if the arrays are identical, a negative integer if b1 is
     *    lexicographically smaller than b2, a positive integer 1 if b2 is
     *    lexicographically larger than b2.
     **/
    public static int compareTo(byte[] b1, byte[] b2) {
        if (b1 == b2) return 0;
        if (b1 == null) return -1; /* cf. an empty string of bytes? */
        if (b2 == null) return +1; /* cf. an empty string of bytes? */

        int len1 = b1.length;
        int len2 = b2.length;
        int len = Math.min(len1,len2);
        int i = 0;
        while (i < len) {
            if (b1[i] != b2[i]) {
                return b1[i] - b2[i];
            }
            i++;
        }
        return len1 - len2;
    }

    // GROWABLE ARRAY UTILITIES

    /**
     * Takes an array as argument and returns an array containing the same
     * elements, but where it is ensured that the array can contain the specified
     * minimum number of elements. The <code>size</code> is the current number of
     * used slots in the array, and <code>minCapacity</code> is the minimum number
     * of slots required. If the length of the array is lower or equal to
     * <code>minCapacity</code>, the same array <code>arr</code> is returned.
     * Otherwise, a new array is allocated and the <code>size</code> first
     * elements of <code>arr</code> is copied to the new array. The length of the
     * new array is not specified, other than it is larger than
     * <code>minCapacity</code>.
     * <p>
     * Typical use, given an <code>Iterator</code> variable <code>it</code>:
     * <p>
     * <blockquote><pre>
     *     int size = 0;
     *     int[] arr = new int[10];
     *     while (it.hasNext()) {
     *         arr = Array.ensureCapacity(arr, size, size+1);
     *         arr[size++] = it.next().hashCode();
     *     }
     * </pre></blockquote>
     */
    public static int[] ensureCapacity(int[] arr, int size, int minCapacity) {
        int oldCapacity = arr.length;
        if (minCapacity > oldCapacity) {
            int[] oldData = arr;
            int newCapacity = (oldCapacity * 3)/2 + 1;
            if (newCapacity < minCapacity) newCapacity = minCapacity;
            arr = new int[newCapacity];
            System.arraycopy(oldData, 0, arr, 0, size);
        }
        return arr;
    }

    /**
     * See {@link #ensureCapacity(int[],int,int)}.
     */
    public static int[][] ensureCapacity(int[][] arr, int size, int minCapacity) {
        int oldCapacity = arr.length;
        if (minCapacity > oldCapacity) {
            int[][] oldData = arr;
            int newCapacity = (oldCapacity * 3)/2 + 1;
            if (newCapacity < minCapacity) newCapacity = minCapacity;
            arr = new int[newCapacity][];
            System.arraycopy(oldData, 0, arr, 0, size);
        }
        return arr;
    }

    /**
     * See {@link #ensureCapacity(int[],int,int)}.
     */
    public static byte[] ensureCapacity(byte[] arr, int size, int minCapacity) {
        int oldCapacity = arr.length;
        if (minCapacity > oldCapacity) {
            byte[] oldData = arr;
            int newCapacity = (oldCapacity * 3)/2 + 1;
            if (newCapacity < minCapacity) newCapacity = minCapacity;
            arr = new byte[newCapacity];
            System.arraycopy(oldData, 0, arr, 0, size);
        }
        return arr;
    }

    /**
     * See {@link #ensureCapacity(int[],int,int)}.
     */
    public static CharSequence[] ensureCapacity(CharSequence[] arr, int size, int minCapacity) {
        int oldCapacity = arr.length;
        if (minCapacity > oldCapacity) {
            CharSequence[] oldData = arr;
            int newCapacity = (oldCapacity * 3)/2 + 1;
            if (newCapacity < minCapacity) newCapacity = minCapacity;
            arr = new CharSequence[newCapacity];
            System.arraycopy(oldData, 0, arr, 0, size);
        }
        return arr;
    }

    /**
     * See {@link #ensureCapacity(int[],int,int)}.
     */
    public static CharSequence[][] ensureCapacity(CharSequence[][] arr, int size, int minCapacity) {
        int oldCapacity = arr.length;
        if (minCapacity > oldCapacity) {
            CharSequence[][] oldData = arr;
            int newCapacity = (oldCapacity * 3)/2 + 1;
            if (newCapacity < minCapacity) newCapacity = minCapacity;
            arr = new CharSequence[newCapacity][];
            System.arraycopy(oldData, 0, arr, 0, size);
        }
        return arr;
    }

    /**
     * See {@link #ensureCapacity(int[],int,int)}.
     */
    public static List[] ensureCapacity(List[] arr, int size, int minCapacity) {
        int oldCapacity = arr.length;
        if (minCapacity > oldCapacity) {
            List[] oldData = arr;
            int newCapacity = (oldCapacity * 3)/2 + 1;
            if (newCapacity < minCapacity) newCapacity = minCapacity;
            arr = new List[newCapacity];
            System.arraycopy(oldData, 0, arr, 0, size);
        }
        return arr;
    }

    /**
     * See {@link #ensureCapacity(int[],int,int)}.
     */
    public static Object[] ensureCapacity(Object[] arr, int size, int minCapacity) {
        int oldCapacity = arr.length;
        if (minCapacity > oldCapacity) {
            Object[] oldData = arr;
            int newCapacity = (oldCapacity * 3)/2 + 1;
            if (newCapacity < minCapacity) newCapacity = minCapacity;
            arr = new Object[newCapacity];
            System.arraycopy(oldData, 0, arr, 0, size);
        }
        return arr;
    }

    /**
     * Sets the length of the array to the specified length. If the array already
     * has this length, it is simply returned. If the array is larger than this
     * length, the first <code>length</code> elements is copied into a new array
     * with the specified length. If the array is smaller than this length,
     * all elements are copied into the beginning of a new array with the specified
     * length.
     */
    public static int[] setLength(int[] arr, int length) {
        int oldLength = arr.length;
        if (oldLength == length) return arr;
        int[] newArr = new int[length];
        int size = (oldLength < length) ? oldLength : length;
        System.arraycopy(arr, 0, newArr, 0, size);
        return newArr;
    }

    /**
     * See {@link #setLength(int[], int)}.
     */
    public static byte[] setLength(byte[] arr, int length) {
        int oldLength = arr.length;
        if (oldLength == length) return arr;
        byte[] newArr = new byte[length];
        int size = (oldLength < length) ? oldLength : length;
        System.arraycopy(arr, 0, newArr, 0, size);
        return newArr;
    }

    // ARRAY TO LIST UTILITIES

    /**
     * Returns a shallow, immutable copy of the specified <code>List</code>.
     */
    public static <E> List<E> unmodifiableCopy(List<E> list) {
        E[] arr = (E[]) list.toArray();
        return new UnmodifiableArrayList<E>(arr);
    }

    public static <E> List<E> unmodifiableCopy(E[] arr) {
        E[] cp = (E[]) new Object[arr.length];
        System.arraycopy(arr, 0, cp, 0, arr.length);
        return new UnmodifiableArrayList<E>(cp);
    }

    public static <E> List<E> asUnmodifiableList(E[] arr) {
        return new UnmodifiableArrayList<E>(arr);
    }
}
