/*
 * #%L
 * Wikitty :: publication Maven plugin
 * 
 * $Id: CopyDependencyUtil.java 1215 2011-08-30 14:37:34Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.2/wp-maven-plugin/src/main/java/org/nuiton/wikitty/plugin/CopyDependencyUtil.java $
 * %%
 * Copyright (C) 2011 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.plugin;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.installer.ArtifactInstallationException;
import org.apache.maven.artifact.installer.ArtifactInstaller;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.ArtifactRepositoryFactory;
import org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.dependency.AbstractFromDependenciesMojo;
import org.apache.maven.plugin.dependency.utils.DependencyStatusSets;
import org.apache.maven.plugin.dependency.utils.DependencyUtil;
import org.apache.maven.plugin.dependency.utils.filters.DestFileFilter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.artifact.filter.collection.ArtifactsFilter;

import java.io.File;
import java.net.MalformedURLException;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class CopyDependencyUtil extends AbstractFromDependenciesMojo {

    public ArtifactInstaller getInstaller() {
        return installer;
    }

    public void setInstaller(ArtifactInstaller installer) {
        this.installer = installer;
    }

    public ArtifactRepositoryFactory getRepositoryFactory() {
        return repositoryFactory;
    }

    public void setRepositoryFactory(ArtifactRepositoryFactory repositoryFactory) {
        this.repositoryFactory = repositoryFactory;
    }

    public Map getRepositoryLayouts() {
        return repositoryLayouts;
    }

    public void setRepositoryLayouts(Map repositoryLayouts) {
        this.repositoryLayouts = repositoryLayouts;
    }

    /** @component */
    protected ArtifactInstaller installer;

    /** @component */
    protected ArtifactRepositoryFactory repositoryFactory;

    /**
     * @component role=
     * "org.apache.maven.artifact.repository.layout.ArtifactRepositoryLayout"
     */
    private Map repositoryLayouts;

    /**
     * Main entry into mojo. Gets the list of dependencies and iterates through
     * calling copyArtifact.
     *
     * @throws MojoExecutionException with a message if an error occurs.
     * @see #getDependencies
     * @see #copyArtifact(Artifact, boolean)
     */
    public void execute() throws MojoExecutionException {
        DependencyStatusSets dss = getDependencySets(this.failOnMissingClassifierArtifact);
        Set artifacts = dss.getResolvedDependencies();

        if (!useRepositoryLayout) {
            for (Iterator i = artifacts.iterator(); i.hasNext(); ) {
                copyArtifact((Artifact) i.next(), this.stripVersion,
                             this.prependGroupId);
            }
        } else {
            try {
                ArtifactRepository targetRepository = repositoryFactory
                        .createDeploymentArtifactRepository("local",
                                                            outputDirectory.toURI().toURL().toExternalForm(),
                                                            (ArtifactRepositoryLayout) repositoryLayouts
                                                                    .get("default"), false /* uniqueVersion */);
                for (Iterator i = artifacts.iterator(); i.hasNext(); ) {
                    installArtifact((Artifact) i.next(), targetRepository);
                }
            } catch (MalformedURLException e) {
                throw new MojoExecutionException(
                        "Could not create outputDirectory repository", e);
            }
        }

        Set skippedArtifacts = dss.getSkippedDependencies();
        for (Iterator i = skippedArtifacts.iterator(); i.hasNext(); ) {
            Artifact artifact = (Artifact) i.next();
            getLog().info(
                    artifact.getFile().getName()
                    + " already exists in destination.");
        }

        if (isCopyPom()) {
            copyPoms(getOutputDirectory(), artifacts, this.stripVersion);
            copyPoms(getOutputDirectory(), skippedArtifacts, this.stripVersion); // Artifacts
            // that
            // already
            // exist
            // may
            // not
            // already
            // have
            // poms.
        }
    }

    private void installArtifact(Artifact artifact,
                                 ArtifactRepository targetRepository) {
        try {
            if ("pom".equals(artifact.getType())) {
                installer.install(artifact.getFile(), artifact,
                                  targetRepository);
                installBaseSnapshot(artifact, targetRepository);
            } else {
                installer.install(artifact.getFile(), artifact,
                                  targetRepository);
                installBaseSnapshot(artifact, targetRepository);

                if (isCopyPom()) {
                    Artifact pomArtifact = getResolvedPomArtifact(artifact);
                    if (pomArtifact.getFile() != null
                        && pomArtifact.getFile().exists()) {
                        installer.install(pomArtifact.getFile(), pomArtifact,
                                          targetRepository);
                        installBaseSnapshot(pomArtifact, targetRepository);
                    }
                }
            }
        } catch (ArtifactInstallationException e) {
            getLog().info(e.getMessage());
        }
    }

    private void installBaseSnapshot(Artifact artifact,
                                     ArtifactRepository targetRepository)
            throws ArtifactInstallationException {
        if (artifact.isSnapshot()
            && !artifact.getBaseVersion().equals(artifact.getVersion())) {
            Artifact baseArtifact = this.factory.createArtifact(
                    artifact.getGroupId(), artifact.getArtifactId(),
                    artifact.getBaseVersion(), artifact.getScope(),
                    artifact.getType());
            installer.install(artifact.getFile(), baseArtifact,
                              targetRepository);
        }
    }

    /**
     * Copies the Artifact after building the destination file name if
     * overridden. This method also checks if the classifier is set and adds it
     * to the destination file name if needed.
     *
     * @param artifact       representing the object to be copied.
     * @param removeVersion  specifies if the version should be removed from the file name
     *                       when copying.
     * @param prependGroupId specifies if the groupId should be prepend to the file while
     *                       copying.
     * @throws MojoExecutionException with a message if an error occurs.
     * @see DependencyUtil#copyFile(File, File, Log)
     * @see DependencyUtil#getFormattedFileName(Artifact, boolean)
     */
    protected void copyArtifact(Artifact artifact, boolean removeVersion,
                                boolean prependGroupId) throws MojoExecutionException {

        String destFileName = DependencyUtil.getFormattedFileName(artifact,
                                                                  removeVersion, prependGroupId);

        File destDir;
        destDir = DependencyUtil.getFormattedOutputDirectory(
                useSubDirectoryPerScope, useSubDirectoryPerType,
                useSubDirectoryPerArtifact, useRepositoryLayout, stripVersion,
                outputDirectory, artifact);
        File destFile = new File(destDir, destFileName);

        copyFile(artifact.getFile(), destFile);
    }

    /** Copy the pom files associated with the artifacts. */
    public void copyPoms(File destDir, Set artifacts, boolean removeVersion)
            throws MojoExecutionException

    {
        Iterator iter = artifacts.iterator();
        while (iter.hasNext()) {
            Artifact artifact = (Artifact) iter.next();
            Artifact pomArtifact = getResolvedPomArtifact(artifact);

            // Copy the pom
            if (pomArtifact.getFile() != null && pomArtifact.getFile().exists()) {
                File pomDestFile = new File(destDir,
                                            DependencyUtil.getFormattedFileName(pomArtifact,
                                                                                removeVersion, prependGroupId));
                if (!pomDestFile.exists()) {
                    copyFile(pomArtifact.getFile(), pomDestFile);
                }
            }
        }
    }

    protected Artifact getResolvedPomArtifact(Artifact artifact) {
        Artifact pomArtifact = this.factory.createArtifact(
                artifact.getGroupId(), artifact.getArtifactId(),
                artifact.getVersion(), "", "pom");
        // Resolve the pom artifact using repos
        try {
            this.resolver.resolve(pomArtifact, this.remoteRepos,
                                  this.getLocal());
        } catch (Exception e) {
            getLog().info(e.getMessage());
        }
        return pomArtifact;
    }

    protected ArtifactsFilter getMarkedArtifactFilter() {
        return new DestFileFilter(this.overWriteReleases,
                                  this.overWriteSnapshots, this.overWriteIfNewer,
                                  this.useSubDirectoryPerArtifact, this.useSubDirectoryPerType,
                                  this.useSubDirectoryPerScope, this.useRepositoryLayout,
                                  this.stripVersion, this.outputDirectory);
    }

    public void setProject(MavenProject pro) {
        project = pro;
    }

}
