/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyServiceEvent.java 439 2010-10-20 22:26:21Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-api/src/main/java/org/nuiton/wikitty/WikittyServiceEvent.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty;

import com.thoughtworks.xstream.XStream;
import java.util.EventObject;
import java.util.Map;
import java.util.Set;

/**
 * Wikitty service event.
 * 
 * Contains :
 * <ul>
 *  <li>Wikitty service as source
 *  <li>ids : wikitty ids or extensions ids
 *  <li>ids & version : only for put : version of added wikitties
 *  <li>ids & extension : only for put : extension of added ids
 * </ul>
 * 
 * @author chatellier
 * @version $Revision: 439 $
 * 
 * Last update : $Date: 2010-10-21 00:26:21 +0200 (jeu., 21 oct. 2010) $
 * By : $Author: bpoussin $
 */
public class WikittyServiceEvent extends EventObject {

    /** serialVersionUID. */
    private static final long serialVersionUID = 9017732163643700599L;

    /** Message type (put, remove, clear...). */
    static public enum WikittyEventType {
        PUT_WIKITTY(WikittyServiceListener.PUT_WIKITTY_METHOD),
        REMOVE_WIKITTY(WikittyServiceListener.REMOVE_WIKITTY_METHOD),
        CLEAR_WIKITTY(WikittyServiceListener.CLEAR_WIKITTY_METHOD),
        PUT_EXTENSION(WikittyServiceListener.PUT_EXTENSION_METHOD),
        REMOVE_EXTENSION(WikittyServiceListener.REMOVE_EXTENSION_METHOD),
        CLEAR_EXTENSION(WikittyServiceListener.CLEAR_EXTENSION_METHOD);

        /** le nom de la methode du listener a appeler pour ce type d'event */
        public String listenerMethodName;
        WikittyEventType(String listenerMethodName) {
            this.listenerMethodName = listenerMethodName;
        }
    }


    /** Remote event (received from server). */
    protected boolean remote;

    /** event type */
    protected WikittyEventType type;

    /** heure de creation de l'event */
    protected long time;

    /**
     * Id managed by event.
     * 
     * Can be :
     * <ul>
     *  <li>wikitty ids (uuid)
     *  <li>extension ids (ex : toto[2.1])
     * </ul>
     */
    protected Set<String> ids;
    
    /**
     * Map between extensions set for each id.
     * Not used for extension event
     */
    protected Map<String, Set<String>> idExtensions;
    
    /**
     * Map between versions for each id.
     */
    protected Map<String, String> idVersions;

    /**
     * Map between deletion date for each id, if event is REMOVE.
     */
    protected Map<String, String> idDeletionDate;

    /**
     * Constructor with source {@link WikittyService}.
     * 
     * @param source wikitty service
     */
    public WikittyServiceEvent(Object source, WikittyEventType type) {
        super(source);
        this.time = System.currentTimeMillis();
        this.type = type;
    }

    /**
     * Return time of event creation
     * @return
     */
    public long getTime() {
        return time;
    }

    /**
     * To allow set transient source after deserialisation.
     * 
     * @param source source
     */
    public void setSource(Object source) {
        this.source = source;
    }

    public WikittyEventType getType() {
        return type;
    }

    public void setType(WikittyEventType type) {
        this.type = type;
    }

    /**
     * Is event remote.
     * 
     * @return remote event
     */
    public boolean isRemote() {
        return remote;
    }

    /**
     * Change remote event property.
     * 
     * @param remote remote
     */
    public void setRemote(boolean remote) {
        this.remote = remote;
    }

    /**
     * Get ids.
     * 
     * @return ids or {@code null} if no ids is related to event
     */
    public Set<String> getIds() {
        return ids;
    }

    /**
     * Set ids.
     * 
     * @param ids ids to set
     */
    public void setIds(Set<String> ids) {
        this.ids = ids;
    }

    /**
     * Get id and extensions map.
     * 
     * @return map with extension set for each id or {@code null} if no ids is related to event
     */
    public Map<String, Set<String>> getIdExtensions() {
        return idExtensions;
    }

    /**
     * Change id and extensions map.
     * 
     * @param idExtensions id and extensions map
     */
    public void setIdExtensions(Map<String, Set<String>> idExtensions) {
        this.idExtensions = idExtensions;
    }
    
    /**
     * Get id and version map.
     * 
     * @return map with version for each id
     */
    public Map<String, String> getIdVersions() {
        return idVersions;
    }

    /**
     * Set id and version map.
     * 
     * @param idVersions id and versions
     */
    public void setIdVersions(Map<String, String> idVersions) {
        this.idVersions = idVersions;
    }

    @Override
    public String toString() {
        String toString = getClass().getName() + "[source=" + source +
        ", type=" + type +
        ", remote=" + remote + ", ids=" + ids + ", idExtensions=" + idExtensions +
        ", idVersions=" + idVersions + ", idDeletionDate=" + idVersions + "]";
        return toString;
    }

    /**
     * Permet de serializer en XML l'event. Pourrait etre utilise pour l'envoi
     * sur un transporteur qui ne peremt pas la serialisation java
     * @return
     */
    public String toXML() {
        XStream xstream = new XStream();
        xstream.setMode(XStream.NO_REFERENCES);
        xstream.alias("event", WikittyServiceEvent.class);
        String result = xstream.toXML(this);
        return result;
    }

    /**
     * Inverse de la methode toXML
     * @param xml
     * @return
     */
    static public WikittyServiceEvent fromXML(String xml) {
        XStream xstream = new XStream();
		xstream.alias("event", WikittyServiceEvent.class);
		WikittyServiceEvent result = (WikittyServiceEvent)xstream.fromXML(xml);
        return result;
    }
}
