package org.nuiton.wikitty;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.search.Search;

/**
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyServiceSecurity implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceSecurity.class);

    /** nom du groupe des administrateurs de l'application */
    static final public String WIKITTY_APPADMIN_GROUP_NAME = "WikittyAppAdmin";

    protected WikittyService ws;

    /** cache de l'id du groupe AppAdmin */
    transient protected String appAdminGroupId = null;

    public static final String APPADMIN_LOGIN = "root";

    // TODO 20100826 bleny look for password in a config file
    public static final String APPADMIN_PASSWORD = "toto";

    public WikittyServiceSecurity(WikittyService ws) {
        this.ws = ws;

        Wikitty appAdminGroup = getAppAdminGroup(null);

        if (WikittyGroupHelper.getMembers(appAdminGroup) == null) {
            // first time boot
            ws.storeExtension(null, WikittyUserAbstract.extensions);
            ws.storeExtension(null, SecurityTokenAbstract.extensions);
            ws.storeExtension(null, WikittyGroupAbstract.extensions);

            // create the appAdmin account
            Wikitty appAdmin = new WikittyImpl();
            WikittyUserHelper.addExtension(appAdmin);
            WikittyUserHelper.setLogin(appAdmin, APPADMIN_LOGIN);
            WikittyUserHelper.setPassword(appAdmin, APPADMIN_PASSWORD);
            ws.store(null, appAdmin);

            // add APPADMIN_LOGIN to AppAdmin group        
            WikittyGroupHelper.addMembers(appAdminGroup, appAdmin.getId());
            ws.store(null, appAdminGroup);

            // login as admin to add some security polices
            String adminToken = login(APPADMIN_LOGIN, APPADMIN_PASSWORD);

            // FIXME 20100923 bleny make all tokens unwritable, except for app admin

            logout(adminToken);
        }
    }

    @Override
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.addWikittyServiceListener(listener, type);
    }

    @Override
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.addWikittyServiceListener(listener, type);
    }

    @Override
    public String login(String login, String password) {
        Wikitty user = ws.findByCriteria(null, Search.query().eq(
                WikittyUser.FQ_FIELD_WIKITTYUSER_LOGIN, login).criteria());
        if (user == null) {
            throw new IllegalArgumentException(String.format("no such account '%s'", login));
        } else {
            // check password is valid
            if (WikittyUserHelper.getPassword(user).equals(password)) {
                String tokenId = WikittyUtil.genSecurityTokenId();
                Wikitty wikittyToken = new WikittyImpl(tokenId);
                // force add extension to wikitty
                SecurityTokenHelper.addExtension(wikittyToken);
                SecurityTokenHelper.setUser(wikittyToken, user.getId());
                ws.store(null, wikittyToken);
                log.debug(String.format("token '%s' is for login '%s'",
                                                               tokenId, login));
                return tokenId;
            } else {
                throw new SecurityException("bad password");
            }
        }
    }
    
    public void createAccount(String securityToken, String login, String password) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            Wikitty user = ws.findByCriteria(securityToken, Search.query().eq(
                    WikittyUser.FQ_FIELD_WIKITTYUSER_LOGIN, login).criteria());
            if (user == null) {
                user = new WikittyImpl();
                WikittyUserHelper.addExtension(user);
                WikittyUserHelper.setLogin(user, login);
                WikittyUserHelper.setPassword(user, password);
                ws.store(null, user);
                log.debug(String.format("login '%s' has userId '%s'", login, user.getId()));
            } else {
                throw new IllegalArgumentException(
                                    String.format("account already exists '%s'",
                                    login));
            }
        } else {
            throw new SecurityException("only admin can create accounts");
        }
    }

    public String getUserWikittyId(String securityToken, String login) {
        String userId = getUserId(securityToken);
        String userWikittyId = null;
        Wikitty user = ws.findByCriteria(null, Search.query().eq(
                WikittyUser.FQ_FIELD_WIKITTYUSER_LOGIN, login).criteria());
        if (user != null) {
            userWikittyId = user.getId();
        }
        return userWikittyId;
    }

    @Override
    public void logout(String securityToken) {
        getUserId(securityToken); // will throw exception if token is not valid
        ws.delete(securityToken, securityToken);
    }

    @Override
    public void clear(String securityToken) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            // seul les AppAdmin on le droit a cette method
            ws.clear(securityToken);
        } else {
            throw new SecurityException(_("user %s can't clear data", userId));
        }
    }
    
    protected String extensionToWikittySecurityId(WikittyExtension extension) {
        return String.format("WikittySecurity'%s'", extension.getName());
    }
    
    /**  */
    public Wikitty addWikittyAuthorisation(String securityToken,
                                        WikittyExtension extension) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            if (restoreExtensionAuthorisation(securityToken, extension) == null) {
                String wikittyAuthorisationId = extensionToWikittySecurityId(extension);
                Wikitty wikittyAuthorisation = new WikittyImpl(wikittyAuthorisationId);
                WikittyAuthorisationHelper.addExtension(wikittyAuthorisation);
                WikittyAuthorisationHelper.setOwner(wikittyAuthorisation, userId);
                ws.store(securityToken, wikittyAuthorisation);
                return wikittyAuthorisation;
            } else {
                throw new SecurityException(String.format(
                          "extension %s already has an security extension attached",
                          extension.getName()));
            }
        } else {
            throw new SecurityException(String.format(
                               "Only members of %s group can add authorisation",
                               WIKITTY_APPADMIN_GROUP_NAME));
        }
    }
    
    /** restore the wikitty authorisation attached to given extension
     *
     * @return a wikitty with WikittyAuthorisation extension, or null if given
     *         extension has no security policy attached
     * @throws SecurityException if user don't have rights required
     */
    public Wikitty restoreExtensionAuthorisation(String securityToken,
                                                  WikittyExtension extension) {
        String userId = getUserId(securityToken);
        String wikittyAuthorisationId = extensionToWikittySecurityId(extension);
        Wikitty wikittyAuthorisation = restore(securityToken, wikittyAuthorisationId);
        if (wikittyAuthorisation == null) {
            log.debug(extension + " has no authorization attached");
        } else {
            /*
            if ( ! canAdmin(securityToken, userId, wikittyAuthorisation)) {
                throw new SecurityException(String.format(
                            "user %s doesn't have admin rights on extension %s",
                            userId, extension.getName()));
            }
            */
        }
        return wikittyAuthorisation;
    }
    
    public void storeWikittyAuthorisation(String securityToken,
                                          Wikitty wikitty) {

        String userId = getUserId(securityToken);

        Wikitty oldVersion = ws.restore(null, wikitty.getId());

        // check that the wikitty does not have 
        if (WikittyAuthorisationHelper.isExtension(wikitty)) {

            if (oldVersion == null) {
                // if this exception is raised, you should use addWikittyAuthorisation()
                throw new IllegalArgumentException("you can't store an authorisation for the fist time");

            } else {
                
                if ( canAdmin(securityToken, userId, oldVersion) ) {
                    
                    if (isAdmin(securityToken, userId, oldVersion)) {                            
                        // admin can't change owner, admin or parent
                        // putting back old values
                        Object oldValue = oldVersion.getFieldAsObject(
                                         WikittyAuthorisation.EXT_WIKITTYAUTHORISATION,
                                         WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_OWNER);
                        wikitty.setField(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION,
                                         WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_OWNER,
                                         oldValue);

                        WikittyAuthorisationHelper.setOwner(wikitty, 
                                WikittyAuthorisationHelper.getOwner(oldVersion));
                        WikittyAuthorisationHelper.setParent(wikitty, 
                                WikittyAuthorisationHelper.getParent(oldVersion));
                        
                    }

                    ws.store(null, wikitty);
                } else {
                    throw new SecurityException(String.format(
                            "user %s can't admin rights for this extension", userId));
                }
            }
        } else {
            throw new IllegalArgumentException(String.format(
                    "wikitty %s is not a wikittyAuthorisation. It misses the extension",
                    wikitty));
        }
        
    }

    /** true if userId has the right to write on extension */
    protected boolean canRead(String securityToken, String userId, Wikitty extensionRights) {
        boolean canRead = isReader(securityToken, userId, extensionRights)
                       || canWrite(securityToken, userId, extensionRights); 
        return canRead;
    }

    /** true if userId has the right to write on extension */
    protected boolean canWrite(String securityToken, String userId, Wikitty extensionRights) {
        boolean canWrite = isWriter(securityToken, userId, extensionRights)
                        || isOwner(securityToken, userId, extensionRights)
                        || isAppAdmin(securityToken, userId);
        return canWrite;
    }

    /** true if userId has the right to admin on extension */
    protected boolean canAdmin(String securityToken, String userId, Wikitty extensionRights) {
        boolean canWrite = isAdmin(securityToken, userId, extensionRights)
                        || isOwner(securityToken, userId, extensionRights)
                        || isAppAdmin(securityToken, userId);
        return canWrite;
    }

    /** true if userId has the right is owner of all the extensions of the given wikitty */
    protected boolean canDelete(String securityToken, String userId, Wikitty wikitty) {
        if (isAppAdmin(securityToken, userId)) {
            return true;
        }
        
        // now read all extensions for this wikitty, and return false
        // if user is not owner on one of those extensions
        for (WikittyExtension extension : wikitty.getExtensions()) {
            Wikitty extensionRights = restoreExtensionAuthorisation(securityToken, extension);
            boolean canDelete = extensionRights == null
                             || isOwner(securityToken, userId, extensionRights);
            // FIXME 20100922 bleny if appadmin ?
            if (! canDelete) {
                return false;
            }
        }
        return true;
    }

    @Override
    public UpdateResponse store(String securityToken, Wikitty wikitty) {
        Collection<Wikitty> wikitties = Arrays.asList(wikitty);
        wikitties = removeUnauthorizedModifications(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, wikitties);
        return result;
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties) {
        Collection<Wikitty> wikittiesToStore = removeUnauthorizedModifications(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, wikittiesToStore);
        return result;
    }
    
    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties, boolean force) {
        Collection<Wikitty> wikittiesToStore = removeUnauthorizedModifications(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, wikittiesToStore, force);
        return result;
    }
    
    /**
     *  
     */
    protected Collection<Wikitty> removeUnauthorizedModifications(String securityToken, Collection<Wikitty> wikitties) {
        String userId = getUserId(securityToken);
        List<Wikitty> wikittiesToStore = new ArrayList<Wikitty>();
        for (Wikitty wikitty : wikitties) {
            // check that the wikitty does not have 
            if (WikittyAuthorisationHelper.isExtension(wikitty)) {
                storeWikittyAuthorisation(securityToken, wikitty);
            } else {
                // usual case, a user want to store a wikitty
                Wikitty oldVersion = ws.restore(null, wikitty.getId());
                if (oldVersion == null) { // it's a creation
                    // check that **reader** right on Security for all extension
                } else { // it's an update
                    // filtering, revert changes on field that this user can't write
                    for (WikittyExtension extension : wikitty.getExtensions()) {
                        Wikitty extensionRights = restoreExtensionAuthorisation(securityToken, extension);
                        if (extensionRights != null) { 
                            if ( ! canWrite(securityToken, userId, extensionRights)) {
                                // the user doesn't have the rights to write
                                // on the fields of extension. Moving back
                                // values to the old one
                                for (String fieldName : extension.getFieldNames()) {
                                    if (oldVersion == null) {
                                        wikitty.setField(extension.getName(), fieldName, null);
                                    } else {
                                        Object oldValue = oldVersion.getFieldAsObject(extension.getName(), fieldName);
                                        wikitty.setField(extension.getName(), fieldName, oldValue);
                                    }
                                }
                            }
                        } // else no particular right on this extension
                    }
                    wikittiesToStore.add(wikitty);
                }
            }
        }
        return wikittiesToStore;
    }

    @Override
    public UpdateResponse store(String securityToken, WikittyTransaction transaction, Collection<Wikitty> wikitties, boolean force) {
        Collection<Wikitty> wikittiesToStore = removeUnauthorizedModifications(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, transaction, wikittiesToStore, force);
        return result;
    }
    
    @Override
    public Wikitty restore(String securityToken, String id) {
        List<String> ids = Arrays.asList(id);
        List<Wikitty> wikitties = restore(securityToken, ids);
        Wikitty wikitty = null;
        if (! wikitties.isEmpty()) {
            wikitty = wikitties.get(0);
        }
        return wikitty;
    }

    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        List<Wikitty> wikitties = new ArrayList<Wikitty>();
        for (String id : ids) {
            // do it first, will throw an exception if security token is invalid

            String userId = getUserId(securityToken);

            Wikitty wikitty = ws.restore(securityToken, id);
            if (wikitty != null) {
                // FIXME 20100827 bleny copy on write is done because setting some field to null below modify stored wikitty if WikittyServiceInMemory is used
                wikitty = new WikittyCopyOnWrite(wikitty);

                for (WikittyExtension extension : wikitty.getExtensions()) {
                    Wikitty extensionRights = restoreExtensionAuthorisation(securityToken, extension);
                    
                    // field of extension can be read if no policy attached
                    // if a policy is attached, check that user has right to read
                    boolean canRead = extensionRights == null || canRead(securityToken, userId, extensionRights);
                    if ( ! canRead) {
                        for (String fieldName : extension.getFieldNames()) {
                            wikitty.setField(extension.getName(), fieldName, null);
                        }
                    }
                }
                wikitties.add(wikitty);
            }
        }
        return wikitties;
    }

    @Override
    public List<Wikitty> restore(String securityToken, WikittyTransaction transaction, List<String> ids) {
        throw new UnsupportedOperationException();
        // ws.restore(securityToken, transaction, ids);
    }
    
    @Override
    public void delete(String securityToken, String id) {
        Collection<String> ids = Arrays.asList(id);
        delete(securityToken, ids);
    }

    @Override
    public void delete(String securityToken, Collection<String> ids) {
        List<String> idsAsList = new ArrayList<String>(ids);
        secureDelete(securityToken, idsAsList);
    }

    /** delete wikitties only if user has right to */
    protected void secureDelete(String securityToken, List<String> ids) {
        String userId = getUserId(securityToken);

        List<Wikitty> wikitties = ws.restore(securityToken, ids);
        List<String> idsToRemove = new ArrayList<String>();
        
        for (Wikitty wikitty : wikitties) {
            if ( canDelete(securityToken, userId, wikitty)) {
                idsToRemove.add(wikitty.getId());
            }
        }

        ws.delete(securityToken, idsToRemove);
    }

    @Override
    @Deprecated
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public boolean canDelete(String securityToken, String wikittyId) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public boolean canRead(String securityToken, String wikittyId) {
        throw new UnsupportedOperationException();
    }

    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        // All people can read extension
        return ws.getAllExtensionIds(securityToken);
    }

    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        // All people can read extension
        return ws.getAllExtensionsRequires(securityToken, extensionName);
    }

    @Override
    public UpdateResponse storeExtension(
            String securityToken, WikittyExtension ext) {
        Collection<WikittyExtension> exts = Arrays.asList(ext);
        return storeExtension(securityToken, exts);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken,
            Collection<WikittyExtension> exts) {
        // TODO poussin 20100607 check security, mais qui a le droit ?
        
        
        
        
        return ws.storeExtension(securityToken, exts);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken,
            WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        String userId = getUserId(securityToken);
        UpdateResponse response = null;
        if (isAppAdmin(securityToken, userId)) {
            response = ws.storeExtension(securityToken, transaction, exts);
        }
        return response;
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken, String id) {
        // All people can read extension
        return ws.restoreExtension(securityToken, id);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken,
            WikittyTransaction transaction, String id) {
        // All people can read extension
        return ws.restoreExtension(securityToken, transaction, id);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name) {
        // All people can read extension
        return ws.restoreExtensionLastVersion(securityToken, name);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, WikittyTransaction transaction, String name) {
        // All people can read extension
        return ws.restoreExtensionLastVersion(securityToken, transaction, name);
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken, Criteria criteria) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByCriteria(securityToken, criteria);
        return result;
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByCriteria(
                securityToken, transaction, criteria);
        return result;
    }

    @Override
    public Wikitty findByCriteria(String securityToken, Criteria criteria) {
        Wikitty result = ws.findByCriteria(securityToken, criteria);
        if (!canRead(securityToken, result.getId())) {
            // user don't have correct right, return null
            result = null;
        }
        return result;
    }

    @Override
    public void addLabel(String securityToken, String wikittyId, String label) {
        // TODO poussin 20100607 check security
        ws.addLabel(securityToken, wikittyId, label);
    }

    @Override
    public PagedResult<String> findAllByLabel(String securityToken,
            String label, int firstIndex, int endIndex) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByLabel(
                securityToken, label, firstIndex, endIndex);
        return result;
    }

    @Override
    public Wikitty findByLabel(String securityToken, String label) {
        Wikitty result = ws.findByLabel(securityToken, label);
        if (!canRead(securityToken, result.getId())) {
            // user don't have correct right, return null
            result = null;
        }
        return result;
    }

    @Override
    public Set<String> findAllAppliedLabels(String securityToken, String wikittyId) {
        Set<String> result = ws.findAllAppliedLabels(securityToken, wikittyId);
        return result;
    }

    @Override
    public Tree restoreTree(String securityToken, String wikittyId) {
        // FIXME poussin 20100607 check security
        return ws.restoreTree(securityToken, wikittyId);
    }

    @Override
    public Entry<TreeNode, Integer> restoreNode(
            String securityToken, String wikittyId, Criteria filter) {
        // FIXME poussin 20100607 check security
        return ws.restoreNode(securityToken, wikittyId, filter);
    }

    @Override
    public Map<TreeNode, Integer> restoreChildren(
            String securityToken, String wikittyId, Criteria filter) {
        // FIXME poussin 20100607 check security
        return ws.restoreChildren(securityToken, wikittyId, filter);
    }

    @Override
    public Wikitty restoreVersion(
            String securityToken, String wikittyId, String version) {
        Wikitty result = ws.restoreVersion(securityToken, wikittyId, version);
        if (!canRead(securityToken, result.getId())) {
            // user don't have correct right, return null
            result = null;
        }
        return result;
    }

    @Override
    public UpdateResponse syncEngin(String securityToken) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            // seul les AppAdmin on le droit a cette method
            return ws.syncEngin(securityToken);
        } else {
            throw new SecurityException(_("user %s can't sync search engine",
                    getUserId(securityToken)));
        }
    }


    //
    // Method helper to check right
    //

    /**
     * 
     * @param pagedResult
     * @return
     */
    protected PagedResult<Wikitty> checkPagedResult(PagedResult<Wikitty> pagedResult) {
        // TODO poussin 20100610 que faire
        // TODO  - parcourir tous les resultats pour retirer ceux auquel on a pas le droit
        // TODO  - lever une exception des qu'on trouve un element interdit

        return pagedResult;
    }

    /**
     * Recupere l'identifiant de l'utilisateur associe au securityToken
     *
     * @param securityToken
     * @return l'identifiant de l'utilisateur, ou null si le token est invalide
     */
    protected String getUserId(String securityToken) {
        String result = null;
        // recuperation de l'utilisateur associe au securityToken
        // le securityToken est aussi l'id de l'objet
        Wikitty securityTokenWikitty = ws.restore(securityToken, securityToken);
        if (securityTokenWikitty == null) {
            throw new SecurityException(_("trying to use an invalidate security token %s", securityToken));
        } else {
            result = SecurityTokenHelper.getUser(securityTokenWikitty);
        }
        return result;
    }

    /**
     * verifie que l'utilisateur est dans la liste des admin
     *
     * @param userId
     * @param w
     * @return vrai si et seulement si l'utilisateur est dans la liste des
     * admin
     */
    protected boolean isAdmin(String securityToken, String userId, Wikitty extensionRights) {
        boolean result = isMember(
                securityToken, userId, extensionRights, WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_ADMIN);
        return result;
    }

    /**
     * verifie que l'utilisateur est dans la liste des writer
     *
     * @param userId
     * @param w
     * @return vrai si et seulement si l'utilisateur est dans la liste des
     * writers
     */
    protected boolean isWriter(String securityToken, String userId, Wikitty extensionRights) {
        boolean result = isMember(
                securityToken, userId, extensionRights, WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_WRITER);
        return result;
    }

    /**
     * Verifie que l'utilisateur est bien le proprietaire de l'objet
     *
     * @param userId
     * @param w
     * @return
     */
    protected boolean isOwner(String securityToken, String userId, Wikitty extensionRights) {
        String owner = WikittyAuthorisationHelper.getOwner(extensionRights);
        boolean result = userId.equals(owner);
        return result;
    }
    
    protected boolean isMember(String securityToken, String userId, Wikitty extensionRights, String fieldName) {
        Set<String> groupOrUser = extensionRights.getFieldAsSet(
                    WikittyAuthorisation.EXT_WIKITTYAUTHORISATION,
                    fieldName,
                    String.class);
        boolean result = isMember(securityToken, userId, groupOrUser);
        if (!result) {
            // user don't have right on current object, check parent right
            String parentId = WikittyAuthorisationHelper.getParent(extensionRights);
            if (parentId != null) {
                Wikitty parent = ws.restore(securityToken, parentId);
                result = isMember(securityToken, userId, parent, fieldName);
            }
        }
        return result;
    }

    /**
     * Par defaut un objet est lisible par tous, sauf s'il a l'extension
     * d'autorisation et que la liste des readers existe et n'est pas vide
     *
     * @param userId
     * @param w
     * @return true si l'utilisateur est dans la liste des reader (ou que cette
     * liste n'existe pas ce qui indique que tout le monde est reader)
     */
    protected boolean isReader(String securityToken, String userId, Wikitty w) {
        boolean result = true;
        if (WikittyAuthorisationHelper.isExtension(w)) {
            Set<String> groupOrUser = WikittyAuthorisationHelper.getReader(w);
            if (groupOrUser == null || groupOrUser.size() == 0) {
                // il n'y a pas de reader sur l'objet actuel, il faut regarder
                // sur le parent s'il y en a
                String parentId = WikittyAuthorisationHelper.getParent(w);
                if (parentId != null) {
                    Wikitty parent = ws.restore(securityToken, parentId);
                    result = isReader(securityToken, userId, parent);
                }
            } else {
                // il y a des readers sur l'objet actuel, il faut donc checker
                // comme pour les autres droits en parent aussi les parents
                result = isMember(
                        securityToken, userId, w, WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_READER);
            }
        }
        return result;
    }

    /**
     * Verifie si l'utilisateur est considere comme un AppAdmin
     *
     * @param userId
     * @return
     */
    protected boolean isAppAdmin(String securityToken, String userId) {
        Wikitty group = getAppAdminGroup(securityToken);
        Set<String> ids = WikittyGroupHelper.getMembers(group);
        boolean result = isMember(securityToken, userId, ids);
        return result;
    }

    protected Wikitty getAppAdminGroup(String securityToken) {
        Wikitty group;
        if (appAdminGroupId == null) {
            // 1er fois, on le recherche
            group = ws.findByCriteria(securityToken, Search.query().eq(
                    WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME, WIKITTY_APPADMIN_GROUP_NAME).criteria());
            // group peut-etre null s'il n'existe pas
        } else {
            // on a deja fait la recherche precedement, on essaie de reutilise
            // le meme id
            group = ws.restore(securityToken, appAdminGroupId);
            // group peut-etre null, si entre temps un admin a supprime le group
        }

        if (group == null) {
            // il n'existe pas on le cree.
            WikittyGroupAbstract appAdminGroup = new WikittyGroupImpl();
            appAdminGroup.setName(WIKITTY_APPADMIN_GROUP_NAME);
            ws.store(securityToken, appAdminGroup.getWikitty()); 
            group = appAdminGroup.getWikitty();

            // on garde l'id pour ne plus faire la recherche,
            // vu que le groupe doit etre unique cela ne pose pas de probleme
            appAdminGroupId = group.getId();
        }
        return group;
    }

    /**
     * Verifie recursivement si un utilisateur est dans un groupe qui peut etre
     * constitue d'autre groupe ou d'utilisateur
     *
     * @param userId l'utilisateur recherche
     * @param groupOrUser la liste des id d'utilisateurs ou d'autres groupes
     * @return vrai si userId est retrouve, false sinon
     */
    protected boolean isMember(
            String securityToken, String userId, Set<String> groupOrUser) {
        if (groupOrUser != null) {
            for (String id : groupOrUser) {
                if (userId.equals(id)) {
                    return true;
                } else {
                    Wikitty groupWikitty = ws.restore(securityToken, id);
                    if (WikittyGroupHelper.isExtension(groupWikitty)) {
                        Set<String> members = WikittyGroupHelper.getMembers(groupWikitty);
                        return isMember(securityToken, userId, members);
                    }
                }
            }
        }
        return false; // not found in groupOrUser
    }

    @Override
    public Wikitty findByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public List<String> deleteTree(String securityToken, String treeNodeId) {
        // TODO Auto-generated method stub
        return null;
    }

}
