package org.nuiton.topia.service.sql.batch.tables;

/*
 * #%L
 * ToPIA :: Service Replication
 * %%
 * Copyright (C) 2004 - 2015 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;

import java.util.Objects;

/**
 * This object represents a sql table.
 *
 * Created on 30/12/15.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0.1
 */
public class TopiaSqlTable {

    /**
     * Table schema name.
     */
    protected final String schemaName;

    /**
     * Table name.
     */
    protected final String tableName;

    /**
     * Fully table name (including the schema name).
     */
    protected final String fullyTableName;

    /**
     * From clause.
     */
    protected final String fromClause;

    /**
     * Where clause alias.
     */
    protected final String whereClauseAlias;

    /**
     * Join clauses.
     */
    protected final ImmutableSet<String> joinClauses;

    public TopiaSqlTable(String schemaName,
                         String tableName,
                         String fromClause,
                         String whereClauseAlias,
                         ImmutableSet<String> joinClauses) {
        this.schemaName = schemaName.toLowerCase();
        this.tableName = tableName.toLowerCase();
        this.fullyTableName = this.schemaName + "." + this.tableName;
        this.fromClause = fromClause;
        this.whereClauseAlias = whereClauseAlias;
        this.joinClauses = joinClauses;
    }

    public String getSchemaName() {
        return schemaName;
    }

    public String getTableName() {
        return tableName;
    }

    public String getFullyTableName() {
        return fullyTableName;
    }

    public String getFromClause() {
        return fromClause;
    }

    public String getWhereClauseAlias() {
        return whereClauseAlias;
    }

    public String getWhereClause(ImmutableSet<String> ids) {
        String result = whereClauseAlias;
        if (ids.size() == 1) {
            result += " = ?";
        } else {
            String in = "";
            for (String ignored : ids) {
                in += ", ?";
            }
            result += " IN (" + in.substring(2) + ")";
        }
        return result;
    }

    public ImmutableSet<String> getJoinClauses() {
        return joinClauses;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        TopiaSqlTable that = (TopiaSqlTable) o;
        return Objects.equals(fullyTableName, that.fullyTableName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(fullyTableName);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("fullyTableName", fullyTableName)
                .toString();
    }
}
