package org.nuiton.topia.persistence.internal;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: AbstractTopiaDao.java 3049 2014-04-04 14:43:22Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-2/topia-persistence/src/main/java/org/nuiton/topia/persistence/internal/AbstractTopiaDao.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.HqlAndParametersBuilder;
import org.nuiton.topia.persistence.QueryMissingOrderException;
import org.nuiton.topia.persistence.TopiaDao;
import org.nuiton.topia.persistence.TopiaDaoSupplier;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityContextable;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.topia.persistence.TopiaIdFactory;
import org.nuiton.topia.persistence.TopiaNoResultException;
import org.nuiton.topia.persistence.TopiaNonUniqueResultException;
import org.nuiton.topia.persistence.TopiaPersistenceContext;
import org.nuiton.topia.persistence.TopiaQueryBuilderAddCriteriaOrRunQueryStep;
import org.nuiton.topia.persistence.TopiaQueryBuilderRunQueryStep;
import org.nuiton.topia.persistence.support.TopiaHibernateSupport;
import org.nuiton.topia.persistence.support.TopiaJpaSupport;
import org.nuiton.topia.persistence.support.TopiaListenableSupport;
import org.nuiton.topia.persistence.support.TopiaSqlSupport;
import org.nuiton.topia.persistence.event.TopiaEntityListener;
import org.nuiton.topia.persistence.event.TopiaEntityVetoable;
import org.nuiton.topia.persistence.internal.support.TopiaFiresSupport;
import org.nuiton.topia.persistence.util.TopiaUtil;
import org.nuiton.topia.persistence.pager.TopiaPagerBean;
import org.nuiton.util.PagerBeanUtil;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

/**
 * This class has the common methods usable for each Dao managed by Topia. It is not JPA implementation dependent, it
 * only relies on {@link TopiaJpaSupport}.
 * <p/>
 * This class is directly extended by the GeneratedXyzTopiaDao which groups all the Xyz specific methods.
 * <p/>
 * Instances are created bt the model's specific {@link TopiaPersistenceContext}, which implements the
 * {@link TopiaDaoSupplier} contract.
 *
 * @param <E> the managed entity type
 * @author bpoussin <poussin@codelutin.com>
 * @author Arnaud Thimel : thimel@codelutin.com
 * @author Tony CHEMIT <chemit@codelutin.com>
 */
public abstract class AbstractTopiaDao<E extends TopiaEntity> implements TopiaDao<E> {

    private static Log log = LogFactory.getLog(AbstractTopiaDao.class);

    /**
     * Default batch size used to iterate on data.
     *
     * @since 2.6.14
     */
    protected int batchSize = 1000;

    protected TopiaJpaSupport topiaJpaSupport;

    protected TopiaHibernateSupport topiaHibernateSupport;

    protected TopiaSqlSupport topiaSqlSupport;

    protected TopiaListenableSupport topiaListenableSupport;

    protected TopiaIdFactory topiaIdFactory;

    protected TopiaFiresSupport topiaFiresSupport;

    protected TopiaDaoSupplier topiaDaoSupplier;

    public abstract TopiaEntityEnum getTopiaEntityEnum();

    public abstract Class<E> getEntityClass();

    /**
     * When AbstractTopiaContext create the TopiaDAOHibernate, it must call this
     * method just after.
     */
    public void init(
            TopiaJpaSupport topiaJpaSupport,
            TopiaHibernateSupport topiaHibernateSupport,
            TopiaSqlSupport topiaSqlSupport,
            TopiaListenableSupport topiaListenableSupport,
            TopiaIdFactory topiaIdFactory,
            TopiaFiresSupport topiaFiresSupport,
            TopiaDaoSupplier topiaDaoSupplier) {
        if (log.isDebugEnabled()) {
            log.debug("init dao for " + getEntityClass());
        }
        this.topiaJpaSupport = topiaJpaSupport;
        this.topiaHibernateSupport = topiaHibernateSupport;
        this.topiaSqlSupport = topiaSqlSupport;
        this.topiaListenableSupport = topiaListenableSupport;
        this.topiaIdFactory = topiaIdFactory;
        this.topiaFiresSupport = topiaFiresSupport;
        this.topiaDaoSupplier = topiaDaoSupplier;
    }

    public TopiaFiresSupport getTopiaFiresSupport() {
        return topiaFiresSupport;
    }

    @Override
    public int getBatchSize() {
        return batchSize;
    }

    @Override
    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    protected String newFromClause() {
        return newFromClause(null);
    }

    protected String newFromClause(String alias) {
        String hql = "from " + getTopiaEntityEnum().getImplementationFQN();
        if (StringUtils.isNotBlank(alias)) {
            hql += " " + alias;
        }
        return hql;
    }

    @Override
    public E newInstance() {
        if (log.isTraceEnabled()) {
            log.trace("entityClass = " + getEntityClass());
        }
        Class<E> implementation = (Class<E>)
                getTopiaEntityEnum().getImplementation();
        try {
            E newInstance = implementation.newInstance();
            return newInstance;
        } catch (InstantiationException e) {
            throw new TopiaException(
                    "Impossible de trouver ou d'instancier la classe "
                    + implementation);
        } catch (IllegalAccessException e) {
            throw new TopiaException(
                    "Impossible de trouver ou d'instancier la classe "
                    + implementation);
        }
    }

    @Override
    public TopiaPagerBean newPager(int pageSize) {
        TopiaPagerBean pager = newPager(newFromClause(), null, pageSize);
        return pager;
    }

    @Override
    public TopiaPagerBean newPager(String hql, Map<String, Object> params, int pageSize) {
        TopiaPagerBean pager = new TopiaPagerBean();

        if (hqlContainsOrderBy(hql)) {

            // must remove the order by clause, otherwise some sql queries won't work.
            hql = hql.substring(0, hql.toLowerCase().indexOf("order by"));
        }
        long count = count("SELECT COUNT(*) " + hql, params);
        pager.setRecords(count);
        pager.setPageSize(pageSize);
        PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);
        return pager;
    }

    @Override
    public void addTopiaEntityListener(TopiaEntityListener listener) {
        topiaListenableSupport.addTopiaEntityListener(getEntityClass(), listener);
    }

    @Override
    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaListenableSupport.addTopiaEntityVetoable(getEntityClass(), vetoable);
    }

    @Override
    public void removeTopiaEntityListener(TopiaEntityListener listener) {
        topiaListenableSupport.removeTopiaEntityListener(getEntityClass(), listener);
    }

    @Override
    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        topiaListenableSupport.removeTopiaEntityVetoable(getEntityClass(), vetoable);
    }

    @Override
    public E create(E entity) {
        // first set topiaId
        if (StringUtils.isBlank(entity.getTopiaId())) {

            // only set id if not already on
            String topiaId = topiaIdFactory.newTopiaId(getEntityClass(), entity);
            entity.setTopiaId(topiaId);
        }

        if (entity instanceof TopiaEntityContextable) {
            TopiaEntityContextable contextable = (TopiaEntityContextable) entity;
            contextable.setTopiaDaoSupplier(this.topiaDaoSupplier);
        }

        // save entity
        topiaJpaSupport.save(entity);
        topiaFiresSupport.warnOnCreateEntity(entity);
        return entity;
    }

    @Override
    public E create(String propertyName, Object propertyValue, Object... otherPropertyNamesAndValues) {
        Map<String, Object> properties =
                TopiaUtil.convertPropertiesArrayToMap(propertyName, propertyValue, otherPropertyNamesAndValues);
        E result = create(properties);
        return result;
    }

    @Override
    public E create(Map<String, Object> properties) {

        E result = newInstance();

        try {
            for (Map.Entry<String, Object> e : properties.entrySet()) {
                String propertyName = e.getKey();
                Object value = e.getValue();
                PropertyUtils.setProperty(result, propertyName, value);
            }
        } catch (IllegalAccessException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        } catch (InvocationTargetException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        } catch (NoSuchMethodException eee) {
            throw new IllegalArgumentException(
                    "Can't put properties on new Object", eee);
        }

        create(result);

        return result;
    }

    @Override
    public E create() {
        E result = newInstance();
        create(result);
        return result;
    }

    @Override
    public Iterable<E> createAll(Iterable<E> entities) {
        for (E entity : entities) {
            create(entity);
        }
        return entities;
    }

    @Override
    public E update(E entity) {
        topiaJpaSupport.saveOrUpdate(entity);
        topiaFiresSupport.warnOnUpdateEntity(entity);
        return entity;
    }

    @Override
    public Iterable<E> updateAll(Iterable<E> entities) {
        for (E entity : entities) {
            update(entity);
        }
        return entities;
    }

    @Override
    public void delete(E entity) {
        topiaJpaSupport.delete(entity);
        entity.notifyDeleted();
        topiaFiresSupport.warnOnDeleteEntity(entity);
    }

    @Override
    public void deleteAll(Iterable<E> entities) {
        for (E entity : entities) {
            delete(entity);
        }
    }

    protected HqlAndParametersBuilder<E> newHqlAndParametersBuilder() {
        HqlAndParametersBuilder<E> result = new HqlAndParametersBuilder<E>(getEntityClass());
        return result;
    }

    protected HqlAndParametersBuilder<E> getHqlForProperties(String propertyName,
                                                             Object propertyValue,
                                                             Object... otherPropertyNamesAndValues) {
        Map<String, Object> properties =
                TopiaUtil.convertPropertiesArrayToMap(propertyName, propertyValue, otherPropertyNamesAndValues);
        HqlAndParametersBuilder<E> result = getHqlForProperties(properties);
        return result;
    }

    protected HqlAndParametersBuilder<E> getHqlForNoConstraint() {
        Map<String, Object> properties = Collections.emptyMap();
        HqlAndParametersBuilder<E> result = getHqlForProperties(properties);
        return result;
    }

    protected HqlAndParametersBuilder<E> getHqlForProperties(Map<String, Object> properties) {
        HqlAndParametersBuilder<E> result = newHqlAndParametersBuilder();
        for (Map.Entry<String, Object> property : properties.entrySet()) {
            result.addEquals(property.getKey(), property.getValue());
        }
        return result;
    }

    protected InnerTopiaQueryBuilderRunQueryStep<E> forHql(String hql) {
        Map<String, Object> properties = Collections.emptyMap();
        InnerTopiaQueryBuilderRunQueryStep<E> result = forHql(hql, properties);
        return result;
    }

    protected InnerTopiaQueryBuilderRunQueryStep<E> forHql(String hql, Map<String, Object> hqlParameters) {
        InnerTopiaQueryBuilderRunQueryStep<E> result = new InnerTopiaQueryBuilderRunQueryStep<E>(this, true, hql, hqlParameters);
        return result;
    }

    protected InnerTopiaQueryBuilderRunQueryStep<E> forHql(String hql, String parameterName,
                                                      Object parameterValue,
                                                      Object... otherParameterNamesAndValues) {
        Map<String, Object> hqlParameters =
                TopiaUtil.convertPropertiesArrayToMap(parameterName, parameterValue, otherParameterNamesAndValues);
        InnerTopiaQueryBuilderRunQueryStep<E> result = forHql(hql, hqlParameters);
        return result;
    }

    @Override
    public InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forAll() {
        InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = newQueryBuilder();
        return result;
    }

    @Override
    public InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forProperties(Map<String, Object> properties) {
        HqlAndParametersBuilder<E> hqlAndParametersBuilder = getHqlForProperties(properties);
        InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = new InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E>(this, hqlAndParametersBuilder);
        return result;
    }

    @Override
    public InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forProperties(String propertyName,
                                                             Object propertyValue,
                                                             Object... otherPropertyNamesAndValues) {
        HqlAndParametersBuilder<E> hqlAndParametersBuilder = getHqlForProperties(propertyName, propertyValue, otherPropertyNamesAndValues);
        InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = new InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E>(this, hqlAndParametersBuilder);
        return result;
    }

    @Override
    public InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> newQueryBuilder() {
        HqlAndParametersBuilder<E> hqlAndParametersBuilder = newHqlAndParametersBuilder();
        InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = new InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E>(this, hqlAndParametersBuilder);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forContains(String propertyName, Object propertyValue) {
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = newQueryBuilder().addContains(propertyName, propertyValue);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forEquals(String propertyName, Object propertyValue) {
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = newQueryBuilder().addEquals(propertyName, propertyValue);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forIn(String propertyName, Collection<?> propertyValues) {
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = newQueryBuilder().addIn(propertyName, propertyValues);
        return result;
    }

    protected boolean exists(String hql, Map<String, Object> hqlParameters) {
        List<E> entities = topiaJpaSupport.find(hql, 0, 0, hqlParameters);
        boolean result = !entities.isEmpty();
        return result;
    }

    protected long count(String hql, Map<String, Object> hqlParameters) {
        Preconditions.checkArgument(hql.toLowerCase().trim().startsWith("select count("));
        Long result = findUnique(hql, hqlParameters);
        return result;
    }

    protected <K> K findUnique(String hql, Map<String, Object> hqlParameters) throws TopiaNoResultException, TopiaNonUniqueResultException {
        K result = findUniqueOrNull(hql, hqlParameters);
        if (result == null) {
            throw new TopiaNoResultException(hql, hqlParameters);
        }
        return result;
    }

    protected <K> Optional<K> tryFindUnique(String hql, Map<String, Object> hqlParameters) throws TopiaNonUniqueResultException {
        K uniqueOrNull = findUniqueOrNull(hql, hqlParameters);
        Optional<K> result = Optional.fromNullable(uniqueOrNull);
        return result;
    }

    protected <K> K findUniqueOrNull(String hql, Map<String, Object> hqlParameters) throws TopiaNonUniqueResultException {
        List<K> results = find(hql, hqlParameters, 0, 1);
        // If there is more than 1 result, throw an exception
        if (results.size() > 1) {
            throw new TopiaNonUniqueResultException(hql, hqlParameters);
        }
        // otherwise return the first one, or null
        K result = Iterables.getOnlyElement(results, null);
        return result;
    }

    protected <K> K findFirst(String hql, Map<String, Object> hqlParameters) throws QueryMissingOrderException {
        K result = findFirstOrNull(hql, hqlParameters);
        if (result == null) {
            throw new TopiaNoResultException(hql, hqlParameters);
        }
        return result;
    }

    protected <K> Optional<K> tryFindFirst(String hql, Map<String, Object> hqlParameters) throws QueryMissingOrderException {
        K firstOrNull = findFirstOrNull(hql, hqlParameters);
        Optional<K> result = Optional.fromNullable(firstOrNull);
        return result;
    }

    protected <K> K findFirstOrNull(String hql, Map<String, Object> hqlParameters) throws QueryMissingOrderException {
        if (!hqlContainsOrderBy(hql)) {
            throw new QueryMissingOrderException(hql, hqlParameters);
        }
        K result = findAnyOrNull(hql, hqlParameters);
        return result;
    }

    protected <K> K findAny(String hql, Map<String, Object> hqlParameters) throws TopiaNoResultException {
        K result = findAnyOrNull(hql, hqlParameters);
        if (result == null) {
            throw new TopiaNoResultException(hql, hqlParameters);
        }
        return result;
    }

    protected <K> Optional<K> tryFindAny(String hql, Map<String, Object> hqlParameters) {
        K anyOrNull = findAnyOrNull(hql, hqlParameters);
        Optional<K> result = Optional.fromNullable(anyOrNull);
        return result;
    }

    protected <K> K findAnyOrNull(String hql) {
        Preconditions.checkNotNull(hql);
        Map<String, Object> hqlParameters = Collections.emptyMap();
        K result = findAnyOrNull(hql, hqlParameters);
        return result;
    }

    protected <K> K findAnyOrNull(String hql, Map<String, Object> hqlParameters) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        List<K> results = find(hql, hqlParameters, 0, 0);
        K result = Iterables.getOnlyElement(results, null);
        return result;
    }

    protected <K> List<K> findAll(String hql) {
        Preconditions.checkNotNull(hql);
        Map<String, Object> hqlParameters = Collections.emptyMap();
        List<K> result = findAll(hql, hqlParameters);
        return result;
    }

    protected <K> List<K> findAll(String hql, Map<String, Object> hqlParameters) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        List<K> result = topiaJpaSupport.findAll(hql, hqlParameters);
        return result;
    }

    protected <K> List<K> find(String hql, int startIndex, int endIndex) {
        Preconditions.checkNotNull(hql);
        Map<String, Object> hqlParameters = Collections.emptyMap();
        List<K> result = find(hql, hqlParameters, startIndex, endIndex);
        return result;
    }

    protected <K> List<K> find(String hql, Map<String, Object> hqlParameters, int startIndex, int endIndex) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        List<K> result = topiaJpaSupport.find(hql, startIndex, endIndex, hqlParameters);
        return result;
    }

    public <K> List<K> find(String hql, Map<String, Object> hqlParameters, TopiaPagerBean pager) {
        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);
        Preconditions.checkNotNull(pager);

        if (StringUtils.isNotBlank(pager.getSortColumn())) {

            // can't have a order by clause in hql query
            Preconditions.checkState(
                    !hqlContainsOrderBy(hql),
                    "An 'order by' clause was already found in hql, can't use the order of the pager");

            hql += " order by " + pager.getSortColumn();
            if (!pager.isSortAscendant()) {
                hql += " desc";
            }
        }

        List<K> result = topiaJpaSupport.find(
                hql,
                (int) pager.getRecordStartIndex(),
                (int) pager.getRecordEndIndex() - 1,
                hqlParameters);

        return result;
    }

    protected <K> Iterable<K> findAllLazy(String hql) {
        Map<String, Object> hqlParameters = Collections.emptyMap();
        Iterable<K> result= findAllLazy(hql, hqlParameters);
        return result;
    }

    protected <K> Iterable<K> findAllLazy(String hql, Map<String, Object> hqlParameters) {

        Iterable<K> result= findAllLazy(hql, hqlParameters, batchSize);
        return result;
    }

    protected <K> Iterable<K> findAllLazy(String hql, int batchSize) {
        Map<String, Object> hqlParameters = Collections.emptyMap();
        Iterable<K> result = findAllLazy(hql, hqlParameters, batchSize);
        return result;
    }

    protected <K> Iterable<K> findAllLazy(String hql, Map<String, Object> hqlParameters, int batchSize) {

        Preconditions.checkNotNull(hql);
        Preconditions.checkNotNull(hqlParameters);

        final Iterator<K> iterator = new FindAllIterator<E, K>(this,
                                                               batchSize,
                                                               hql,
                                                               hqlParameters);

        Iterable<K> result = new Iterable<K>() {
            @Override
            public Iterator<K> iterator() {
                return iterator;
            }
        };

        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forTopiaIdEquals(String topiaId) {
        Preconditions.checkArgument(StringUtils.isNotBlank(topiaId), "given topiaId is blank");
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = forEquals(TopiaEntity.PROPERTY_TOPIA_ID, topiaId);
        return result;
    }

    @Override
    public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> forTopiaIdIn(Collection<String> topiaIds) {
        Preconditions.checkNotNull(topiaIds, "given topiaIds is null");
        TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> result = forIn(TopiaEntity.PROPERTY_TOPIA_ID, (Collection) topiaIds);
        return result;
    }

    @Override
    public E findByTopiaId(String topiaId) throws TopiaNoResultException {
        // AThimel 30/10/13 Not using findUnique to avoid querying several elements (cf. findUnique implementation)
        try {
            E result = forTopiaIdEquals(topiaId).findAny();
            return result;
        } catch (TopiaNoResultException tnre) {
            if (log.isWarnEnabled()) {
                String message = String.format("Unexpected behavior : entity '%s' not found with topiaId='%s'",
                        getEntityClass().getName(), topiaId);
                log.warn(message);
            }
            throw tnre;
        }
    }

    @Override
    public Optional<E> tryFindByTopiaId(String topiaId) {
        Optional<E> result = forTopiaIdEquals(topiaId).tryFindAny();
        return result;
    }

    @Override
    public List<String> findAllIds() {
        List<String> result = newQueryBuilder().findAllIds();
        return result;
    }

    @Override
    public List<E> findAll() {
        List<E> result = newQueryBuilder().findAll();
        return result;
    }

    @Override
    public Iterable<E> findAllLazy() {
        String hql = "from " + getTopiaEntityEnum().getImplementationFQN() + " order by id";
        Map<String, Object> hqlParameters = Collections.emptyMap();
        Iterable<E> result = findAllLazy(hql, hqlParameters);
        return result;
    }

    @Override
    public Iterator<E> iterator() {
        Iterator<E> result = findAllLazy().iterator();
        return result;
    }

    @Override
    public long count() {
        long result = newQueryBuilder().count();
        return result;
    }

    protected boolean hqlContainsOrderBy(String hql) {
        return hql.toLowerCase().contains("order by");
    }

    protected boolean hqlContainsCount(String hql) {
        return hql.toLowerCase().contains("count(");
    }

//    /**
//     * package locale method because this is hibernate specific method and
//     * we don't want expose it.
//     *
//     * @return the meta-data of the entity
//     * @throws org.nuiton.topia.persistence.TopiaException if any pb
//     */
//    protected ClassMetadata getClassMetadata() {
//        ClassMetadata meta = topiaHibernateSupport.getHibernateFactory().getClassMetadata(getEntityClass());
//        if (meta == null) {
//            String implementationFQN = getTopiaEntityEnum().getImplementationFQN();
//            meta = topiaHibernateSupport.getHibernateFactory().getClassMetadata(implementationFQN);
//        }
//        return meta;
//    }

    public static class FindAllIterator<E extends TopiaEntity, K> implements Iterator<K> {

        protected Iterator<K> data;

        protected final AbstractTopiaDao<E> dao;

        protected final String hql;

        protected final Map<String, Object> params;

        protected final TopiaPagerBean pager;

        public FindAllIterator(AbstractTopiaDao<E> dao,
                               int batchSize,
                               String hql,
                               Map<String, Object> params) {

            if (!dao.hqlContainsOrderBy(hql)) {
                throw new QueryMissingOrderException(hql, params);
            }

            this.dao = dao;
            this.hql = hql;
            this.params = params;
            pager = dao.newPager(hql, params, batchSize);

            // empty iterator (will be changed at first next call)
            data = Iterators.emptyIterator();
        }


        public boolean hasNext() {
            boolean result = data.hasNext() || // no more data
                             pager.getPageIndex() < pager.getPagesNumber();
            return result;
        }


        public K next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }

            if (!data.hasNext()) {

                // must load iterator

                // increments page index
                pager.setPageIndex(pager.getPageIndex() + 1);
                PagerBeanUtil.computeRecordIndexesAndPagesNumber(pager);

                // load new window of data
                List<K> values = dao.find(hql, params, pager);
                data = values.iterator();

            }

            K next = data.next();
            return next;
        }


        public void remove() {
            throw new UnsupportedOperationException(
                    "This iterator does not support remove operation.");
        }
    }

    public static class InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep<E extends TopiaEntity> implements TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> {

        protected AbstractTopiaDao<E> topiaDAO;

        protected HqlAndParametersBuilder<E> hqlAndParametersBuilder;

        protected InnerTopiaQueryBuilderAddCriteriaOrRunQueryStep(AbstractTopiaDao<E> topiaDAO, HqlAndParametersBuilder<E> hqlAndParametersBuilder) {
            this.topiaDAO = topiaDAO;
            this.hqlAndParametersBuilder = hqlAndParametersBuilder;
        }

        @Override
        public InnerTopiaQueryBuilderRunQueryStep<E> setOrderByArguments(Set<String> orderByArguments) {
            hqlAndParametersBuilder.setOrderByArguments(orderByArguments);
            return getNextStep();
        }

        @Override
        public InnerTopiaQueryBuilderRunQueryStep<E> setOrderByArguments(String... orderByArguments) {
            hqlAndParametersBuilder.setOrderByArguments(orderByArguments);
            return getNextStep();
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addEquals(String property, Object value) {
            hqlAndParametersBuilder.addEquals(property, value);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addNotEquals(String property, Object value) {
            hqlAndParametersBuilder.addNotEquals(property, value);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addIn(String property, Collection<?> values) {
            hqlAndParametersBuilder.addIn(property, values);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addNotIn(String property, Collection<?> values) {
            hqlAndParametersBuilder.addNotIn(property, values);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addContains(String property, Object value) {
            hqlAndParametersBuilder.addContains(property, value);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addNotContains(String property, Object value) {
            hqlAndParametersBuilder.addNotContains(property, value);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addNull(String property) {
            hqlAndParametersBuilder.addNull(property);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addNotNull(String property) {
            hqlAndParametersBuilder.addNotNull(property);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addTopiaIdEquals(String property, String topiaId) {
            hqlAndParametersBuilder.addTopiaIdEquals(property, topiaId);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addTopiaIdIn(String property, Collection<String> topiaIds) {
            hqlAndParametersBuilder.addTopiaIdIn(property, topiaIds);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addTopiaIdNotEquals(String property, String topiaId) {
            hqlAndParametersBuilder.addTopiaIdNotEquals(property, topiaId);
            return this;
        }

        @Override
        public TopiaQueryBuilderAddCriteriaOrRunQueryStep<E> addTopiaIdNotIn(String property, Collection<String> topiaIds) {
            hqlAndParametersBuilder.addTopiaIdNotIn(property, topiaIds);
            return this;
        }

        // shortcuts to next step

        @Override
        public boolean exists() {
            return getNextStep().exists();
        }

        @Override
        public E findAnyOrNull() {
            return getNextStep().findAnyOrNull();
        }

        @Override
        public E findUniqueOrNull() {
            return getNextStep().findUniqueOrNull();
        }

        @Override
        public E findAny() {
            return getNextStep().findAny();
        }

        @Override
        public E findUnique() {
            return getNextStep().findUnique();
        }

        @Override
        public E findFirst() {
            return getNextStep().findFirst();
        }

        @Override
        public E findFirstOrNull() {
            return getNextStep().findFirstOrNull();
        }

        @Override
        public Optional<E> tryFindAny() {
            return getNextStep().tryFindAny();
        }

        @Override
        public Optional<E> tryFindFirst() {
            return getNextStep().tryFindFirst();
        }

        @Override
        public Optional<E> tryFindUnique() {
            return getNextStep().tryFindUnique();
        }

        @Override
        public List<E> findAll() {
            return getNextStep().findAll();
        }

        @Override
        public List<E> find(int startIndex, int endIndex) {
            return getNextStep().find(startIndex, endIndex);
        }

        @Override
        public List<E> find(TopiaPagerBean pager) {
            return getNextStep().find(pager);
        }

        @Override
        public Iterable<E> findAllLazy() {
            return getNextStep().findAllLazy();
        }

        @Override
        public Iterable<E> findAllLazy(int batchSize) {
            return getNextStep().findAllLazy(batchSize);
        }

        @Override
        public long count() {
            return getNextStep().count();
        }

        @Override
        public List<String> findIds(int startIndex, int endIndex) {
            return getNextStep().findIds(startIndex, endIndex);
        }

        @Override
        public List<String> findIds(TopiaPagerBean pager) {
            return getNextStep().findIds(pager);
        }

        @Override
        public List<String> findAllIds() {
            return getNextStep().findAllIds();
        }

        protected InnerTopiaQueryBuilderRunQueryStep<E> getNextStep() {
            String hql = hqlAndParametersBuilder.getHql();
            Map<String, Object> hqlParameters = hqlAndParametersBuilder.getHqlParameters();
            InnerTopiaQueryBuilderRunQueryStep<E> nextStep = new InnerTopiaQueryBuilderRunQueryStep<E>(topiaDAO, false, hql, hqlParameters);
            return nextStep;
        }

    }

    public static class InnerTopiaQueryBuilderRunQueryStep<E extends TopiaEntity> implements TopiaQueryBuilderRunQueryStep<E> {

        protected final String hql;

        protected final Map<String, Object> hqlParameters;

        protected final AbstractTopiaDao<E> topiaDAO;

        protected final boolean fromHql;

        protected InnerTopiaQueryBuilderRunQueryStep(AbstractTopiaDao<E> topiaDAO,
                                                     boolean fromHql,
                                                     String hql,
                                                     Map<String, Object> hqlParameters) {
            this.fromHql = fromHql;
            this.hql = hql;
            this.hqlParameters = hqlParameters;
            this.topiaDAO = topiaDAO;
        }

        @Override
        public boolean exists() {
            return topiaDAO.exists(hql, hqlParameters);
        }

        @Override
        public long count() {
            String hqlWithSelectClause = "select count(topiaId) " + hql;
            return topiaDAO.count(hqlWithSelectClause, hqlParameters);
        }

        @Override
        public E findUnique() throws TopiaNoResultException, TopiaNonUniqueResultException {
            return topiaDAO.findUnique(hql, hqlParameters);
        }

        @Override
        public E findUniqueOrNull() throws TopiaNonUniqueResultException {
            return topiaDAO.findUniqueOrNull(hql, hqlParameters);
        }

        @Override
        public Optional<E> tryFindUnique() throws TopiaNonUniqueResultException {
            return topiaDAO.tryFindUnique(hql, hqlParameters);
        }

        @Override
        public E findFirst() throws QueryMissingOrderException, TopiaNoResultException {
            return topiaDAO.findFirst(hql, hqlParameters);
        }

        @Override
        public E findFirstOrNull() throws QueryMissingOrderException {
            return topiaDAO.findFirstOrNull(hql, hqlParameters);
        }

        @Override
        public Optional<E> tryFindFirst() throws QueryMissingOrderException {
            return topiaDAO.tryFindFirst(hql, hqlParameters);
        }

        @Override
        public E findAny() throws TopiaNoResultException {
            return topiaDAO.findAny(hql, hqlParameters);
        }

        @Override
        public E findAnyOrNull() {
            return topiaDAO.findAnyOrNull(hql, hqlParameters);
        }

        @Override
        public Optional<E> tryFindAny() {
            return topiaDAO.tryFindAny(hql, hqlParameters);
        }

        @Override
        public List<E> findAll() {
            return topiaDAO.findAll(hql, hqlParameters);
        }

        @Override
        public Iterable<E> findAllLazy() {
            return topiaDAO.findAllLazy(hql +(fromHql?"":" ORDER BY id") , hqlParameters);
        }

        @Override
        public Iterable<E> findAllLazy(int batchSize) {
            return topiaDAO.findAllLazy(hql +(fromHql?"":" ORDER BY id"), hqlParameters, batchSize);
        }

        @Override
        public List<E> find(int startIndex, int endIndex) {
            return topiaDAO.find(hql, hqlParameters, startIndex, endIndex);
        }

        @Override
        public List<E> find(TopiaPagerBean pager) {
            return topiaDAO.find(hql, hqlParameters, pager);
        }

        @Override
        public List<String> findAllIds() {
            String hqlWithSelectClause = "select topiaId " + hql;
            return topiaDAO.findAll(hqlWithSelectClause, hqlParameters);
        }

        @Override
        public List<String> findIds(int startIndex, int endIndex) {
            String hqlWithSelectClause = "select topiaId " + hql;
            return topiaDAO.find(hqlWithSelectClause, hqlParameters, startIndex, endIndex);
        }

        @Override
        public List<String> findIds(TopiaPagerBean pager) {
            String hqlWithSelectClause = "select topiaId " + hql;
            return topiaDAO.find(hqlWithSelectClause, hqlParameters, pager);
        }
    }

}
