/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: TopiaUtil.java 2892 2013-11-25 15:28:01Z athimel $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-7/topia-persistence/src/main/java/org/nuiton/topia/framework/TopiaUtil.java $
 * %%
 * Copyright (C) 2004 - 2013 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.framework;

import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.dialect.Dialect;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Table;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.service.ServiceRegistryBuilder;
import org.hibernate.service.jdbc.connections.spi.ConnectionProvider;
import org.hibernate.tool.hbm2ddl.DatabaseMetadata;
import org.hibernate.tool.hbm2ddl.TableMetadata;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaHibernateSupport;
import org.nuiton.topia.TopiaNotFoundException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.RecursiveProperties;
import org.nuiton.util.Resource;

import java.io.Closeable;
import java.io.IOException;
import java.net.URL;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

/**
 * TODO-fdesbois-20100507 : Need javadoc + translations for existing methods.
 *
 * @author bpoussin <poussin@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @version $Id: TopiaUtil.java 2892 2013-11-25 15:28:01Z athimel $
 */
public class TopiaUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static final Log log = LogFactory.getLog(TopiaUtil.class);

    /**
     * Permet de récupérer le fichier de propriété ayant le nom passé en
     * argument.
     *
     * @param pathOrUrl le nom du fichier de propriété à charger, s'il est null
     *                  ou vide retourne un objet Properties vide.
     * @return Un nouvel objet de propriete
     * @throws TopiaNotFoundException Si pathOrUrl n'est pas null ou vide et que
     *                                le fichier devant contenir les propriétés
     *                                n'est pas retrouvé.
     */
    public static Properties getProperties(String pathOrUrl)
            throws TopiaNotFoundException {
        return getProperties(null, pathOrUrl);
    }

    /**
     * Permet de récupérer le fichier de propriété ayant le nom passé en
     * argument.
     *
     * @param parent    l'objet properties utilisé comme parent de l'objet
     *                  retourné
     * @param pathOrUrl le nom du fichier de propriété à charger, s'il est null
     *                  ou vide retourne un objet Properties vide.
     * @return Un nouvel objet de propriete
     * @throws TopiaNotFoundException Si pathOrUrl n'est pas null ou vide et que
     *                                le fichier devant contenir les propriétés
     *                                n'est pas retrouvé.
     */
    public static Properties getProperties(Properties parent, String pathOrUrl)
            throws TopiaNotFoundException {
        Properties result = new RecursiveProperties(parent);

        // load properties for helper
        if (pathOrUrl != null && !pathOrUrl.equals("")) {
            try {
                URL propURL = Resource.getURL(pathOrUrl);
                log.info("Properties file used for " + pathOrUrl + " is: " + propURL);
                result.load(propURL.openStream());
            } catch (Exception eee) {
                throw new TopiaNotFoundException(
                        "Properties file can't be found: " + pathOrUrl, eee);
            }
        }
        return result;
    }

    /**
     * Compute a regex pattern given a format string.
     * <p/>
     * A {@link String#format(String, Object...)} will be apply to
     * <code>format</code>, with for parameters the list of <code>klass</code>
     * transformed in topia pattern via method {@link #getTopiaIdPattern(Class)}
     * ready to be capture (enclosed by ()).
     *
     * @param format  the format
     * @param classes the list of class to use
     * @return the pattern computed
     */
    public static Pattern getTopiaPattern(String format,
                                          Class<? extends TopiaEntity>... classes) {
        String[] entityPatterns = new String[classes.length];
        for (int i = 0; i < classes.length; i++) {
            Class<? extends TopiaEntity> aClass = classes[i];
            entityPatterns[i] = "(" + getTopiaIdPattern(aClass) + ")";
        }
        String s = String.format(format, (Object[]) entityPatterns);
        if (log.isDebugEnabled()) {
            log.debug(s);
        }
        return Pattern.compile(s);
    }

    /**
     * Compute the pattern to be used to capture a topia id for a given entity
     * class.
     *
     * @param klass the entity class
     * @return the pattern to capture a topia id for the given entity class.
     */
    public static String getTopiaIdPattern(Class<? extends TopiaEntity> klass) {
        StringBuilder buffer = new StringBuilder();
        StringTokenizer stk = new StringTokenizer(klass.getName(), ".");
        while (stk.hasMoreTokens()) {
            buffer.append("\\.").append(stk.nextToken());
        }
        buffer.append("#(?:\\d+?)#(?:\\d+)\\.(?:\\d+)");
        return buffer.substring(2);
    }

    /**
     * Test si une entite donnee correspondant a une configuration existe en
     * base.
     *
     * @param topiaHibernateSupport the Hibernate support required for this operation
     * @param entityName le nom de l'entite a tester
     * @return <tt>true</tt> si le schema de la table existe
     * @since 2.6.4
     */
    public static boolean isSchemaExist(TopiaHibernateSupport topiaHibernateSupport,
                                        String entityName) {

        ConnectionProviderSupplier connectionProviderSupplier =
                new ConnectionProviderSupplier(((SessionFactoryImplementor) topiaHibernateSupport.getHibernateFactory()).getServiceRegistry());

        boolean exist = false;

        try {

            Configuration configuration = topiaHibernateSupport.getHibernateConfiguration();
            PersistentClass classMapping =
                    configuration.getClassMapping(entityName);
            if (classMapping == null) {
                if (log.isInfoEnabled()) {
                    Iterator<?> itr = configuration.getClassMappings();
                    while (itr.hasNext()) {
                        log.info("available mapping " + itr.next());
                    }
                }
                throw new IllegalArgumentException(
                        "could not find entity with name " + entityName);
            }
            Table testTable = classMapping.getTable();

            if (testTable == null) {
                throw new IllegalArgumentException(
                        "could not find entity with name " + entityName);
            }

            ConnectionProvider connectionProvider =
                    connectionProviderSupplier.get();

            Dialect dialect = Dialect.getDialect(configuration.getProperties());

            Connection connection = null;
            try {
                connection = connectionProvider.getConnection();

                DatabaseMetadata meta = new DatabaseMetadata(connection, dialect, configuration);

                TableMetadata tmd = meta.getTableMetadata(
                        testTable.getName(), testTable.getSchema(),
                        testTable.getCatalog(), testTable.isQuoted());

                if (tmd != null) {
                    //table exist
                    exist = true;
                }
            } finally {
                connectionProvider.closeConnection(connection);
            }

        } catch (SQLException e) {
            log.error("Cant connect to database", e);
        } catch (TopiaNotFoundException e) {
            log.error("Cant connect to database", e);
        }

        return exist;
    }

    /**
     * Test si une entite donnee correspondant a une configuration existe en
     * base.
     *
     * @param configuration la configuration hibernate
     * @param entityName    le nom de l'entite a tester
     * @return <tt>true</tt> si le schema de la table existe
     */
    public static boolean isSchemaExist(Configuration configuration,
                                        String entityName) {

        ConnectionProviderSupplier connectionProviderSupplier =
                new ConnectionProviderSupplier(configuration);

        boolean exist = false;

        try {
            PersistentClass classMapping =
                    configuration.getClassMapping(entityName);
            if (classMapping == null) {
                if (log.isInfoEnabled()) {
                    Iterator<?> itr = configuration.getClassMappings();
                    while (itr.hasNext()) {
                        log.info("available mapping " + itr.next());
                    }
                }
                throw new IllegalArgumentException(
                        "could not find entity with name " + entityName);
            }
            Table testTable = classMapping.getTable();

            if (testTable == null) {
                throw new IllegalArgumentException(
                        "could not find entity with name " + entityName);
            }

            ConnectionProvider connectionProvider =
                    connectionProviderSupplier.get();

            Dialect dialect = Dialect.getDialect(configuration.getProperties());

            Connection connection = null;
            try {
                connection = connectionProvider.getConnection();

                DatabaseMetadata meta = new DatabaseMetadata(connection, dialect, configuration);

                TableMetadata tmd = meta.getTableMetadata(
                        testTable.getName(), testTable.getSchema(),
                        testTable.getCatalog(), testTable.isQuoted());

                if (tmd != null) {
                    //table exist
                    exist = true;
                }
            } finally {
                connectionProvider.closeConnection(connection);
            }

        } catch (SQLException e) {
            log.error("Cant connect to database", e);
        }

        // close connectionProviderSupplier
        try {
            connectionProviderSupplier.close();
        } catch (IOException e) {
            log.error("Cant close connection provider", e);
        }

        return exist;
    }

    /**
     * Test if the db associated to the given {@code configuration} contaisn any of
     * the dealed entities.
     *
     * @param configuration hibernate db configuration
     * @return {@code true} if there is no schema for any of the dealed entities,
     *         {@code false} otherwise.
     * @since 2.5.3
     */
    public static boolean isSchemaEmpty(Configuration configuration) {

        ConnectionProviderSupplier connectionProviderSupplier =
                new ConnectionProviderSupplier(configuration);

        try {

            ConnectionProvider connectionProvider =
                    connectionProviderSupplier.get();

            Dialect dialect = Dialect.getDialect(configuration.getProperties());

            Connection connection = null;
            try {
                connection = connectionProvider.getConnection();

                DatabaseMetadata meta = new DatabaseMetadata(connection, dialect, configuration);

                Iterator<?> itr = configuration.getClassMappings();
                while (itr.hasNext()) {
                    PersistentClass classMapping = (PersistentClass) itr.next();
                    Table testTable = classMapping.getTable();

                    if (testTable == null) {
                        throw new IllegalArgumentException(
                                "could not find entity with name " +
                                classMapping.getClassName());
                    }


                    TableMetadata tmd = meta.getTableMetadata(
                            testTable.getName(), testTable.getSchema(),
                            testTable.getCatalog(), testTable.isQuoted());

                    if (tmd != null) {
                        //table exist


                        if (log.isDebugEnabled()) {
                            log.debug("Existing table found " +
                                      testTable.getName() + " for entity " +
                                      classMapping.getClassName() +
                                      ", db is not empty.");
                        }

                        return false;
                    }
                }

            } finally {
                connectionProvider.closeConnection(connection);
            }

        } catch (SQLException e) {
            log.error("Cant connect to database", e);
        }

        // close connectionProviderSupplier
        try {
            connectionProviderSupplier.close();
        } catch (IOException e) {
            log.error("Cant close connection provider", e);
        }

        return true;
    }

    /**
     * Test if the db associated to the given {@code configuration} contains any of
     * the dealed entities.
     *
     * @param topiaHibernateSupport the Hibernate support required for this operation
     * @return {@code true} if there is no schema for any of the dealed entities,
     *         {@code false} otherwise.
     * @since 2.5.3
     */
    public static boolean isSchemaEmpty(TopiaHibernateSupport topiaHibernateSupport) {

        Configuration configuration = topiaHibernateSupport.getHibernateConfiguration();

        ConnectionProviderSupplier connectionProviderSupplier =
                new ConnectionProviderSupplier(((SessionFactoryImplementor) topiaHibernateSupport.getHibernateFactory()).getServiceRegistry());

        try {

            ConnectionProvider connectionProvider =
                    connectionProviderSupplier.get();

            Dialect dialect = Dialect.getDialect(configuration.getProperties());

            Connection connection = null;
            try {
                connection = connectionProvider.getConnection();

                DatabaseMetadata meta = new DatabaseMetadata(connection, dialect, configuration);

                Iterator<?> itr = configuration.getClassMappings();
                while (itr.hasNext()) {
                    PersistentClass classMapping = (PersistentClass) itr.next();
                    Table testTable = classMapping.getTable();

                    if (testTable == null) {
                        throw new IllegalArgumentException(
                                "could not find entity with name " +
                                classMapping.getClassName());
                    }


                    TableMetadata tmd = meta.getTableMetadata(
                            testTable.getName(), testTable.getSchema(),
                            testTable.getCatalog(), testTable.isQuoted());

                    if (tmd != null) {
                        //table exist

                        if (log.isDebugEnabled()) {
                            log.debug("Existing table found " +
                                      testTable.getName() + " for entity " +
                                      classMapping.getClassName() +
                                      ", db is not empty.");
                        }

                        return false;
                    }
                }

            } finally {
                connectionProvider.closeConnection(connection);
            }

        } catch (SQLException e) {
            log.error("Cant connect to database", e);
        }

        // close connectionProviderSupplier
        try {
            connectionProviderSupplier.close();
        } catch (IOException e) {
            log.error("Cant close connection provider", e);
        }

        return true;
    }

    /**
     * @param configuration
     * @return
     * @deprecated since 3.0, will be remove soon, do not use it, prefer use {@link ConnectionProviderSupplier}.
     */
    @Deprecated
    protected static ConnectionProvider getConnectionProvider(Configuration configuration) {
        ServiceRegistry serviceRegistry = new ServiceRegistryBuilder().applySettings(
                configuration.getProperties()).buildServiceRegistry();
        return serviceRegistry.getService(ConnectionProvider.class);
    }

    /**
     * Return hibernate schema name
     *
     * @param config of hibernate
     * @return schema name
     */
    public static String getSchemaName(Configuration config) {
        return config.getProperty(TopiaContextFactory.CONFIG_DEFAULT_SCHEMA);
    }

    public static Map<String, Object> convertPropertiesArrayToMap(Object... propertyNamesAndValues) throws IllegalArgumentException {
        int propertiesLength = propertyNamesAndValues.length;
        Preconditions.checkArgument(propertiesLength % 2 == 0,
                "Wrong number of argument "
                        + propertiesLength
                        + ", you must have even number.");
        Map<String, Object> properties = new LinkedHashMap<String, Object>();
        for (int i = 0; i < propertyNamesAndValues.length; ) {
            Object aPropertyName = propertyNamesAndValues[i++];
            Object value = propertyNamesAndValues[i++];
            Preconditions.checkArgument(
                    aPropertyName instanceof String,
                    "Argument at position [" + (i - 1) + "] " +
                            "should be a property name (says a String) but was " +
                            aPropertyName);
            properties.put((String) aPropertyName, value);
        }
        return properties;
    }

    public static Map<String, Object> convertPropertiesArrayToMap(
            String propertyName, Object propertyValue, Object... otherPropertyNamesAndValues) throws IllegalArgumentException {
        Map<String, Object> properties = new HashMap<String, Object>();
        properties.put(propertyName, propertyValue);
        properties.putAll(convertPropertiesArrayToMap(otherPropertyNamesAndValues));
        return properties;
    }

    public static class ConnectionProviderSupplier implements Supplier<ConnectionProvider>, Closeable {

        protected ServiceRegistry serviceRegistry;

        protected ConnectionProvider connectionProvider;

        protected final boolean inlineRegistry;

        public ConnectionProviderSupplier(ServiceRegistry serviceRegistry) {
            inlineRegistry = false;
            this.serviceRegistry = serviceRegistry;
        }

        public ConnectionProviderSupplier(Configuration configuration) {
            inlineRegistry = true;
            this.serviceRegistry = new ServiceRegistryBuilder().applySettings(
                    configuration.getProperties()).buildServiceRegistry();
        }

        @Override
        public ConnectionProvider get() {
            if (connectionProvider == null) {
                connectionProvider = serviceRegistry.getService(ConnectionProvider.class);
            }
            return connectionProvider;
        }

        @Override
        public void close() throws IOException {
            if (inlineRegistry) {
                ServiceRegistryBuilder.destroy(serviceRegistry);
            }
        }
    }
}
