/* *##% 
 * ToPIA :: Persistence
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/

/**
 * DeleteEntityTest.java
 *
 * Created: 4 juin 2009
 *
 * @author Florian Desbois <fdesbois@codelutin.com>
 * @version $Revision: 1457 $
 *
 * Mise a jour: $Date: 2009-05-15 04:12:45 +0200 (Fri, 15 May 2009) $
 * par : $Author: desbois $
 */

package org.nuiton.topiatest.deletetest;

import java.net.URL;
import java.util.Properties;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaContextFactory;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.TopiaTestDAOHelper;
import org.nuiton.topiatest.Personne;
import org.nuiton.topiatest.PersonneDAO;
import org.nuiton.util.Resource;
import static org.junit.Assert.*;

/**
 * Deleting tests with DAO and Entities generated with ToPIA (diagram delete-test in topiatest.zargo).
 * Different case of deleting, with inheritance or NMultiplicity relationship between two entities, or both.
 * Initiate from an issue with DAOAbstractGenerator delete method generation.
 * Tests with H2 Database. Configuration in src/test/resources/TopiaContextImpl.properties
 */
public class DeleteEntityTest {

    private static TopiaContext rootContext;

    private static final Log log = LogFactory.getLog(DeleteEntityTest.class);

    public DeleteEntityTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
        try {
            Properties conf = new Properties();
            URL url = Resource.getURL("TopiaContextImpl.properties");
            if (log.isDebugEnabled()) {
                log.debug(url);
            }
            conf.load(url.openStream());
            conf.setProperty("topia.persistence.classes", TopiaTestDAOHelper.getImplementationClassesAsString());

            rootContext = TopiaContextFactory.getContext(conf);
            if (log.isInfoEnabled()) {
                log.info("Context Ready !");
            }
        } catch (Exception ex) {
            log.error("Initialize error !!",ex);
        }
    }


    @AfterClass
    public static void tearDownClass() throws Exception {
        rootContext.closeContext();
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }


    /**
     * Test for deleting entities with inheritance. Delete from the DAO linked with the subclass
     * entity and from the DAO linked with the superclass entity. In the test model, the two entities have
     * NMultiplicity relationship without association class entity.
     */
    @Test
    public void testDeleteEntityWithInheritance() {
        log.debug("START TEST : testDeleteEntityWithInheritance");
        try {
            TopiaContext transaction = rootContext.beginTransaction();

            log.debug("DAO : PersonneDAO");
            PersonneDAO dao = TopiaTestDAOHelper.getPersonneDAO(transaction);

            log.debug("CREATE PERSONNE : Bob Marley");
            Personne personne = dao.create(Personne.NAME,"Bob Marley");
            transaction.commitTransaction();
            String idPersonne = personne.getTopiaId();
            assertNotNull(idPersonne);
            log.debug("ENTITY PERSONNE SAVED !");

            log.debug("DELETE PERSONNE");
            dao.delete(personne);
            transaction.commitTransaction();
            Personne res = dao.findByTopiaId(idPersonne);
            assertNull(res);
            log.debug("ENTITY PERSONNE DELETED !");

            log.debug("CREATE PERSONNE : Ziggy Marley");
            Personne personne2 = dao.create(Personne.NAME,"Ziggy Marley");
            transaction.commitTransaction();
            String idPersonne2 = personne2.getTopiaId();
            assertNotNull(idPersonne2);
            log.debug("ENTITY PERSONNE SAVED !");

            log.debug("DAO parent (abstract) : PartyDAO");
            Party2DAO dao2 = TopiaTestDAOHelper.getParty2DAO(transaction);

            log.debug("DELETE PERSONNE with PartyDAO");
            dao2.delete(personne2);
            transaction.commitTransaction();
            Party2 res2 = dao2.findByTopiaId(idPersonne2);
            assertNull(res2);
            log.debug("ENTITY PERSONNE DELETED !");

            transaction.closeContext();
        } catch (TopiaException ex) {
            log.error("TopiaException",ex);
        }
        log.debug("END TEST");
    }

    /**
     * Test for deleting entities with NMultiplicity relation without association class entity.
     * Test DAO generation for deleting references between two entities with NMultiplicity relation.
     * In the test model, the two entities have both inheritance.
     * @see org.nuiton.topia.generator.DAOAbstractTransformer
     */
    @Test
    public void testDeleteEntityWithManyToManyRelation() {
        log.debug("START TEST : testDeleteEntityWithManyToManyRelation");
        try {
            TopiaContext transaction = rootContext.beginTransaction();

            PersonneDAO dao = TopiaTestDAOHelper.getPersonneDAO(transaction);

            log.debug("CREATE PERSONNE : Bob Marley");
            Personne personne = dao.create(Personne.NAME,"Bob Marley");
            transaction.commitTransaction();
            String idPersonne = personne.getTopiaId();
            assertNotNull(idPersonne);
            log.debug("ENTITY PERSONNE SAVED !");

            Contact2DAO contactDAO = TopiaTestDAOHelper.getContact2DAO(transaction);

            log.debug("CREATE CONTACT : jaja@codelutin.com");
            Contact2 contact = contactDAO.create(Contact2.CONTACT_VALUE,"jaja@codelutin.com");
            transaction.commitTransaction();
            String idContact = contact.getTopiaId();
            assertNotNull(idContact);
            log.debug("ENTITY CONTACT SAVED !");

            log.debug("ADD CONTACT TO PERSONNE");
            personne.addContacts(contact);
            transaction.commitTransaction();
            assertEquals(personne.getContacts().size(),1);
            log.debug("CONTACT ADDED !");

            log.debug("DELETE PERSONNE");
            dao.delete(personne);
            transaction.commitTransaction();
            Personne res = dao.findByTopiaId(idPersonne);
            assertNull(res);
            log.debug("ENTITY PERSONNE DELETED !");

            assertEquals(contact.getParty2().size(),0);

            log.debug("DELETE CONTACT");
            contactDAO.delete(contact);
            transaction.commitTransaction();
            Contact2 res2 = contactDAO.findByTopiaId(idContact);
            assertNull(res2);
            log.debug("ENTITY PERSONNE DELETED !");


            transaction.closeContext();
        } catch (TopiaException ex) {
            log.error("TopiaException",ex);
        }
        log.debug("END TEST");
    }

}
