/*
 * *##% 
 * Maven License Plugin
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package org.nuiton.license.plugin;

import org.nuiton.processor.LicenseProcessor;

import java.io.File;
import java.io.StringWriter;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import org.apache.maven.project.MavenProject;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.codehaus.plexus.velocity.VelocityComponent;
import org.nuiton.license.plugin.header.generator.HeaderGenerator;
import org.nuiton.license.plugin.repository.License;
import org.nuiton.license.plugin.repository.LicenseRepository;
import org.nuiton.license.plugin.repository.LicenseRepositoryFactory;
import org.nuiton.plugin.AbstractPlugin;
import org.nuiton.plugin.PluginHelper;

/**
 * The goal to update (or add) the licence header on some files.
 *
 * @author chemit
 * @requiresProject true
 * @goal update-header
 * 
 * @since 1.0.1
 */
public class UpdateHeaderPlugin extends AbstractPlugin {

    /**
     * Dependance du projet.
     *
     * @parameter default-value="${project}"
     * @required
     * @since 1.0.0
     */
    protected MavenProject project;
    /**
     * Encoding a utiliser pour lire et ecrire les fichiers.
     *
     * @parameter expression="${license.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     * @since 1.0.0
     */
    protected String encoding;
    /**
     * l'annee de creation du module (sera place dans le header)
     *
     * @parameter expression="${license.inceptionYear}" default-value="${project.inceptionYear}"
     * @required
     * @since 1.0.0
     */
    protected String inceptionYear;
    /**
     * le nom de l'organisation (sera place dans le header)
     *
     * @parameter expression="${license.organizationName}"  default-value="${project.organization.name}"
     * @required
     * @since 1.0.0
     */
    protected String organizationName;
    /**
     * le nom du projet (sera place dans le header)
     *
     * @parameter expression="${license.projectName}" default-value="${project.name}"
     * @required
     * @since 1.0.0
     */
    protected String projectName;
    /**
     * Le type de license a appliquer.
     *
     * @parameter expression="${license.licenseName}"
     * @required
     * @since 1.0.0
     */
    protected String licenseName;
    /**
     * Le type de générateur a utiliser pour encapsuler le header.
     *
     * Par défaut, on veut utiliser le plugin sur des fichiers sources java.
     * 
     * @parameter expression="${license.generatorName}" default-value="license-java"
     * @required
     * @since 1.0.1
     */
    protected String generatorName;
    /**
     * La liste des patterns de fichiers à inclure (séparés par des virgules).
     *
     * Exemple : <code>**\/*.java,**\/*.properties</code>
     *
     * On recherchera alors les fichiers respectant l'un des patterns dans
     * tous les répertoires de sources et de tests.
     *
     * Par défaut, on veut utiliser le plugin sur des fichiers sources java.
     * 
     * @parameter expression="${license.includes}" default-value="**\/*.java"
     * @required
     * @since 1.0.1
     */
    protected String includes = "**/*.java";
    /**
     * La liste des patterns de fichiers à exclure (séparés par des virgules).
     *
     * Exemple : <code>**\/*.java,**\/*.properties</code>
     *
     * On recherchera alors les fichiers respectant l'un des patterns dans
     * tous les répertoires de sources et de tests.
     *
     * Par défaut, on n'exclue rien.
     *
     * @parameter expression="${license.excludes}"
     * @since 1.0.1
     */
    protected String excludes;
    /**
     * Repertoires des fichiers sources a traiter.
     *
     * @parameter expression="${license.compileSourceRoots}" default-value="${project.compileSourceRoots}"
     * @required
     * @since 1.0.0
     */
    protected List<String> compileSourceRoots;
    /**
     * Repertoires des fichiers sources de test a traiter.
     *
     * @parameter expression="${license.testCompileSourceRoots}" default-value="${project.testCompileSourceRoots}"
     * @required
     * @since 1.0.0
     */
    protected List<String> testCompileSourceRoots;
    /**
     * Un resolver externe
     *
     * @parameter expression="${license.licenseResolver}"
     * @since 1.0.0
     */
    protected String licenseResolver;
    /**
     * La template (velocity) a utiliser pour construire le header.
     *
     * Cette template doit être dans le class-path ou être un fichier existant
     *
     * @parameter expression="${license.template}" default-value="/license/defaultHeader.vm"
     * @since 1.0.1
     */
    protected String template;
    /**
     * Des paramètres supplémentaires à utiliser dans la template du header.
     * 
     * @parameter
     * @since 1.0.1
     */
    protected Map<String, String> templateParameters;
    /**
     * Un flag pour conserver un backup des fichiers modifies.
     *
     * @parameter expression="${license.keepBackup}"  default-value="false"
     * @since 1.0.0
     */
    protected boolean keepBackup;
    /**
     * Un flag pour activer le mode verbeux.
     *
     * @parameter expression="${license.verbose}"  default-value="${maven.verbose}"
     * @since 1.0.0
     */
    protected boolean verbose;
    /**
     * A flag to skip the goal.
     *
     * @parameter expression="${skipUpdateHeader}" default-value="false"
     * @since 1.0.3
     */
    private boolean skipUpdateHeader;
    /**
     * A flag to test plugin but modify no file.
     *
     * @parameter expression="${dryRun}" default-value="false"
     * @since 1.0.3
     */
    protected boolean dryRun;
    /**
     * Velocity Component.
     *
     * @component roleHint="maven-license-plugin"
     * @since 2.0.0
     */
    protected VelocityComponent velocity;
    /**
     * All available generators
     *
     * @component role="org.nuiton.license.plugin.header.generator.HeaderGenerator"
     */
    protected Map<String, HeaderGenerator> _generators;
    /** le header a ajouter dans chaque fichier source java */
    protected String licenseHeaderContent;
    /** le header complet (avec les balises de commentaires) */
    protected String boxedLicenseHeaderContent;
    /** la liste des chemin relatifs des sources java a traiter pour chaque repertoire contenant des sources */
    protected Map<File, String[]> filesToTreate;
    /** le timestamp utilise pour la generation*/
    protected long timestamp;

    public UpdateHeaderPlugin() {
        super("all files are up-to-date.");
    }

    @Override
    public boolean isVerbose() {
        return verbose;
    }

    @Override
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    @Override
    public MavenProject getProject() {
        return project;
    }

    @Override
    public void setProject(MavenProject project) {
        this.project = project;
    }

    @Override
    public boolean checkPackaging() {
        return true;
    }

    @Override
    public boolean init() throws Exception {

        if (skipUpdateHeader) {
            return true;
        }
        timestamp = System.nanoTime();

        if (_generators == null) {
            // should never happen...
            getLog().warn("no header generator found");
            return false;
        }
        if (verbose) {
            Iterator<Entry<String, HeaderGenerator>> itr = _generators.entrySet().iterator();
            while (itr.hasNext()) {
                Entry<String, HeaderGenerator> next = itr.next();
                getLog().info("config - available generator : " + next.getKey());
            }
        }

        if (!_generators.containsKey(generatorName.trim())) {
            getLog().warn("the generator named '" + generatorName + "' is unknown (use generator-list goal to see all of them)");
            return false;
        }
        HeaderGenerator generator = _generators.get(generatorName);

        // obtain all files to be treated

        filesToTreate = new HashMap<File, String[]>();
        String[] in = includes.split(",");
        String[] ex = excludes == null ? null : excludes.split(",");
        getFilesToTreateForRoots(in, ex, compileSourceRoots, filesToTreate, null);
        getFilesToTreateForRoots(in, ex, testCompileSourceRoots, filesToTreate, null);

        if (!filesToTreate.isEmpty()) {

            File templateFile = new File(template);

            // verifie que la template existe (dans le class-path ou en tant que fichier)
            checkResource(templateFile);

            // recuperation de la license a utiliser
            LicenseRepository factory = LicenseRepositoryFactory.newLicenseRepository(true, true, licenseResolver);
            License license = factory.getLicense(licenseName);

            if (verbose) {
                getLog().info("config - use license " + license.getName());
                getLog().info("config - use generator " + generator.getName());
                getLog().info("config - use template " + template);
            }
            // obtain content of license header
            licenseHeaderContent = computeHeader(license, generator);

            // build the comment boxed header content
            boxedLicenseHeaderContent = generator.getHeader(licenseHeaderContent);
            if (verbose) {
                getLog().info("config - header to use\n" + boxedLicenseHeaderContent);
            }

        } else {
            getLog().info("no file to treate.");
        }

        return true;
    }

    @Override
    protected void doAction() throws Exception {
        if (skipUpdateHeader) {
            getLog().info("skip flag is on, the goal is skip.");
            return;
        }

        if (filesToTreate.isEmpty()) {
            getLog().info("No file to treate.");
            return;
        }

        // create a licence processor with given header
        LicenseProcessor p = new LicenseProcessor(licenseHeaderContent);

        for (Entry<File, String[]> entry : filesToTreate.entrySet()) {
            File src = entry.getKey();
            for (String javaRelativePath : entry.getValue()) {
                File file = new File(src, javaRelativePath);

                if (verbose) {
                    getLog().info("process file " + file);
                }

                // file where to writeFile result
                File processFile = new File(file.getAbsolutePath() + "_" + timestamp);

                try {
                    p.process(file, processFile);
                    boolean foundLicenseHeader = p.getLicenceFilter().wasTouched();

                    if (!foundLicenseHeader) {
                        if (p.getLicenceFilter().isDetectHeader()) {
                            getLog().warn("skip file " + file + " (no license footer tag found : '##%*' !)");
                        } else {
                            // no license header found in file, add it
                            getLog().info("adding license header on file " + file);
                            String content = PluginHelper.readAsString(file, encoding);
                            content = boxedLicenseHeaderContent + content;
                            if (!dryRun) {
                                writeFile(processFile, content, encoding);
                            }
                        }
                    }

                    if (keepBackup && !dryRun) {
                        File backupFile = new File(file.getAbsolutePath() + "~");
                        if (verbose) {
                            getLog().debug("backup original file " + file);
                        }
                        file.renameTo(backupFile);
                    }
                    if (dryRun) {
                        processFile.delete();
                    } else {
                        processFile.renameTo(file);

                    }

                } catch (Exception e) {
                    getLog().warn("skip file " + file + " (could not process for reason : " + e.getMessage() + ")");
                    processFile.delete();
                } finally {
                    // toujours cleaner les états du filtre du processeur
                    p.getLicenceFilter().reset();
                }
            }
        }
    }

    /**
     * Construction du header a utiliser.
     *
     * Le header est généré à partir de la template velocity donnée et de la
     * license donnée.
     *
     * @param license
     * @param generator 
     * @return le header construit
     * @throws Exception pour toute erreur pendant la construction du header
     */
    protected String computeHeader(License license, HeaderGenerator generator) throws Exception {

        // recuperation de la license a mettre dans le header
        String licenseContent = license.getHeaderContent(encoding);

        // defined inceptionYear (if year is older than now suffix with a - thisYear)
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        String thisYear = cal.get(Calendar.YEAR) + "";
        if (!thisYear.equals(inceptionYear)) {
            inceptionYear = inceptionYear + " - " + thisYear;
        }

        // preparation de l'environnement velocity

        Context context = new VelocityContext();

        context.put("inceptionYear", inceptionYear);
        context.put("projectName", projectName);
        context.put("organizationName", organizationName);
        context.put("licenseContent", licenseContent);

        if (templateParameters != null) {
            for (Entry<String, String> e : templateParameters.entrySet()) {
                context.put(e.getKey(), e.getValue());
            }
        }

        if (verbose) {
            StringBuilder buffer = new StringBuilder();

            buffer.append("config - parameters for template : ");
            for (Object key : context.getKeys()) {
                buffer.append("\n   * " + key + " : " + context.get(key + ""));
            }
            getLog().info(buffer.toString());
        }

        // generation du contenu via velocity
        VelocityEngine engine = velocity.getEngine();

        StringWriter writer = new StringWriter();

        Template velocityTemplate = engine.getTemplate(template, encoding);

        velocityTemplate.merge(context, writer);

        writer.flush();

        writer.close();

        // recuperation du contenu du texte genere
        String r = writer.getBuffer().toString();

        // prefixage de toutes les lignes par un " * "
        String result = generator.prefixContent(r.trim() + "\n\n");

        return '\n' + result;
    }
}
