/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags.swing;

import jaxx.CompilerException;
import jaxx.UnsupportedAttributeException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptor;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.runtime.swing.Table;
import jaxx.tags.DefaultComponentHandler;
import jaxx.types.TypeManager;

import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.util.ArrayList;
import java.util.List;

public class TableHandler extends DefaultComponentHandler {
    public static final Insets DEFAULT_INSETS = new Insets(3, 3, 3, 3);

    public TableHandler(ClassDescriptor beanClass) {
        super(beanClass);
        ClassDescriptorLoader.checkSupportClass(getClass(), beanClass, Table.class);
    }

    @Override
    public void setAttribute(CompiledObject object, String propertyName, String stringValue, boolean inline, JAXXCompiler compiler) throws CompilerException {
        try {
            if (object instanceof CompiledTable) {
                CellHandler.setAttribute(((CompiledTable) object).getTableConstraints(), propertyName, stringValue);
            } else {
                super.setAttribute(object, propertyName, stringValue, inline, compiler);
            }
        }
        catch (UnsupportedAttributeException e) {
            super.setAttribute(object, propertyName, stringValue, inline, compiler);
        }
    }

    class CompiledTable extends CompiledObject {
        private List<Integer> rowSpans = new ArrayList<Integer>();

        private GridBagConstraints tableConstraints;
        private GridBagConstraints rowConstraints = null;
        private GridBagConstraints cellConstraints = null;
        private boolean emptyCell;

        public CompiledTable(String id, ClassDescriptor objectClass, JAXXCompiler compiler) throws CompilerException {
            super(id, objectClass, compiler);
            tableConstraints = new GridBagConstraints();
            tableConstraints.gridx = -1;
            tableConstraints.gridy = -1;
            tableConstraints.insets = DEFAULT_INSETS;
        }

        @Override
        public void addChild(CompiledObject child, String constraints, JAXXCompiler compiler) throws CompilerException {
            if (constraints != null) {
                compiler.reportError("Table does not accept constraints");
            }
            GridBagConstraints c = getCellConstraints();
            if (c == null) {
                compiler.reportError("Table tag may only contain row tags");
                return;
            }
            if (!emptyCell) {
                compiler.reportError("Table cells may only have one child component");
            }
            while (rowSpans.size() < c.gridx + c.gridwidth) {
                rowSpans.add(null);
            }
            for (int x = c.gridx; x < c.gridx + c.gridwidth; x++) {
                rowSpans.set(x, c.gridheight);
            }

            super.addChild(child, TypeManager.getJavaCode(c), compiler);

            emptyCell = false;
        }

        public GridBagConstraints getTableConstraints() {
            return tableConstraints;
        }

        public GridBagConstraints getRowConstraints() {
            return rowConstraints;
        }

        public GridBagConstraints getCellConstraints() {
            return cellConstraints;
        }

        public void newRow() {
            tableConstraints.gridy++;
            tableConstraints.gridx = -1;
            rowConstraints = (GridBagConstraints) tableConstraints.clone();

            for (int x = 0; x < rowSpans.size(); x++) {
                int rowSpan = rowSpans.get(x);
                if (rowSpan > 0) {
                    rowSpans.set(x, rowSpan - 1);
                }
            }
        }

        public void newCell() {
            emptyCell = true;
            rowConstraints.gridx++;
            while (rowConstraints.gridx < rowSpans.size() && rowSpans.get(rowConstraints.gridx) > 0) {
                rowConstraints.gridx++;
            }
            cellConstraints = (GridBagConstraints) rowConstraints.clone();
        }
    }

    @Override
    public CompiledObject createCompiledObject(String id, JAXXCompiler compiler) throws CompilerException {
        return new CompiledTable(id, getBeanClass(), compiler);
    }
}