package jaxx.compiler;

import java.util.Map;
import java.util.TreeMap;
import jaxx.CompilerException;

/**
 *
 * A class to decorate a compiled object at generation time.
 *
 * Contains also a cache of decorator in a dictonnary indexed by fqn of decorator.
 *
 * Use the {@link #getDecorator(java.lang.String)} to obtain the cached decorator.
 *
 * Note : The implementation of this class must be stateless.
 * 
 * @author tony
 * @since 1.2
 */
public abstract class CompiledObjectDecorator {

    protected static Map<String, CompiledObjectDecorator> cache;

    public static void registerDecorator(String key, Class<? extends CompiledObjectDecorator> klass) {
        synchronized (getCache()) {
            if (getCache().containsKey(key)) {
                throw new IllegalArgumentException("the decorator with key [" + key + "] is already registred! use another key name");
            }
            try {
                getCache().put(key, klass.newInstance());
            } catch (InstantiationException ex) {
                throw new IllegalArgumentException("could not create decorator " + klass + " for reason " + ex.getMessage(), ex);
            } catch (IllegalAccessException ex) {
                throw new IllegalArgumentException("could not create decorator " + klass + " for reason " + ex.getMessage(), ex);
            }
        }
    }

    public static CompiledObjectDecorator getDecorator(String name) {
        CompiledObjectDecorator decorator = getCache().get(name);
        if (decorator == null) {
            throw new IllegalArgumentException("could not find decorator with key " + name + " (known decorators : " + getCache().keySet());
        }
        return decorator;
    }

    public static CompiledObjectDecorator getDecorator(Class<?> type) {
        for (CompiledObjectDecorator decorator : getCache().values()) {
            if (type == decorator.getClass()) {
                return decorator;
            }
        }
        return null;
    }

    protected static synchronized Map<String, CompiledObjectDecorator> getCache() {
        if (cache == null) {
            cache = new TreeMap<String, CompiledObjectDecorator>();
        }
        return cache;
    }

    public static void reset() {
        if (cache != null) {
            cache.clear();
        }
    }

    public abstract void finalizeCompiler(JAXXCompiler compiler, CompiledObject root, CompiledObject object, JavaFile javaFile, String packageName, String className, String fullClassName);

    public abstract String getCreationCode(JAXXCompiler compiler, CompiledObject object) throws CompilerException;

    public abstract boolean createInitializer(JAXXCompiler compiler, CompiledObject root, CompiledObject object, StringBuffer code, boolean lastWasMethodCall);

    public abstract String createCompleteSetupMethod(JAXXCompiler compiler, CompiledObject object, JavaFile javaFile, StringBuffer initDataBindings);
}
